package ru.yandex.direct.core.entity.masterreport

import java.math.BigDecimal
import java.time.Duration
import java.time.format.DateTimeFormatter
import java.util.UUID
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.bannersystem.BannerSystemClient
import ru.yandex.direct.bannersystem.BsUriFactory
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportDimension
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportGroupByDate
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportMetric
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportRequest
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportResponse
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportRow
import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatusCalculator
import ru.yandex.direct.core.entity.campaign.container.CampaignsSelectionCriteria
import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.campaign.model.CampaignAttributionModel
import ru.yandex.direct.core.entity.campaign.model.CampaignCalcType
import ru.yandex.direct.core.entity.campaign.model.CampaignFilterStatus
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds
import ru.yandex.direct.core.entity.campaign.model.StrategyType
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.masterreport.model.MasterReportFilters
import ru.yandex.direct.core.entity.masterreport.model.MasterReportPeriod
import ru.yandex.direct.core.entity.strategy.StrategyExtractorHelper
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.tracing.Trace

@Service
class MasterReportService(
        private val bannerSystemClient: BannerSystemClient,
        private val clientService: ClientService,
        private val campaignService: CampaignService,
        private val aggregatedStatusesViewService: AggregatedStatusesViewService
) {
    companion object {
        private val logger = LoggerFactory.getLogger(MasterReportService::class.java)
        private val TIMEOUT = Duration.ofSeconds(60)
        const val SUCCESSFUL_RESPONSE_CODE = 0
        private val MASTER_REPORT_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd")
        private val MILLION = BigDecimal.valueOf(1_000_000)
    }

    fun getStatistics(
            clientId: ClientId,
            chiefUid: Long,
            period: MasterReportPeriod,
            groupByDate: MasterReportGroupByDate,
            attrModel: CampaignAttributionModel?,
            filters: MasterReportFilters?,
            dimensions: Set<MasterReportDimension>,
            metrics: Set<MasterReportMetric>,
            limitOffset: LimitOffset
    ): MasterReportResponse {
        val campaignIdByOrderId = getCampaignIdByOrderId(clientId, chiefUid, filters)
        if (campaignIdByOrderId.isEmpty()) {
            return MasterReportResponse()
                    .setData(emptyList())
                    .setTotals(MasterReportRow())
        }
        val currency = clientService.getWorkCurrency(clientId).code.name
        val dateFrom = MASTER_REPORT_DATE_FORMATTER.format(period.from)
        val dateTo = MASTER_REPORT_DATE_FORMATTER.format(period.to)
        val meaningfulGoalValues = mapMeaningfulGoalValues(clientId, campaignIdByOrderId, filters?.goalIds)
        val request = MasterReportRequest()
                .setWithVat(0)
                .setWithPerformanceCoverage(0)
                .setWithDiscount(1)
                .setDontGroupAndFilterZerosForTotals(1)
                .setCountableFieldsByTargettype(emptyList())
                .setDateFrom(dateFrom)
                .setDateTo(dateTo)
                .setGroupByDate(groupByDate)
                .setGroupBy(dimensions)
                .setCountableFields(metrics)
                .setCurrency(currency)
                .setMapping(convertMapping(dimensions, campaignIdByOrderId, meaningfulGoalValues))
                .setFiltersPre(convertFilters(attrModel, filters, metrics))
                .setOrderIds(campaignIdByOrderId.keys.map { it.toString() })
                .setLimits(convertLimits(limitOffset))
                .setOrderBy(convertOrderBy(groupByDate, dimensions))
        try {
            val campaignsCount = campaignIdByOrderId.size.toLong()
            val profile = Trace.current().profile("master_report:process_campaigns", "", campaignsCount)
            profile.use {
                val response = bannerSystemClient.doRequest(
                        BsUriFactory.MASTER_REPORT,
                        request,
                        UUID.randomUUID(),
                        TIMEOUT
                )
                if (!isSuccess(response)) {
                    throw RuntimeException("Unsuccessful master report response: " +
                            "${response.status}, error: ${response.errorText}")
                }
                return response
            }
        } catch (e: RuntimeException) {
            logger.error("Failed to get stat from master report", e)
            throw e
        }
    }

    protected fun getCampaignIdByOrderId(
            clientId: ClientId,
            chiefUid: Long,
            filters: MasterReportFilters?
    ): Map<Long, Long> {
        val criteria = buildSelectionCriteria(clientId, filters)
        var campaigns = campaignService.searchCampaignsByCriteria(clientId, criteria).asSequence()

        campaigns = filterByStrategyType(campaigns, filters?.strategyTypeIn)

        campaigns = filterByStatus(clientId, chiefUid, campaigns, filters)

        return replaceByMasterId(clientId, campaigns.associate { it.orderId to it.id })
    }

    private fun buildSelectionCriteria(clientId: ClientId, filters: MasterReportFilters?): CampaignsSelectionCriteria {
        val criteria = CampaignsSelectionCriteria().withClientId(clientId)
        if (filters == null) {
            return criteria
        }
        if (filters.campaignIds.isNotEmpty()) {
            criteria.withCampaignIds(filters.campaignIds)
        }
        if (filters.calcType == CampaignCalcType.CPM) {
            criteria.withCampaignTypes(CampaignTypeKinds.CPM)
        } else if (filters.calcType == CampaignCalcType.CPC) {
            criteria.withCampaignTypes(CampaignTypeKinds.ALL - CampaignTypeKinds.CPM)
        }
        return criteria
    }

    private fun filterByStrategyType(
            campaigns: Sequence<Campaign>,
            strategyTypeIn: Set<StrategyType>?
    ): Sequence<Campaign> {
        return if (strategyTypeIn?.isNullOrEmpty() != false) {
            campaigns
        } else {
            campaigns.filter {
                val s = it.strategy
                val strategyType = StrategyExtractorHelper.toStrategyTypeSafely(s.strategyName, s.strategyData, it.type)
                if (strategyType == null) {
                    false
                } else {
                    strategyTypeIn.contains(strategyType)
                }
            }
        }
    }

    private fun filterByStatus(
            clientId: ClientId,
            chiefUid: Long,
            campaigns: Sequence<Campaign>,
            filters: MasterReportFilters?
    ): Sequence<Campaign> {
        val favCampaigns = filterFavoriteCampaigns(chiefUid, campaigns, filters)
        return favCampaigns + filterByStatus(clientId, campaigns, filters)
    }

    private fun filterByStatus(
            clientId: ClientId,
            campaigns: Sequence<Campaign>,
            filters: MasterReportFilters?
    ): Sequence<Campaign> {
        val filterStatuses = buildFilterStatuses(filters)
        return when {
            filterStatuses == null -> campaigns
            filterStatuses.isEmpty() -> emptySequence()
            else -> {
                val campaignIds = campaigns.map { it.id }.toList()
                val statusesByIds = aggregatedStatusesViewService.getCampaignStatusesByIds(clientId, campaignIds)

                return campaigns.filter {
                    val filterStatus = CampaignStatusCalculator.convertToFilterStatus(statusesByIds[it.id])
                    filterStatuses.contains(filterStatus)
                }
            }
        }
    }

    private fun buildFilterStatuses(filters: MasterReportFilters?): Set<CampaignFilterStatus>? {
        return if (filters == null || filters.statusIn.isEmpty()) {
            null
        } else {
            val filterStatuses = mutableSetOf<CampaignFilterStatus>()
            if (MasterReportCampaignStatus.ARCHIVED in filters.statusIn) {
                filterStatuses.add(CampaignFilterStatus.ARCHIVED)
            }
            if (MasterReportCampaignStatus.STOPPED in filters.statusIn) {
                filterStatuses.add(CampaignFilterStatus.STOPPED)
            }
            if (MasterReportCampaignStatus.ACTIVE in filters.statusIn) {
                filterStatuses.addAll(
                        setOf(
                                CampaignFilterStatus.ACTIVE,
                                CampaignFilterStatus.TEMPORARILY_PAUSED,
                                CampaignFilterStatus.RUN_WARN,
                        )
                )
            }
            filterStatuses
        }
    }

    private fun filterFavoriteCampaigns(
            chiefUid: Long,
            campaigns: Sequence<Campaign>,
            filters: MasterReportFilters?
    ): Sequence<Campaign> {
        if (filters == null || filters.statusIn.isNullOrEmpty()) {
            return emptySequence()
        }
        if (MasterReportCampaignStatus.FAVORITE in filters.statusIn) {
            val favCampaignIds = campaignService.searchFavoriteCampaignIds(chiefUid)
            return campaigns.filter { favCampaignIds.contains(it.id) }
        }
        return emptySequence()
    }

    private fun mapMeaningfulGoalValues(
            clientId: ClientId,
            campaignIdByOrderId: Map<Long, Long>,
            goalIds: Set<Long>?
    ): Map<String, Long> {
        if (goalIds.isNullOrEmpty()) {
            return emptyMap()
        }
        val result = mutableMapOf<String, Long>()
        val meaningfulGoalsByCid = campaignService.getMeaningfulGoalsByCampaignId(clientId, campaignIdByOrderId.values)
        for ((orderId, cid) in campaignIdByOrderId) {
            meaningfulGoalsByCid[cid]?.let {
                for (goal in it) {
                    if (goal.isMetrikaSourceOfValue != true && goalIds.contains(goal.goalId)) {
                        result["$orderId,${goal.goalId}"] = goal.conversionValue.multiply(MILLION).toLong()
                    }
                }
            }
        }
        return result
    }

    private fun replaceByMasterId(clientId: ClientId, campaignIdByOrderId: Map<Long, Long>): Map<Long, Long> {
        val subCampaignIdToMasterId = campaignService
            .getMasterIdBySubCampaignId(clientId, campaignIdByOrderId.values)
        return campaignIdByOrderId.mapValues { (_, cid) ->
            subCampaignIdToMasterId[cid] ?: cid
        }
    }

    private fun isSuccess(response: MasterReportResponse): Boolean {
        return SUCCESSFUL_RESPONSE_CODE == response.status && response.totals != null && response.data != null
    }
}
