package ru.yandex.direct.core.entity.mdsfile.repository;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.mdsfile.model.MdsFileCustomName;
import ru.yandex.direct.core.entity.mdsfile.model.MdsFileMetadata;
import ru.yandex.direct.core.entity.mdsfile.model.MdsStorageHost;
import ru.yandex.direct.core.entity.mdsfile.model.MdsStorageType;
import ru.yandex.direct.dbschema.ppc.Indexes;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.time.LocalDateTime.now;
import static ru.yandex.direct.dbschema.ppc.tables.MdsCustomNames.MDS_CUSTOM_NAMES;
import static ru.yandex.direct.dbschema.ppc.tables.MdsMetadata.MDS_METADATA;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;


@Repository
public class MdsFileRepository {
    private DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<MdsFileMetadata> mdsFileMetadataMapper;
    private final JooqMapperWithSupplier<MdsFileCustomName> mdsFileNameMapper;


    private final ShardHelper shardHelper;

    @Autowired
    public MdsFileRepository(DslContextProvider dslContextProvider,
                             ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.mdsFileMetadataMapper = createMdsFileMetadataMapper();
        this.mdsFileNameMapper = createMdsFileNameMapper();
        this.shardHelper = shardHelper;
    }

    /**
     * Добавляет метаданные в таблицу в базе
     * Обновляет дату, в случае, если
     *
     * @param mdsFiles список записей метаданных
     * @return список идентификаторов записей метаданных
     */
    public List<Long> addMetadata(int shard, List<MdsFileMetadata> mdsFiles) {
        return addMetadata(dslContextProvider.ppc(shard), mdsFiles);
    }

    private List<Long> addMetadata(DSLContext dslContext, List<MdsFileMetadata> mdsFiles) {
        generateMdsFileIds(mdsFiles);

        LocalDateTime createTime = now();
        mdsFiles.forEach(mdsFileMetadata -> mdsFileMetadata.setCreateTime(createTime));

        new InsertHelper<>(dslContext, MDS_METADATA)
                .addAll(mdsFileMetadataMapper, mdsFiles)
                .executeIfRecordsAdded();

        return mapList(mdsFiles, MdsFileMetadata::getId);
    }

    public Map<String, MdsFileMetadata> getMetadata(int shard, ClientId clientId, List<String> hashes) {
        return dslContextProvider.ppc(shard)
                .select(mdsFileMetadataMapper.getFieldsToRead())
                .from(MDS_METADATA)
                .where(MDS_METADATA.FILE_IMPRINT.in(hashes))
                .and(MDS_METADATA.CLIENT_ID.eq(clientId.asLong()))
                .fetchMap(MDS_METADATA.FILE_IMPRINT, mdsFileMetadataMapper::fromDb);
    }

    public void updateCreateTimeMetadata(int shard, ClientId clientId, Long fileId) {
        dslContextProvider.ppc(shard)
                .update(MDS_METADATA)
                .set(MDS_METADATA.CREATE_TIME, LocalDateTime.now())
                .where(MDS_METADATA.ID.eq(fileId).and(MDS_METADATA.CLIENT_ID.eq(clientId.asLong())))
                .execute();
    }

    /**
     * Возвращает список {@link MdsFileMetadata} по переданному типу {@param type} с датой создания <
     * {@param borderDateTime} либо (с датой создания = {@param borderDateTime} и ID < {@param borderId}), с лимитом
     * {@param limit}, с сортировкой по CREATE_TIME, ID
     *
     * @param shard          шард
     * @param type           тип
     * @param borderDateTime пороговая дата создания
     * @param borderId       пороговое значение ID (для CREATE_TIME = {@param borderDateTime})
     * @param limit          количество строк
     */
    public List<MdsFileMetadata> getMetadataLessThanCreateTimeWithSort(int shard, MdsStorageType type,
                                                                       LocalDateTime borderDateTime, long borderId,
                                                                       int limit) {
        return dslContextProvider.ppc(shard)
                .select(mdsFileMetadataMapper.getFieldsToRead())
                .from(MDS_METADATA.forceIndex(Indexes.MDS_METADATA_I_TYPE_CREATE_TIME.getName()))
                .where(MDS_METADATA.TYPE.eq(MdsStorageType.toSource(type)))
                .and(MDS_METADATA.CREATE_TIME.lessThan(borderDateTime)
                        .or(MDS_METADATA.CREATE_TIME.eq(borderDateTime)
                                .and(MDS_METADATA.ID.lt(borderId))))
                .orderBy(MDS_METADATA.CREATE_TIME.desc(), MDS_METADATA.ID.desc())
                .limit(limit)
                .fetch(mdsFileMetadataMapper::fromDb);
    }

    /**
     * Удаление записей с заданным id из MDS_METADATA
     *
     * @param shard шард
     * @param id    id файла
     * @return количество удаленных записей
     */
    public int deleteMetadata(int shard, long id) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(MDS_METADATA)
                .where(MDS_METADATA.ID.eq(id))
                .execute();
    }

    /**
     * Удаление записей с заданным mdsId из MDS_CUSTOM_NAMES
     *
     * @param shard шард
     * @param mdsId id файла
     * @return количество удаленных записей
     */
    public int deleteCustomNames(int shard, long mdsId) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(MDS_CUSTOM_NAMES)
                .where(MDS_CUSTOM_NAMES.MDS_ID.eq(mdsId))
                .execute();
    }

    /**
     * Добавляет пользовательское имя в таблицу в базе
     *
     * @param mdsCustomNames список записей метаданных
     */
    public List<Long> addCustomName(int shard, List<MdsFileCustomName> mdsCustomNames) {
        return addCustomName(dslContextProvider.ppc(shard), mdsCustomNames);
    }

    public List<Long> addCustomName(DSLContext dslContext, List<MdsFileCustomName> mdsCustomNames) {
        new InsertHelper<>(dslContext, MDS_CUSTOM_NAMES)
                .addAll(mdsFileNameMapper, mdsCustomNames)
                .executeIfRecordsAdded();
        return mapList(mdsCustomNames, MdsFileCustomName::getMdsId);
    }

    public Map<Long, String> getCustomNames(int shard, List<Long> mdsIds) {
        return dslContextProvider.ppc(shard)
                .select(mdsFileNameMapper.getFieldsToRead())
                .from(MDS_CUSTOM_NAMES)
                .where(MDS_CUSTOM_NAMES.MDS_ID.in(mdsIds))
                .fetchMap(MDS_CUSTOM_NAMES.MDS_ID, MDS_CUSTOM_NAMES.FILENAME);
    }

    /**
     * Для каждой записи метаданных генерирует id (mds_id)
     * Выставляет записям сгенерированный id для последующей записи в базу.
     *
     * @param mdsFiles список записей метаданных
     */
    private void generateMdsFileIds(List<MdsFileMetadata> mdsFiles) {
        List<Long> ids = shardHelper.generateMdsFileIds(mdsFiles.size());
        StreamEx.of(mdsFiles).zipWith(ids.stream())
                .forKeyValue(MdsFileMetadata::setId);
    }


    public static JooqMapperWithSupplier<MdsFileMetadata> createMdsFileMetadataMapper() {
        return JooqMapperWithSupplierBuilder.builder(MdsFileMetadata::new)
                .map(property(MdsFileMetadata.ID, MDS_METADATA.ID))
                .map(property(MdsFileMetadata.CLIENT_ID, MDS_METADATA.CLIENT_ID))
                .map(convertibleProperty(MdsFileMetadata.STORAGE_HOST, MDS_METADATA.STORAGE_HOST,
                        MdsStorageHost::fromSource, MdsStorageHost::toSource))
                .map(convertibleProperty(MdsFileMetadata.TYPE, MDS_METADATA.TYPE,
                        MdsStorageType::fromSource, MdsStorageType::toSource))
                .map(property(MdsFileMetadata.MDS_KEY, MDS_METADATA.MDS_KEY))
                .map(property(MdsFileMetadata.FILENAME, MDS_METADATA.FILENAME))
                .map(property(MdsFileMetadata.FILE_IMPRINT, MDS_METADATA.FILE_IMPRINT))
                .map(property(MdsFileMetadata.SIZE, MDS_METADATA.SIZE))
                .map(property(MdsFileMetadata.CREATE_TIME, MDS_METADATA.CREATE_TIME))
                .build();
    }

    public static JooqMapperWithSupplier<MdsFileCustomName> createMdsFileNameMapper() {
        return JooqMapperWithSupplierBuilder.builder(MdsFileCustomName::new)
                .map(property(MdsFileCustomName.MDS_ID, MDS_CUSTOM_NAMES.MDS_ID))
                .map(property(MdsFileCustomName.FILENAME, MDS_CUSTOM_NAMES.FILENAME))
                .build();
    }


}
