package ru.yandex.direct.core.entity.mdsfile.service;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.mds.MdsHolder;
import ru.yandex.direct.core.entity.mdsfile.model.MdsFileMetadata;
import ru.yandex.direct.core.entity.mdsfile.model.MdsFileSaveRequest;
import ru.yandex.direct.core.entity.mdsfile.model.MdsStorageHost;
import ru.yandex.direct.core.entity.mdsfile.model.MdsStorageType;
import ru.yandex.direct.core.entity.mdsfile.repository.MdsFileRepository;
import ru.yandex.direct.dbschema.ppc.enums.MdsMetadataStorageHost;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.inside.mds.MdsFileKey;
import ru.yandex.inside.mds.MdsPostResponse;
import ru.yandex.inside.mds.MdsStorageHttpException;
import ru.yandex.inside.mds.MdsStorageKeyAlreadyExistsHttpException;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class MdsFileService {

    private static final Logger logger = LoggerFactory.getLogger(MdsFileService.class);

    private static final int GET_METADATA_CHUNK_SIZE = 1_000;

    private final ShardHelper shardHelper;
    private final MdsHolder directFilesMds;
    private final MdsFileRepository mdsFileRepository;

    private final MdsStorageHost storageHost;

    @Autowired
    public MdsFileService(ShardHelper shardHelper,
                          MdsFileRepository mdsFileRepository,
                          MdsHolder directFilesMds) {
        this(shardHelper, mdsFileRepository, directFilesMds, getStorageHost(directFilesMds));
    }

    public MdsFileService(ShardHelper shardHelper,
                          MdsFileRepository mdsFileRepository,
                          MdsHolder directFilesMds,
                          MdsStorageHost storageHost) {
        this.shardHelper = shardHelper;
        this.mdsFileRepository = mdsFileRepository;
        this.directFilesMds = directFilesMds;
        this.storageHost = storageHost;
    }

    public List<MdsFileSaveRequest> saveMdsFiles(List<MdsFileSaveRequest> requests, Long clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(ClientId.fromLong(clientId));

        for (MdsFileSaveRequest request : requests) {
            String mdsPath = request.getMdsPath();
            String mdsKey;
            try (TraceProfile ignore = Trace.current().profile("mds:upload")) {
                MdsPostResponse mdsResponse = directFilesMds.upload(mdsPath, request.getInputStream());
                mdsKey = mdsResponse.getKey().toString();
            } catch (MdsStorageKeyAlreadyExistsHttpException e) {
                mdsKey = e.getMdsKey().toString();
            }

            String filename = request.getMd5Hash();
            MdsFileMetadata mdsMetadata = new MdsFileMetadata()
                    .withClientId(clientId)
                    .withFileImprint(filename)
                    .withFilename(filename)
                    .withSize((long) request.getData().length)
                    .withStorageHost(this.storageHost)
                    .withMdsKey(mdsKey)
                    .withType(request.getType());
            request.setMdsMetadata(mdsMetadata);
        }

        mdsFileRepository.addMetadata(shard, mapList(requests, MdsFileSaveRequest::getMdsMetadata));

        for (MdsFileSaveRequest request : requests) {
            request.updateCustomNameId();
        }

        mdsFileRepository.addCustomName(shard, filterList(
                mapList(requests, MdsFileSaveRequest::getMdsCustomName),
                Objects::nonNull
        ));

        return requests;
    }

    /**
     * Удаление файлов по метаданным с TYPE = {@param mdsStorageType} и CREATE_TIME < {@param toDateTime},
     * с удалением записей о них в mds_custom_names и mds_metadata.
     *
     * @param shard          шард
     * @param toDateTime     пороговая дата создания
     * @param mdsStorageType тип
     */
    public void deleteOldMdsFiles(int shard, LocalDateTime toDateTime, MdsStorageType mdsStorageType) {
        int countMetadataRemoved = 0;
        int countMetadataNotRemoved = 0;
        long borderId = 0L;
        LocalDateTime borderDateTime = toDateTime;
        while (true) {
            List<MdsFileMetadata> mdsFilesMetadata = mdsFileRepository.getMetadataLessThanCreateTimeWithSort(
                    shard, mdsStorageType, borderDateTime, borderId, GET_METADATA_CHUNK_SIZE);
            if (mdsFilesMetadata.isEmpty()) {
                break;
            }
            MdsFileMetadata lastElement = Iterables.getLast(mdsFilesMetadata);
            borderId = lastElement.getId();
            borderDateTime = lastElement.getCreateTime();
            for (MdsFileMetadata fileMetadata : mdsFilesMetadata) {
                if (deleteMdsFile(shard, fileMetadata)) {
                    countMetadataRemoved++;
                } else {
                    countMetadataNotRemoved++;
                }
            }
        }
        if (countMetadataNotRemoved > 0) {
            logger.warn("deleted {} metadata files and found {} files that were not deleted", countMetadataRemoved,
                    countMetadataNotRemoved);
        } else {
            logger.info("deleted {} metadata files", countMetadataRemoved);
        }
    }

    /**
     * Удаление файла по {@param mdsFileMetadata}, с удалением записей о нем в mds_custom_names и mds_metadata.
     *
     * @param shard           шард
     * @param mdsFileMetadata метаданные файла в MDS
     * @return true - если файл и его данные в mds_custom_names и mds_metadata были удалены, иначе - false
     */
    public boolean deleteMdsFile(int shard, MdsFileMetadata mdsFileMetadata) {
        if (getStorageHost(directFilesMds) != mdsFileMetadata.getStorageHost()) {
            logger.warn("This file is from a different storage host, {}", mdsFileMetadata);
            return false;
        }
        // Сначала пытаемся удалить файл -> если exception нет (либо есть, но это not_found) то удаляем mds_metadata и
        // mds_custom_names
        try (TraceProfile ignore = Trace.current().profile("mds:delete")) {
            directFilesMds.delete(MdsFileKey.parse(mdsFileMetadata.getMdsKey()));
            logger.info("File was deleted by metadata {}", mdsFileMetadata);
        } catch (MdsStorageHttpException ex) {
            if (ex.getStatusLine().get().getStatusCode() == HttpStatus.NOT_FOUND.value()) {
                logger.info("File not found by metadata {}", mdsFileMetadata);
            } else {
                logger.warn("Failed to delete file by metadata {}, exception: {}", mdsFileMetadata, ex);
                return false;
            }
        }

        if (hasCustomName(mdsFileMetadata)) {
            logger.info("delete from mds_custom_names by mds_id {}", mdsFileMetadata.getId());
            mdsFileRepository.deleteCustomNames(shard, mdsFileMetadata.getId());
        }
        mdsFileRepository.deleteCustomNames(shard, mdsFileMetadata.getId());
        logger.info("delete from mds_metadata by id {}", mdsFileMetadata.getId());
        mdsFileRepository.deleteMetadata(shard, mdsFileMetadata.getId());
        return true;
    }

    /**
     * Указан ли при сохранении filename, не совпадающий с хешом от содержимого
     * Взято из MDS_FILE_TYPES:
     * https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Direct/Storage/Types.pm
     *
     * @return true - filename указан
     */
    private boolean hasCustomName(MdsFileMetadata mdsFileMetadata) {
        switch (mdsFileMetadata.getType()) {
            case BANNER_IMAGES_UPLOADS:
            case API_FORECAST_NAMELESS:
            case API_WORDSTAT_NAMELESS:
            case MOD_LICENSES:
            case API5_OFFLINE_REPORT:
            case API_REPORT_STAT:
                return false;
            default:
                return true;
        }
    }

    public Map<String, MdsFileMetadata> getMetaData(ClientId clientId, List<String> hashes) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return mdsFileRepository.getMetadata(shard, clientId, hashes);
    }

    public Map<Long, String> getCustomNames(ClientId clientId, List<Long> mdsIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return mdsFileRepository.getCustomNames(shard, mdsIds);
    }

    public void updateCreateTime(ClientId clientId, Long fileId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        mdsFileRepository.updateCreateTimeMetadata(shard, clientId, fileId);
    }

    public String getUrl(MdsFileSaveRequest mdsFileSaveRequest) {
        return getUrl(mdsFileSaveRequest.getMdsMetadata());
    }

    public String getUrl(MdsFileMetadata mdsFileMetadata) {
        MdsFileKey parsedKey = MdsFileKey.parse(mdsFileMetadata.getMdsKey());
        return directFilesMds.downloadUrl("get", parsedKey);
    }

    public static MdsStorageHost getStorageHost(MdsHolder mdsHolder) {
        String host = mdsHolder.getHosts().getHostPortForRead().getHost().toString();
        MdsMetadataStorageHost dbhost = Arrays.stream(MdsMetadataStorageHost.values())
                .filter(v -> v.getLiteral().equals(host))
                .findFirst()
                .orElseThrow(() -> new IllegalStateException("No such mds host: " + host));

        return MdsStorageHost.fromSource(dbhost);
    }
}
