package ru.yandex.direct.core.entity.metrika.container;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.metrika.client.model.response.CounterGoal;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.metrika.client.model.response.GetExistentCountersResponseItem;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;

import static ru.yandex.direct.core.entity.metrika.service.campaigngoals.CampaignGoalsUtils.convertToCountersById;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@ParametersAreNonnullByDefault
public class MetrikaInformationForCampaignGoals {
    private final List<GetExistentCountersResponseItem> campaignsCounters;

    private final List<UserCountersExtended> userCounters;
    private final Map<Integer, CounterInfoDirect> availableCountersById;
    private final Set<CounterInfoDirect> availableCounters;
    private final Map<Integer, CounterInfoDirect> allCountersById;

    private final Map<Integer, List<CounterGoal>> goalsByCounterId;
    private final Map<Long, CounterInfoDirect> counterByGoalId;
    private final Map<Long, CounterGoal> goals;

    public MetrikaInformationForCampaignGoals(List<GetExistentCountersResponseItem> campaignsCounters,
                                              List<UserCountersExtended> userCounters,
                                              Map<Integer, List<CounterGoal>> goalsByCounterId) {
        this.campaignsCounters = campaignsCounters;
        this.userCounters = userCounters;
        this.availableCountersById = convertToCountersById(userCounters);
        Map<Integer, GetExistentCountersResponseItem> campaignsCounterById = listToMap(campaignsCounters,
                c -> c.getCounterId().intValue());

        availableCounters = Set.copyOf(availableCountersById.values());

        this.goalsByCounterId = goalsByCounterId;

        this.counterByGoalId = EntryStream.of(goalsByCounterId)
                .flatMapValues(Collection::stream)
                .mapValues(goal -> (long) goal.getId())
                .mapKeys(counterId -> availableCountersById.get(counterId) != null ?
                        availableCountersById.get(counterId) :
                        toCounterInfoDirect(campaignsCounterById.get(counterId))
                )
                //так как цели запрашиваются и для недоступных счетчиков справочника
                //информации о счетчике может не быть
                .nonNullKeys()
                .invert()
                .nonNullValues()
                .toMap();

        goals = StreamEx.ofValues(goalsByCounterId)
                .flatMap(Collection::stream)
                .mapToEntry(CounterGoal::getId, Function.identity())
                .mapKeys(Integer::longValue)
                .toMap();

        Set<Integer> allCounterIds = Sets.union(availableCountersById.keySet(), campaignsCounterById.keySet());
        allCountersById = StreamEx.of(allCounterIds)
                .mapToEntry(counterId -> availableCountersById.get(counterId) != null ?
                        availableCountersById.get(counterId) :
                        toCounterInfoDirect(campaignsCounterById.get(counterId)))
                .nonNullValues()
                .toMap();
    }

    private CounterInfoDirect toCounterInfoDirect(@Nullable GetExistentCountersResponseItem counter) {
        if (counter == null) {
            return null;
        }
        return new CounterInfoDirect()
                .withId(counter.getCounterId().intValue())
                .withCounterSource(counter.getCounterSource())
                .withEcommerce(nvl(counter.getEcommerce(), false));
    }

    public List<GetExistentCountersResponseItem> getCampaignsCounters() {
        return campaignsCounters;
    }

    public List<UserCountersExtended> getUserCounters() {
        return userCounters;
    }

    public Set<CounterInfoDirect> getAvailableCounters() {
        return availableCounters;
    }

    public Set<Integer> getAvailableCounterIds() {
        return availableCountersById.keySet();
    }

    public Map<Integer, List<CounterGoal>> getGoalsByCounterId() {
        return goalsByCounterId;
    }

    public CounterInfoDirect getCounterByGoalId(Long goalId) {
        return counterByGoalId.get(goalId);
    }

    public CounterInfoDirect getCounterById(Integer counterId) {
        return allCountersById.get(counterId);
    }

    public boolean isGoalAvailableInMetrika(Long goalId) {
        return counterByGoalId.containsKey(goalId);
    }

    public Map<Long, CounterGoal> getGoals() {
        return goals;
    }
}
