package ru.yandex.direct.core.entity.metrika.repository;

import java.time.Instant;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.jooq.tools.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.metrika.container.CounterIdWithDomain;
import ru.yandex.direct.grid.schema.yt.Tables;
import ru.yandex.direct.grid.schema.yt.tables.CalltrackingNumberClicks;
import ru.yandex.direct.ytcomponents.service.CalltrackingNumberClicksDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static org.jooq.impl.DSL.max;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.YtTableUtils.longValueGetter;
import static ru.yandex.direct.ytwrapper.YtTableUtils.stringValueGetter;

@Repository
public class CalltrackingNumberClicksRepository {

    private static final Logger logger = LoggerFactory.getLogger(CalltrackingNumberClicksRepository.class);

    private static final CalltrackingNumberClicks CALLTRACKING_NUMBER_CLICKS =
            Tables.CALLTRACKING_NUMBER_CLICKS.as("C");
    private static final Field<Long> COUNTER_ID_FIELD = aliased(CALLTRACKING_NUMBER_CLICKS.COUNTER_ID);
    private static final Field<String> PHONE_FIELD = aliased(CALLTRACKING_NUMBER_CLICKS.NUM);
    private static final Field<Long> COUNT_FIELD = aliased(CALLTRACKING_NUMBER_CLICKS.COUNT);
    private static final Field<String> DOMAIN_FIELD = aliased(CALLTRACKING_NUMBER_CLICKS.DOMAIN);

    private final CalltrackingNumberClicksDynContextProvider dynContextProvider;

    @Autowired
    public CalltrackingNumberClicksRepository(CalltrackingNumberClicksDynContextProvider dynContextProvider) {
        this.dynContextProvider = dynContextProvider;
    }

    /**
     * @return отображение: счетчик -> телефоны и количество кликов на заданном домене с этим счетчиком
     */
    public Map<Long, Map<String, Integer>> getClicksOnPhonesByCounters(String domain, Collection<Long> counterIds) {
        if (StringUtils.isEmpty(domain)) {
            return Map.of();
        }
        try {
            var query = YtDSL.ytContext()
                    .select(COUNTER_ID_FIELD, PHONE_FIELD, COUNT_FIELD)
                    .from(CALLTRACKING_NUMBER_CLICKS)
                    .where(CALLTRACKING_NUMBER_CLICKS.DOMAIN.eq(domain))
                    .and(CALLTRACKING_NUMBER_CLICKS.COUNTER_ID.in(counterIds));
            UnversionedRowset rowset = dynContextProvider.getContext().executeTimeoutSafeSelect(query);

            TableSchema tableSchema = rowset.getSchema();

            Map<Long, Map<String, Integer>> result = new HashMap<>();
            for (UnversionedRow row : rowset.getRows()) {
                Long counterId = longValueGetter(tableSchema, COUNTER_ID_FIELD).apply(row);
                String number = stringValueGetter(tableSchema, PHONE_FIELD).apply(row);
                int count = longValueGetter(tableSchema, COUNT_FIELD).apply(row).intValue();

                result.computeIfAbsent(counterId, key -> new HashMap<>()).put(number, count);
            }
            return result;
        } catch (RuntimeException e) {
            logger.error("Failed to get phone clicks for domain {} and counters {}", domain, counterIds, e);
            return Map.of();
        }
    }

    /**
     * Returns last click times by phones.
     * [{phone -> timestamp}]
     */
    public Map<String, LocalDateTime> getLastClickTimesByPhones(
            Collection<Long> counterIds,
            List<String> telephonyPhones) {
        if (telephonyPhones.isEmpty()) {
            return Map.of();
        }
        try {
            var query = YtDSL.ytContext()
                    .select(PHONE_FIELD,
                            max(CALLTRACKING_NUMBER_CLICKS.TIMESTAMP).as(CALLTRACKING_NUMBER_CLICKS.TIMESTAMP))
                    .from(CALLTRACKING_NUMBER_CLICKS)
                    .where(CALLTRACKING_NUMBER_CLICKS.COUNTER_ID.in(counterIds))
                    .and(CALLTRACKING_NUMBER_CLICKS.NUM.in(telephonyPhones))
                    .groupBy(PHONE_FIELD);
            UnversionedRowset rowset = dynContextProvider.getContext().executeTimeoutSafeSelect(query);

            TableSchema tableSchema = rowset.getSchema();

            Map<String, LocalDateTime> result = new HashMap<>();
            for (UnversionedRow row : rowset.getRows()) {
                String number = stringValueGetter(tableSchema, PHONE_FIELD).apply(row);
                long timestamp = longValueGetter(tableSchema, CALLTRACKING_NUMBER_CLICKS.TIMESTAMP).apply(row);
                LocalDateTime lastClickTime = LocalDateTime.ofInstant(Instant.ofEpochSecond(timestamp), MSK);
                result.put(number, lastClickTime);
            }
            return result;
        } catch (RuntimeException e) {
            logger.error("Failed to get click times for phones {} and counters {}", telephonyPhones, counterIds, e);
            return Map.of();
        }
    }

    /**
     * @return отображение: счетчик+домен -> телефоны и количество кликов на этом домене с этим счетчиком
     */
    public Map<CounterIdWithDomain, Map<String, Integer>> getClicksOnPhonesByDomainWithCounterIds(
            Collection<CounterIdWithDomain> counterIdWithDomains) {
        if (counterIdWithDomains.isEmpty()) {
            return Map.of();
        }
        try {
            var query = YtDSL.ytContext()
                    .select(DOMAIN_FIELD, COUNTER_ID_FIELD, PHONE_FIELD, COUNT_FIELD)
                    .from(CALLTRACKING_NUMBER_CLICKS)
                    .where(buildConditionByCounterIdWithDomain(counterIdWithDomains));

            UnversionedRowset rowset = dynContextProvider.getContext().executeTimeoutSafeSelect(query);

            TableSchema tableSchema = rowset.getSchema();

            Map<CounterIdWithDomain, Map<String, Integer>> result = new HashMap<>();
            for (UnversionedRow row : rowset.getRows()) {
                String domain = stringValueGetter(tableSchema, DOMAIN_FIELD).apply(row);
                Long counterId = longValueGetter(tableSchema, COUNTER_ID_FIELD).apply(row);
                String number = stringValueGetter(tableSchema, PHONE_FIELD).apply(row);
                int count = longValueGetter(tableSchema, COUNT_FIELD).apply(row).intValue();

                CounterIdWithDomain index = new CounterIdWithDomain(counterId, domain);
                result.computeIfAbsent(index, key -> new HashMap<>()).put(number, count);
            }
            return result;
        } catch (RuntimeException e) {
            logger.error("Failed to get phone clicks for domains and counters {}", counterIdWithDomains, e);
            return Map.of();
        }
    }

    private Condition buildConditionByCounterIdWithDomain(Collection<CounterIdWithDomain> counterIdWithDomains) {
        Condition[] conditions = counterIdWithDomains.stream()
                .map(item -> DSL.and(
                        CALLTRACKING_NUMBER_CLICKS.COUNTER_ID.eq(item.getCounterId()),
                        CALLTRACKING_NUMBER_CLICKS.DOMAIN.eq(item.getDomain())))
                .toArray(Condition[]::new);

        return DSL.or(conditions);
    }
}
