package ru.yandex.direct.core.entity.metrika.repository;

import java.util.HashSet;
import java.util.Set;

import org.jooq.Field;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.schema.yt.Tables;
import ru.yandex.direct.grid.schema.yt.tables.CalltrackingPhoneWithoutReplacements;
import ru.yandex.direct.ytcomponents.service.CalltrackingNumbersWithoutChangesDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.YtTableUtils.stringValueGetter;

@Repository
public class CalltrackingPhonesWithoutReplacementsRepository {
    private static final Logger logger = LoggerFactory.getLogger(CalltrackingPhonesWithoutReplacementsRepository.class);

    private static final CalltrackingPhoneWithoutReplacements CALLTRACKING_PHONE_WITHOUT_REPLACEMENTS =
            Tables.CALLTRACKING_PHONE_WITHOUT_REPLACEMENTS.as("C");
    private static final Field<String> FROM_PHONE_FIELD = aliased(CALLTRACKING_PHONE_WITHOUT_REPLACEMENTS.FROM_PHONE);

    private final CalltrackingNumbersWithoutChangesDynContextProvider dynContextProvider;

    @Autowired
    public CalltrackingPhonesWithoutReplacementsRepository(
            CalltrackingNumbersWithoutChangesDynContextProvider dynContextProvider) {
        this.dynContextProvider = dynContextProvider;
    }

    /**
     * Возвращает телефонные номера, для которых за определенную дату не сработала подмена на сайте
     */
    public Set<String> getPhonesWithoutReplacements(ClientId clientId, Set<Long> counterIds, Long timestamp) {
        try {
            var query = YtDSL.ytContext()
                    .select(FROM_PHONE_FIELD)
                    .from(CALLTRACKING_PHONE_WITHOUT_REPLACEMENTS)
                    .where(CALLTRACKING_PHONE_WITHOUT_REPLACEMENTS.CLIENT_ID.eq(clientId.asLong()))
                    .and(CALLTRACKING_PHONE_WITHOUT_REPLACEMENTS.COUNTER_ID.in(counterIds))
                    .and(CALLTRACKING_PHONE_WITHOUT_REPLACEMENTS.TIMESTAMP.eq(timestamp))
                    .and(CALLTRACKING_PHONE_WITHOUT_REPLACEMENTS.SUCCESSFUL_CHANGES.eq(0L));
            UnversionedRowset rowset = dynContextProvider.getContext().executeTimeoutSafeSelect(query);

            TableSchema tableSchema = rowset.getSchema();

            Set<String> result = new HashSet<>();
            for (UnversionedRow row : rowset.getRows()) {
                String phone = stringValueGetter(tableSchema, FROM_PHONE_FIELD).apply(row);
                result.add(phone);
            }
            return result;
        } catch (RuntimeException e) {
            logger.warn(
                    "Got an exception when querying for phones without replacements from YT table " +
                            "for clientID = {}, counterIds = {} and timestamp = {}",
                    clientId, counterIds, timestamp, e
            );
            return Set.of();
        }
    }

}
