package ru.yandex.direct.core.entity.metrika.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.dbschema.ppcdict.tables.records.LalSegmentsRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.util.RepositoryUtils.TRUE;
import static ru.yandex.direct.dbschema.ppcdict.tables.LalSegments.LAL_SEGMENTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Репозиторий для работы с LAL-сегментами.
 * <p>
 * LAL — look-alike сегмент, описывающий группу пользователей максимально похожую на целевую
 * (родительскую цель). LAL-сегменты создаются и живут в директе, информация о них экспортируется
 * в крипту с помощью {@code PpcDataExportJob} (см. соответствующую
 * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/jobs/src/main/resources/export/ppcdataexport/crypta/exportLalSegment.conf">конфигурацию</a>).
 *
 * @see <a href="https://wiki.yandex-team.ru/users/kozobrodov/lalsegments/">Техдизайн</a>
 */
@Repository
@ParametersAreNonnullByDefault
public class LalSegmentRepository {
    private static final Logger logger = LoggerFactory.getLogger(LalSegmentRepository.class);

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<Goal> jooqMapper;
    private final ShardHelper shardHelper;

    @Autowired
    public LalSegmentRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(Goal::new)
            .map(property(Goal.ID, LAL_SEGMENTS.LAL_SEGMENT_ID))
            .map(property(Goal.PARENT_ID, LAL_SEGMENTS.PARENT_GOAL_ID))
            .build();
    }

    public JooqMapperWithSupplier<Goal> getLalSegmentMapper() {
        return jooqMapper;
    }

    public List<Goal> getLalSegmentsByIds(Collection<Long> lalSegmentsIds) {
        if (lalSegmentsIds.isEmpty()) {
            return Collections.emptyList();
        }
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(LAL_SEGMENTS)
                .where(LAL_SEGMENTS.LAL_SEGMENT_ID.in(lalSegmentsIds))
                .orderBy(LAL_SEGMENTS.LAL_SEGMENT_ID)
                .fetch(jooqMapper::fromDb);
    }

    public List<Goal> getLalSegmentsByParentIds(Collection<Long> parentGoalIds) {
        return dslContextProvider.ppcdict()
            .select(jooqMapper.getFieldsToRead())
            .from(LAL_SEGMENTS)
            .where(LAL_SEGMENTS.PARENT_GOAL_ID.in(parentGoalIds))
            .orderBy(LAL_SEGMENTS.LAL_SEGMENT_ID)
            .fetch(jooqMapper::fromDb);
    }

    public List<Goal> createLalSegments(Collection<Long> parentGoalIds) {
        if (parentGoalIds.isEmpty()) {
            return List.of();
        }

        var uniqueParentGoalIds = new HashSet<>(parentGoalIds);
        var generatedIds = shardHelper.generateLalSegmentIds(uniqueParentGoalIds.size()).iterator();

        InsertHelper<LalSegmentsRecord> insertHelper = new InsertHelper<>(dslContextProvider.ppcdict(), LAL_SEGMENTS);
        insertHelper.addAll(jooqMapper, mapList(uniqueParentGoalIds,
                parentGoalId -> (Goal) new Goal().withId(generatedIds.next()).withParentId(parentGoalId)));
        int rowsInserted = insertHelper.execute();
        logger.debug("Rows inserted: {}", rowsInserted);

        return getLalSegmentsByParentIds(parentGoalIds);
    }

    public void activateLalSegments(Collection<Long> lalIds) {
        if (lalIds.isEmpty()) {
            return;
        }
        dslContextProvider.ppcdict()
                .update(LAL_SEGMENTS)
                .set(LAL_SEGMENTS.IS_ACTIVE, TRUE)
                .set(LAL_SEGMENTS.SYNC_TIME, DSL.currentLocalDateTime())
                .where(LAL_SEGMENTS.LAL_SEGMENT_ID.in(lalIds))
                .execute();
    }
}
