package ru.yandex.direct.core.entity.metrika.repository;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Record;
import org.jooq.Record3;
import org.jooq.Result;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.metrika.model.GroupForMetrika;
import ru.yandex.direct.core.entity.metrika.model.objectinfo.AdGroupInfoForMetrika;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Arrays.asList;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.utils.CommonUtils.mapByKey;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class MetrikaAdGroupRepository {
    private final DslContextProvider ppcDslContextProvider;
    private final JooqMapperWithSupplier<GroupForMetrika> jooqMapper;

    @Autowired
    public MetrikaAdGroupRepository(DslContextProvider ppcDslContextProvider) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(GroupForMetrika::new)
                .map(property(GroupForMetrika.GROUP_NAME, PHRASES.GROUP_NAME))
                .map(property(GroupForMetrika.PID, PHRASES.PID))
                .map(property(GroupForMetrika.CID, PHRASES.CID))
                .build();
    }

    public Map<Long, GroupForMetrika> getGroups(int shard, List<Long> groupIds) {
        Result<Record3<Long, Long, String>> groups = ppcDslContextProvider.ppc(shard)
                .selectDistinct(PHRASES.PID, PHRASES.CID, PHRASES.GROUP_NAME)
                .from(PHRASES)
                .where(PHRASES.PID.in(groupIds))
                .fetch();

        return StreamEx.of(groups)
                .map(jooqMapper::fromDb)
                .collect(mapByKey(GroupForMetrika::getPid));
    }

    public List<AdGroupInfoForMetrika> getAdGroupsInfo(int shard, @Nullable LimitOffset limitOffset) {
        return getAdGroupsInfo(shard, LocalDateTime.MIN, 0L, limitOffset);
    }

    public List<AdGroupInfoForMetrika> getAdGroupsInfo(int shard, LocalDateTime lastChange,
                                                       Long lastAdGroupId, @Nullable LimitOffset limitOffset) {
        limitOffset = limitOffset != null ? limitOffset : limited(ObjectInfoConstants.DEFAULT_LIMIT);
        LocalDateTime gapTimestamp = LocalDateTime.now().minusSeconds(ObjectInfoConstants.GAP_SECONDS);
        Condition afterTime = PHRASES.LAST_CHANGE.greaterThan(lastChange);
        Condition equalTimeAndGreaterId =
                PHRASES.LAST_CHANGE.equal(lastChange).and(
                        PHRASES.PID.greaterThan(lastAdGroupId));

        Result<Record> result = ppcDslContextProvider.ppc(shard)
                .select(asList(PHRASES.PID,
                        PHRASES.GROUP_NAME,
                        PHRASES.LAST_CHANGE,
                        CAMPAIGNS.ORDER_ID))
                .from(PHRASES.join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID)))
                .where(afterTime.or(equalTimeAndGreaterId))
                .and(PHRASES.LAST_CHANGE.lessThan(gapTimestamp))
                .and(PHRASES.PRIORITY_ID.greaterThan(0L))
                .orderBy(PHRASES.LAST_CHANGE, PHRASES.PID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch();

        return mapList(result, record -> new AdGroupInfoForMetrika()
                .withId(record.getValue(PHRASES.PID))
                .withOrderId(record.getValue(CAMPAIGNS.ORDER_ID))
                .withName(record.getValue(PHRASES.GROUP_NAME))
                .withLastChange(record.getValue(PHRASES.LAST_CHANGE)));
    }

}
