package ru.yandex.direct.core.entity.metrika.repository;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectForUpdateStep;
import org.jooq.Table;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.BannerImageOpts;
import ru.yandex.direct.core.entity.metrika.container.BannerWithTitleAndBody;
import ru.yandex.direct.core.entity.metrika.model.BannerForMetrika;
import ru.yandex.direct.core.entity.metrika.model.objectinfo.BannerInfoForMetrika;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Arrays.asList;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.utils.CommonUtils.mapByKey;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class MetrikaBannerRepository {

    private static final Field<Long> NESTED_SORT_ID = DSL.field("sort_id", Long.class);
    private static final Field<Long> NESTED_BID = DSL.field("bid", Long.class);
    private static final Field<Long> NESTED_BANNERID = DSL.field("BannerID", Long.class);
    private static final Field<LocalDateTime> NESTED_LAST_CHANGE = DSL.field("last_change", LocalDateTime.class);
    private static final int CHUNK_SIZE = 1000;

    private final DslContextProvider ppcDslContextProvider;
    private final JooqMapperWithSupplier<BannerForMetrika> bannersMapper;
    private final JooqMapperWithSupplier<BannerForMetrika> bannerImagesMapper;


    @Autowired
    public MetrikaBannerRepository(DslContextProvider ppcDslContextProvider) {
        this.bannersMapper = commonMapper()
                .map(property(BannerForMetrika.BANNER_ID, BANNERS.BANNER_ID))
                .build();
        this.bannerImagesMapper = commonMapper()
                .map(property(BannerForMetrika.BANNER_ID, BANNER_IMAGES.BANNER_ID))
                .build();

        this.ppcDslContextProvider = ppcDslContextProvider;
    }

    public Map<Long, BannerForMetrika> getBanners(int shard, List<Long> bannerIds) {
        return StreamEx.of(ppcDslContextProvider.ppc(shard)
                .select(BANNERS.BANNER_ID, BANNERS.BID, BANNERS.CID, BANNERS.TITLE, BANNERS.BODY, BANNERS.DOMAIN)
                .from(BANNERS)
                .where(BANNERS.BANNER_ID.in(bannerIds))
                .fetch())
                .map(r -> this.bannersMapper.fromDb(r).withIsImage(false))
                .collect(mapByKey(BannerForMetrika::getBannerId));
    }

    public Map<Long, BannerForMetrika> getImageBanners(int shard, List<Long> bannerIds) {
        return StreamEx.of(ppcDslContextProvider.ppc(shard)
                .select(BANNER_IMAGES.BANNER_ID, BANNERS.BID, BANNERS.CID, BANNERS.TITLE, BANNERS.BODY, BANNERS.DOMAIN)
                .from(BANNER_IMAGES)
                .join(BANNERS).on(BANNERS.BID.eq(BANNER_IMAGES.BID))
                .where(BANNER_IMAGES.BANNER_ID.in(bannerIds))
                .fetch())
                .map(r -> this.bannerImagesMapper.fromDb(r).withIsImage(true))
                .collect(mapByKey(BannerForMetrika::getBannerId));
    }

    public List<BannerInfoForMetrika> getBannersInfo(int shard, @Nullable LimitOffset limitOffset) {
        return getBannersInfo(shard, LocalDateTime.MIN, 0L, limitOffset);
    }

    /**
     * Возвращает список объектов, отсортированных по LastChange + sort_id,
     * где sort_id - это смесь из banners.bid и banner_images.image_id.
     * <p>
     * Выборка смеси баннеров и картиночных баннеров по условию:
     * (LastChange больше указанного) или
     * (LastChange равно указанному и sort_id больше указанного).
     */
    @QueryWithoutIndex("Индекс тут есть, нужно дорабатывать ExplainListener")
    public List<BannerInfoForMetrika> getBannersInfo(int shard, LocalDateTime lastChange,
                                                     Long lastSortId, @Nullable LimitOffset limitOffset) {
        limitOffset = limitOffset != null ? limitOffset : limited(ObjectInfoConstants.DEFAULT_LIMIT);
        Table<Record> changedBannersAndImageBannersUnion =
                createDerivedTableForChangedBannersAndBannerImages(
                        shard, lastChange, lastSortId, limitOffset);

        Result<Record> result = ppcDslContextProvider.ppc(shard)
                .select(changedBannersAndImageBannersUnion.fields())
                .select(BANNERS.BODY, BANNERS.TITLE)
                .from(changedBannersAndImageBannersUnion)
                .join(BANNERS)
                .on(BANNERS.BID.eq(changedBannersAndImageBannersUnion.field(NESTED_BID)))
                .fetch();

        return mapList(result, r -> new BannerInfoForMetrika()
                .withSortId(r.getValue(NESTED_SORT_ID))
                .withBannerId(r.getValue(NESTED_BID))
                .withBsBannerId(r.getValue(NESTED_BANNERID))
                .withBody(r.getValue(BANNERS.BODY))
                .withTitle(r.getValue(BANNERS.TITLE))
                .withLastChange(r.getValue(NESTED_LAST_CHANGE)));
    }

    private Table<Record> createDerivedTableForChangedBannersAndBannerImages(int shard,
                                                                             LocalDateTime lastChangeTimestamp,
                                                                             Long lastSortId, LimitOffset limitOffset) {
        LocalDateTime gapTimestamp = LocalDateTime.now().minusSeconds(ObjectInfoConstants.GAP_SECONDS);

        Condition bannersAfterTime = BANNERS.LAST_CHANGE.greaterThan(lastChangeTimestamp);
        Condition bannersEqualTimeAndGreaterOrEqualId = BANNERS.LAST_CHANGE.equal(lastChangeTimestamp)
                .and(BANNERS.BID.greaterOrEqual(lastSortId));

        Condition bannerImagesAfterTime = BANNER_IMAGES.DATE_ADDED.greaterThan(lastChangeTimestamp);
        Condition bannerImagesEqualTimeAndGreaterOrEqualId = BANNER_IMAGES.DATE_ADDED.equal(lastChangeTimestamp)
                .and(BANNER_IMAGES.IMAGE_ID.greaterOrEqual(lastSortId));

        SelectForUpdateStep<Record> changedBanners = ppcDslContextProvider.ppc(shard)
                .select(asList(
                        BANNERS.BID.as(NESTED_SORT_ID),
                        BANNERS.BID.as(NESTED_BID),
                        BANNERS.BANNER_ID.as(NESTED_BANNERID),
                        BANNERS.LAST_CHANGE.as(NESTED_LAST_CHANGE)))
                .from(BANNERS)
                .where(BANNERS.BANNER_ID.greaterThan(0L))
                .and(bannersAfterTime.or(bannersEqualTimeAndGreaterOrEqualId))
                .and(BANNERS.LAST_CHANGE.lessThan(gapTimestamp))
                .orderBy(BANNERS.LAST_CHANGE, BANNERS.BID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset());

        SelectForUpdateStep<Record> changedBannerImages = ppcDslContextProvider.ppc(shard)
                .select(asList(
                        BANNER_IMAGES.IMAGE_ID.as(NESTED_SORT_ID),
                        BANNER_IMAGES.BID.as(NESTED_BID),
                        BANNER_IMAGES.BANNER_ID.as(NESTED_BANNERID),
                        BANNER_IMAGES.DATE_ADDED.as(NESTED_LAST_CHANGE)))
                .from(BANNER_IMAGES)
                .where(BANNER_IMAGES.BANNER_ID.greaterThan(0L))
                .and(bannerImagesAfterTime.or(bannerImagesEqualTimeAndGreaterOrEqualId))
                .and(BANNER_IMAGES.DATE_ADDED.lessThan(gapTimestamp))
                .and(BANNER_IMAGES.OPTS.isNull()
                        .or(BANNER_IMAGES.OPTS.notContains(BannerImageOpts.SINGLE_AD_TO_BS.getTypedValue())))
                .orderBy(BANNER_IMAGES.DATE_ADDED, BANNER_IMAGES.IMAGE_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset());

        return ppcDslContextProvider.ppc(shard)
                .select(asList(NESTED_SORT_ID, NESTED_BANNERID, NESTED_LAST_CHANGE, NESTED_BID))
                .from(changedBanners.unionAll(changedBannerImages).asTable())
                .orderBy(NESTED_LAST_CHANGE, NESTED_SORT_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .asTable();
    }

    public List<BannerWithTitleAndBody> getBannerWithTitleAndBodyFromImageIds(int shard, Set<Long> imageIds) {
        List<BannerWithTitleAndBody> result = new ArrayList<>();
        for (var chunkedImageId : Iterables.partition(imageIds, CHUNK_SIZE)) {
            result.addAll(
                    ppcDslContextProvider.ppc(shard)
                            .select(
                                    BANNER_IMAGES.BID,
                                    BANNER_IMAGES.BANNER_ID,
                                    BANNERS.TITLE,
                                    BANNERS.BODY)
                            .from(BANNER_IMAGES)
                            .join(BANNERS).using(BANNER_IMAGES.BID)
                            .where(BANNER_IMAGES.IMAGE_ID.in(chunkedImageId))
                            .and(BANNER_IMAGES.BANNER_ID.ne(0L))
                            .fetch(
                                    r -> new BannerWithTitleAndBody(r.getValue(BANNER_IMAGES.BID),
                                            r.getValue(BANNER_IMAGES.BANNER_ID), r.getValue(BANNERS.TITLE),
                                            r.getValue(BANNERS.BODY))
                            )
            );
        }
        return result;
    }

    public Map<Long, BannerWithTitleAndBody> getBannerWithTitleAndBodyFromBids(int shard, Set<Long> bids) {
        Map<Long, BannerWithTitleAndBody> result = new HashMap<>();
        for (var chunkedBids : Iterables.partition(bids, CHUNK_SIZE)) {
            result.putAll(
                    ppcDslContextProvider.ppc(shard)
                            .select(
                                    BANNERS.BID,
                                    BANNERS.BANNER_ID,
                                    BANNERS.TITLE,
                                    BANNERS.BODY)
                            .from(BANNERS)
                            .where(BANNERS.BID.in(chunkedBids))
                            .and(BANNERS.BANNER_ID.ne(0L))
                            .fetchMap(
                                    r -> r.getValue(BANNERS.BID),
                                    r -> new BannerWithTitleAndBody(r.getValue(BANNERS.BID),
                                            r.getValue(BANNERS.BANNER_ID), r.getValue(BANNERS.TITLE),
                                            r.getValue(BANNERS.BODY))
                            )
            );
        }
        return result;
    }


    private static JooqMapperWithSupplierBuilder<BannerForMetrika> commonMapper() {
        return JooqMapperWithSupplierBuilder.builder(BannerForMetrika::new)
                .map(property(BannerForMetrika.BID, BANNERS.BID))
                .map(property(BannerForMetrika.CID, BANNERS.CID))
                .map(property(BannerForMetrika.DOMAIN, BANNERS.DOMAIN))
                .map(property(BannerForMetrika.TITLE, BANNERS.TITLE))
                .map(property(BannerForMetrika.BODY, BANNERS.BODY));
    }
}
