package ru.yandex.direct.core.entity.metrika.repository;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.Record2;
import org.jooq.Record3;
import org.jooq.RecordMapper;
import org.jooq.Result;
import org.jooq.ResultQuery;
import org.jooq.SelectConditionStep;
import org.jooq.TableField;
import org.jooq.UpdateConditionStep;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.IdModFilter;
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.core.entity.metrika.model.CampaignForMetrika;
import ru.yandex.direct.core.entity.metrika.model.CampaignWithCounterForMetrika;
import ru.yandex.direct.core.entity.metrika.model.GoalCampUsages;
import ru.yandex.direct.core.entity.metrika.model.GoalsByUsageType;
import ru.yandex.direct.core.entity.metrika.model.TurbolandingGoalType;
import ru.yandex.direct.core.entity.metrika.model.objectinfo.CampaignInfoForMetrika;
import ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoal;
import ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoalId;
import ru.yandex.direct.core.entity.retargeting.model.GoalRole;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.MetrikaCountersSource;
import ru.yandex.direct.dbschema.ppc.tables.records.CampMetrikaGoalsRecord;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toList;
import static org.jooq.impl.DSL.choose;
import static org.jooq.impl.DSL.groupConcat;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.convertibleEnumSet;
import static ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoal.CAMPAIGN_ID;
import static ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoal.CONTEXT_GOALS_COUNT;
import static ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoal.GOALS_COUNT;
import static ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoal.GOAL_ID;
import static ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoal.GOAL_ROLE;
import static ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoal.LINKS_COUNT;
import static ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoal.STAT_DATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.CALLTRACKING_SETTINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_CALLTRACKING_SETTINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_METRIKA_COUNTERS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_METRIKA_GOALS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_TURBOLANDING_METRIKA_COUNTERS;
import static ru.yandex.direct.dbschema.ppc.Tables.METRIKA_COUNTERS;
import static ru.yandex.direct.dbschema.ppc.Tables.SITELINKS_LINKS;
import static ru.yandex.direct.dbschema.ppc.Tables.SITELINKS_SET_TO_LINK;
import static ru.yandex.direct.dbschema.ppc.Tables.TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.TURBOLANDING_METRIKA_COUNTERS;
import static ru.yandex.direct.dbschema.ppc.Tables.TURBOLANDING_METRIKA_GOALS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class MetrikaCampaignRepository {
    private static final Logger logger = LoggerFactory.getLogger(MetrikaCampaignRepository.class);

    private static final Long EMPTY_ORDER_ID = 0L;

    private final DslContextProvider ppcDslContextProvider;
    private final JooqMapperWithSupplier<CampaignForMetrika> campignsMetrikaMapper;
    private final JooqMapperWithSupplier<CampaignForMetrika> turbolandingsMetrikaMapper;
    private final JooqMapperWithSupplier<CampMetrikaGoal> campMetrikaGoalMapper;
    private final JooqMapperWithSupplier<CampMetrikaGoalId> campMetrikaGoalIdMapper;

    @Autowired
    public MetrikaCampaignRepository(DslContextProvider ppcDslContextProvider) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.campignsMetrikaMapper = mapperWithCampaignMetrikaCounters();
        this.turbolandingsMetrikaMapper = mapperWithTurbolandingMetrikaCounter();
        this.campMetrikaGoalMapper = createCampMetrikaGoalsMapper();
        this.campMetrikaGoalIdMapper = JooqMapperWithSupplierBuilder.builder(CampMetrikaGoalId::new)
                .map(property(CampMetrikaGoalId.CAMPAIGN_ID, CAMP_METRIKA_GOALS.CID))
                .map(property(CampMetrikaGoalId.GOAL_ID, CAMP_METRIKA_GOALS.GOAL_ID))
                .build();
    }

    public List<CampaignForMetrika> getCampaignsForMetrika(int shard, List<Long> orderIds) {
        DSLContext dslContext = ppcDslContextProvider.ppc(shard);

        List<CampaignForMetrika> campsList = dslContext
                .select(campignsMetrikaMapper.getFieldsToRead())
                .from(CAMPAIGNS)
                .leftJoin(CAMP_METRIKA_COUNTERS).on(CAMPAIGNS.CID.eq(CAMP_METRIKA_COUNTERS.CID))
                .leftJoin(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.ORDER_ID.in(orderIds))
                .fetch(campignsMetrikaMapper::fromDb);

        return processSystemCounters(dslContext, campsList);
    }

    public List<CampaignWithCounterForMetrika> getCampaignsWithCounters(int shard, IdModFilter cidFilter) {
        DSLContext dslContext = ppcDslContextProvider.ppc(shard);

        SelectConditionStep<Record> rowsCampMetrikaCountersCondition = dslContext
                .selectDistinct(campignsMetrikaMapper.getFieldsToRead())
                .from(CAMPAIGNS)
                .join(CAMP_METRIKA_COUNTERS).on(CAMPAIGNS.CID.eq(CAMP_METRIKA_COUNTERS.CID))
                .leftJoin(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CID.mod(cidFilter.getDivisor()).eq(cidFilter.getRemainder()))
                .and(CAMPAIGNS.ORDER_ID.ne(EMPTY_ORDER_ID))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No));

        SelectConditionStep<Record3<Long, Long, Long>> sitelinkTurbolandingsCondition = dslContext
                .selectDistinct(CAMPAIGNS.ORDER_ID, CAMPAIGNS.CID, TURBOLANDING_METRIKA_COUNTERS.COUNTER)
                .from(CAMPAIGNS)
                .leftJoin(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .join(BANNERS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .join(SITELINKS_SET_TO_LINK).on(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID.eq(BANNERS.SITELINKS_SET_ID))
                .join(SITELINKS_LINKS).on(SITELINKS_LINKS.SL_ID.eq(SITELINKS_SET_TO_LINK.SL_ID))
                .join(TURBOLANDING_METRIKA_COUNTERS).on(TURBOLANDING_METRIKA_COUNTERS.TL_ID.eq(SITELINKS_LINKS.TL_ID))
                .where(SITELINKS_LINKS.TL_ID.gt(0L))
                .and(CAMPAIGNS.CID.mod(cidFilter.getDivisor()).eq(cidFilter.getRemainder()))
                .and(CAMPAIGNS.ORDER_ID.ne(EMPTY_ORDER_ID))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No));

        SelectConditionStep<Record3<Long, Long, Long>> bannerTurbolandingsCondition = dslContext
                .selectDistinct(CAMPAIGNS.ORDER_ID, CAMPAIGNS.CID, TURBOLANDING_METRIKA_COUNTERS.COUNTER)
                .from(CAMPAIGNS)
                .leftJoin(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .join(BANNER_TURBOLANDINGS)
                .on(BANNER_TURBOLANDINGS.CID.eq(CAMPAIGNS.CID))
                .join(TURBOLANDING_METRIKA_COUNTERS)
                .on(BANNER_TURBOLANDINGS.TL_ID.eq(TURBOLANDING_METRIKA_COUNTERS.TL_ID))
                .where(CAMPAIGNS.CID.mod(cidFilter.getDivisor()).eq(cidFilter.getRemainder()))
                .and(CAMPAIGNS.ORDER_ID.ne(EMPTY_ORDER_ID))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No));

        Result<Record> rowsCampMetrikaCounters = rowsCampMetrikaCountersCondition.fetch();

        Result<Record3<Long, Long, Long>> rowsCampTurbolandingCounters = bannerTurbolandingsCondition
                .union(sitelinkTurbolandingsCondition)
                .fetch();

        List<CampaignWithCounterForMetrika> campWithCounters = StreamEx.of(rowsCampMetrikaCounters)
                .map(campignsMetrikaMapper::fromDb)
                .map(CampaignWithCounterForMetrika.class::cast)
                .toList();

        Map<Long, List<Long>> campCountersMap = StreamEx.of(campWithCounters)
                .mapToEntry(CampaignWithCounterForMetrika::getId, CampaignWithCounterForMetrika::getMetrikaCounters)
                .toMap();
        for (Record3<Long, Long, Long> camp : rowsCampTurbolandingCounters) {
            List<Long> counters = campCountersMap.get(camp.component2());
            if (counters != null) {
                counters.add(camp.component3());
            } else {
                //Если кампании нет в списке кампаний со счетчиками, добавим ее в список, а счетчики - в map
                CampaignForMetrika additionalCampaign = turbolandingsMetrikaMapper.fromDb(camp);
                campWithCounters.add(additionalCampaign);
                campCountersMap
                        .put(
                                additionalCampaign.getId(),
                                new ArrayList<>(additionalCampaign.getMetrikaCounters())
                        );
            }
        }

        //Записываем собранные счетчики, дедуплицируя перед записью
        campWithCounters
                .forEach(camp -> camp.setMetrikaCounters(
                        campCountersMap.get(camp.getId())
                                .stream().distinct().collect(toList())
                        )
                );

        return campWithCounters;
    }

    /**
     * К "белому списку" счётчиков кампаний добавляет турбосчётчики.
     * Если "белый список" пуст, технические счётчики не добавляем, чтобы не ограничивать возможности Метрики по склейке
     * DIRECT-111776
     */
    private <T extends CampaignWithCounterForMetrika> List<T> processSystemCounters(DSLContext dslContext,
                                                                                    List<T> campsList) {
        // Смысл счётчиков в intapi ручке metrika-export/campaigns -- белый список счётчиков,
        //   к визитам которых Метрика будет клеить клики с рекламы
        // Добавляем технические счётчики, если у кампании указаны пользовательские счётчики
        //   или выключена разметка ссылок yclid'ом. Подробности в DIRECT-111776
        // хотфикс DIRECT-115372: всегда отдаём турбо-счётчики
        List<Long> processingCampaignIds = mapList(campsList, CampaignWithCounterForMetrika::getId);

        Map<Long, List<Long>> campCountersMap = StreamEx.of(campsList)
                .mapToEntry(CampaignWithCounterForMetrika::getId, CampaignWithCounterForMetrika::getMetrikaCounters)
                .removeValues(Objects::isNull)
                .toMap();

        Map<Long, Set<Long>> spravCountersByCampaignId = getSpravCountersByCampaignId(dslContext,
                processingCampaignIds);

        Map<Long, Long> calltrackingCountersByCampaignId = getCalltrackingCountersByCampaignId(dslContext,
                processingCampaignIds);

        SelectConditionStep<Record2<Long, Long>> additionalCounters = dslContext
                .select(CAMPAIGNS.CID, TURBOLANDING_METRIKA_COUNTERS.COUNTER)
                .from(CAMPAIGNS)
                .join(BANNER_TURBOLANDINGS).on(BANNER_TURBOLANDINGS.CID.eq(CAMPAIGNS.CID))
                .join(TURBOLANDING_METRIKA_COUNTERS)
                .on(BANNER_TURBOLANDINGS.TL_ID.eq(TURBOLANDING_METRIKA_COUNTERS.TL_ID))
                .where(CAMPAIGNS.CID.in(processingCampaignIds))
                // турболэндинги есть у малого числа клиентов - таким образом мы ускоряем запрос
                .andExists(dslContext.selectOne().from(TURBOLANDINGS).where(TURBOLANDINGS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID)));

        SelectConditionStep<Record2<Long, Long>> additionalSitelinkCounters = dslContext
                .selectDistinct(CAMPAIGNS.CID, TURBOLANDING_METRIKA_COUNTERS.COUNTER)
                .from(CAMPAIGNS)
                .join(BANNERS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .join(SITELINKS_SET_TO_LINK).on(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID.eq(BANNERS.SITELINKS_SET_ID))
                .join(SITELINKS_LINKS).on(SITELINKS_LINKS.SL_ID.eq(SITELINKS_SET_TO_LINK.SL_ID))
                .join(TURBOLANDING_METRIKA_COUNTERS).on(TURBOLANDING_METRIKA_COUNTERS.TL_ID.eq(SITELINKS_LINKS.TL_ID))
                .where(SITELINKS_LINKS.TL_ID.gt(0L))
                .and(CAMPAIGNS.CID.in(processingCampaignIds))
                // турболэндинги есть у малого числа клиентов - таким образом мы ускоряем запрос
                .andExists(dslContext.selectOne().from(TURBOLANDINGS).where(TURBOLANDINGS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID)));

        StreamEx.of(asList(additionalCounters, additionalSitelinkCounters))
                .map(ResultQuery::fetch)
                .forEach(counters -> counters
                        .forEach(rec -> campCountersMap.computeIfAbsent(rec.component1(), id -> new ArrayList<>())
                                .add(rec.component2())));

        campsList.forEach(camp -> processCampaignSystemCounters(campCountersMap,
                spravCountersByCampaignId,
                calltrackingCountersByCampaignId,
                camp));

        return campsList;
    }

    /**
     * Для счетчиков справочника есть особая логика, позднее эту логику сделаем и для турбов:
     * Пользователь не указал счётчик на кампании и разметка ссылок (yclid) включена: список счётчиков пустой
     * (работает провязка по yclid)
     * Пользователь указал счётчик(и) на кампании: указанные пользовательские счётчики + счетчик справочника
     * Пользователь не указал счётчик и выключил провязку про YCLID: счетчик Справочника
     */
    private void processCampaignSystemCounters(
            Map<Long, List<Long>> campCountersMap,
            Map<Long, Set<Long>> spravCountersByCampaignId,
            Map<Long, Long> calltrackingCounterByCampaignId,
            CampaignWithCounterForMetrika campaign) {
        List<Long> updatedCounters = campCountersMap.get(campaign.getId());

        if (updatedCounters == null) {
            return;
        }
        List<Long> uniqueUpdatedCounters = updatedCounters.stream().distinct().collect(toList());
        Set<Long> spravCounters = spravCountersByCampaignId.getOrDefault(campaign.getId(), emptySet());
        Long calltrackingCounter = calltrackingCounterByCampaignId.get(campaign.getId());
        boolean hasUsersCounters = !spravCounters.containsAll(uniqueUpdatedCounters);

        if (hasUsersCounters || !campaign.getHasAddMetrikaTagToUrl()) {
            var countersToAdd = StreamEx.of(uniqueUpdatedCounters)
                    .append(calltrackingCounter)
                    .filter(Objects::nonNull)
                    .distinct()
                    .toList();
            campaign.setMetrikaCounters(countersToAdd);
        } else {
            campaign.setMetrikaCounters(emptyList());
        }
    }

    public Map<Long, Set<Long>> getSpravCountersByCampaignId(int shard,
                                                             List<Long> campaignIds) {
        return getSpravCountersByCampaignId(ppcDslContextProvider.ppc(shard), campaignIds);
    }

    private Map<Long, Set<Long>> getSpravCountersByCampaignId(DSLContext dslContext,
                                                              List<Long> campaignIds) {
        Map<Long, List<Long>> counterIdsByCampaignId = dslContext
                .select(METRIKA_COUNTERS.CID, METRIKA_COUNTERS.METRIKA_COUNTER)
                .from(METRIKA_COUNTERS)
                .where(METRIKA_COUNTERS.CID.in(campaignIds)
                        .and(METRIKA_COUNTERS.SOURCE.eq(MetrikaCountersSource.sprav)))
                .fetchGroups(METRIKA_COUNTERS.CID, METRIKA_COUNTERS.METRIKA_COUNTER);

        return EntryStream.of(counterIdsByCampaignId)
                .mapValues(Set::copyOf)
                .toMap();
    }

    private Map<Long, Long> getCalltrackingCountersByCampaignId(DSLContext dslContext,
                                                                List<Long> campaignIds) {
        return dslContext
                .select(CAMP_CALLTRACKING_SETTINGS.CID, CALLTRACKING_SETTINGS.COUNTER_ID)
                .from(CAMP_CALLTRACKING_SETTINGS)
                .join(CALLTRACKING_SETTINGS)
                .on(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID
                        .eq(CAMP_CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID))
                .where(CAMP_CALLTRACKING_SETTINGS.CID.in(campaignIds))
                .fetchMap(CAMP_CALLTRACKING_SETTINGS.CID, CALLTRACKING_SETTINGS.COUNTER_ID);
    }

    public List<CampaignInfoForMetrika> getCampaignsInfo(int shard, @Nullable LimitOffset limitOffset) {
        return getCampaignsInfo(shard, LocalDateTime.MIN, 0L, limitOffset);
    }

    public List<CampaignInfoForMetrika> getCampaignsInfo(int shard, LocalDateTime lastChange,
                                                         Long lastId, @Nullable LimitOffset limitOffset) {
        Field<String> nestedReps = DSL.field("reps", String.class);
        limitOffset = limitOffset != null ? limitOffset : limited(ObjectInfoConstants.DEFAULT_LIMIT);
        LocalDateTime gapTimestamp = LocalDateTime.now().minusSeconds(ObjectInfoConstants.GAP_SECONDS);
        Condition afterTime = CAMPAIGNS.LAST_CHANGE.greaterThan(lastChange);
        Condition equalTimeAndGreaterId = CAMPAIGNS.LAST_CHANGE.equal(lastChange)
                .and(CAMPAIGNS.CID.greaterThan(lastId));

        Result<Record> result = ppcDslContextProvider.ppc(shard)
                .select(asList(CAMPAIGNS.CID,
                        CAMPAIGNS.CLIENT_ID,
                        CAMPAIGNS.UID,
                        CAMPAIGNS.ORDER_ID,
                        CAMPAIGNS.NAME,
                        CAMPAIGNS.LAST_CHANGE,
                        groupConcat(USERS.UID).separator(",").as(nestedReps)))
                .from(CAMPAIGNS.join(USERS).on(USERS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID)))
                .where(CAMPAIGNS.ORDER_ID.greaterThan(EMPTY_ORDER_ID))
                .and(afterTime.or(equalTimeAndGreaterId))
                .and(CAMPAIGNS.LAST_CHANGE.lessThan(gapTimestamp))
                .groupBy(CAMPAIGNS.CID, CAMPAIGNS.CLIENT_ID, CAMPAIGNS.UID, CAMPAIGNS.ORDER_ID, CAMPAIGNS.NAME,
                        CAMPAIGNS.LAST_CHANGE)
                .orderBy(CAMPAIGNS.LAST_CHANGE, CAMPAIGNS.CID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch();

        return mapList(result, r -> new CampaignInfoForMetrika()
                .withId(r.getValue(CAMPAIGNS.CID))
                .withOrderId(r.getValue(CAMPAIGNS.ORDER_ID))
                .withClientId(r.getValue(CAMPAIGNS.CLIENT_ID))
                .withUid(r.getValue(CAMPAIGNS.UID))
                .withName(r.getValue(CAMPAIGNS.NAME))
                .withReps(stringRepsToList(r.getValue(nestedReps)))
                .withLastChange(r.getValue(CAMPAIGNS.LAST_CHANGE)));
    }

    /**
     * @param reps строка со списком uid'ов, разделенных запятыми
     * @return список uid'ов в виде списка Long'ов
     */
    private List<Long> stringRepsToList(String reps) {
        if (Strings.isNullOrEmpty(reps)) {
            return new ArrayList<>();
        }
        return StreamEx.of(reps.split(","))
                .map(Long::valueOf)
                .toList();
    }

    /**
     * Удаляет счетчики метрики турболендингов баннеров
     */
    public int deleteBannerTurbolandingCounters(DSLContext context, Collection<Long> bannerIds) {
        return context.deleteFrom(CAMP_TURBOLANDING_METRIKA_COUNTERS)
                .where(CAMP_TURBOLANDING_METRIKA_COUNTERS.BID.in(bannerIds))
                .execute();
    }

    /**
     * Уменьшает счетчики турболендингов для целей.
     */
    public UpdateConditionStep<CampMetrikaGoalsRecord> decreaseMetrikaGoalsLinksCountBatch(DSLContext context,
                                                                                           Map<Long, Long> linksByGoal, Long campaignId) {
        TableField<CampMetrikaGoalsRecord, Long> goalIdField = CAMP_METRIKA_GOALS.GOAL_ID;
        TableField<CampMetrikaGoalsRecord, Long> linksCountField = CAMP_METRIKA_GOALS.LINKS_COUNT;

        Map<Field<Long>, Field<Long>> caseMap = new HashMap<>();
        linksByGoal.forEach((goal, linksCount) -> caseMap.put(
                DSL.field("{0}", goalIdField.getDataType(), goal),
                // if (current > linksCount, current - linksCount, 0)
                // Первая проверка нужна потому что select для создания linksByGoal и следующий ниже update
                // разнесены, и может возникнуть race condition.
                // Если в процессе значение поля успеет стать меньше linksCount, то при попытке записать отрицательное
                // число в Bigint Unsigned мы упадем с 1000
                DSL.iif(linksCountField.ge(linksCount),
                        DSL.field("{0} - {1}", linksCountField.getDataType(), linksCountField, linksCount),
                        0L)));

        return context.update(CAMP_METRIKA_GOALS)
                .set(linksCountField, choose(goalIdField).mapFields(caseMap).otherwise(linksCountField))
                .where(CAMP_METRIKA_GOALS.CID.eq(campaignId));
    }

    /**
     * Возвращает список целей по клиенту и кампаниям
     *
     * @param shard       шард
     * @param clientId    id клиента
     * @param campaignIds список идентификаторов кампаний
     * @return список идентификаторов целей
     */
    public Set<Long> getGoalIds(int shard, long clientId, @Nullable Collection<Long> campaignIds) {
        SelectConditionStep<Record1<Long>> where = ppcDslContextProvider.ppc(shard)
                .select(CAMP_METRIKA_GOALS.GOAL_ID)
                .from(CAMPAIGNS)
                .join(CAMP_METRIKA_GOALS).on(CAMP_METRIKA_GOALS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId));

        if (campaignIds != null) {
            where.and(CAMPAIGNS.CID.in(campaignIds));
        }

        return where
                .fetchSet(record -> record.getValue(CAMP_METRIKA_GOALS.GOAL_ID));
    }

    /**
     * Возвращает мапу целей для клиенту по кампаниям
     *
     * @param shard       шард
     * @param clientId    id клиента
     * @param campaignIds список идентификаторов кампаний
     * @return список идентификаторов целей
     */
    public Map<Long, List<Long>> getGoalIdsByCampaignId(int shard, long clientId,
                                                        @Nullable Collection<Long> campaignIds) {
        SelectConditionStep<Record2<Long, Long>> where = ppcDslContextProvider.ppc(shard)
                .select(CAMP_METRIKA_GOALS.GOAL_ID, CAMP_METRIKA_GOALS.CID)
                .from(CAMPAIGNS)
                .join(CAMP_METRIKA_GOALS).on(CAMP_METRIKA_GOALS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId));

        if (campaignIds != null) {
            where.and(CAMPAIGNS.CID.in(campaignIds));
        }

        return where
                .fetchGroups(CAMP_METRIKA_GOALS.CID, CAMP_METRIKA_GOALS.GOAL_ID);
    }

    /**
     * Возвращает словарь используемых целей стратегии или, если их нет, ключевых целей по кампаниям.
     * Не возвращаются цель стратегии "По нескольким целям" и ключевая цель "Вовлеченные сессии"
     *
     * @param shard       шард
     * @param campaignIds список идентификаторов кампаний
     * @return список идентификаторов целей
     */
    public Map<Long, Set<Long>> getStrategyOrMeaningfulGoalIdsByCampaignId(int shard, Collection<Long> campaignIds) {
        Map<Long, GoalsByUsageType> goalIdsByCampaignIds = getUsedGoalsByCampaignIds(shard, null, campaignIds);
        return EntryStream.of(goalIdsByCampaignIds)
                .mapValues(goalsByUsageType ->
                        !goalsByUsageType.getStrategyGoalIds().isEmpty()
                                ? goalsByUsageType.getStrategyGoalIds()
                                : goalsByUsageType.getMeaningfulGoalIds()
                )
                .toMap();
    }

    /**
     * Возвращает цели, которые используются в настройках стратегий или как ключевые.
     * Не фильтрует кампании по статусу (а-ля Черновик/Архивная)
     */
    public Map<Long, GoalCampUsages> getGoalsUsedInCampaignsByClientId(int shard, ClientId clientId) {
        Map<Long, GoalsByUsageType> goalIdsByCampaignIds = getUsedGoalsByCampaignIds(shard, clientId.asLong(), null);
        Set<Long> meaningfulGoalIds =
                StreamEx.ofValues(goalIdsByCampaignIds)
                        .flatCollection(GoalsByUsageType::getMeaningfulGoalIds)
                        .toSet();
        Set<Long> goalsUsedInStrategies =
                StreamEx.ofValues(goalIdsByCampaignIds)
                        .flatCollection(GoalsByUsageType::getStrategyGoalIds)
                        .toSet();
        Map<Long, GoalCampUsages> result = new HashMap<>();

        for (Long goalId : Sets.union(meaningfulGoalIds, goalsUsedInStrategies)) {
            boolean isUsedInStrategy = goalsUsedInStrategies.contains(goalId);
            boolean isMeaningful = meaningfulGoalIds.contains(goalId);
            result.put(goalId, new GoalCampUsages(isMeaningful, isUsedInStrategy));
        }
        return result;
    }

    /**
     * Для кампаний, в которых используются цели, возвращает информацию о том, какие цели и как используются.
     *
     * @see GoalsByUsageType
     */
    private Map<Long, GoalsByUsageType> getUsedGoalsByCampaignIds(
            int shard,
            @Nullable Long clientId,
            @Nullable Collection<Long> campaignIds
    ) {
        checkArgument(clientId != null || campaignIds != null, "At least one filter should be specified");

        Condition whereCondition = DSL.trueCondition();
        if (campaignIds != null) {
            whereCondition = whereCondition.and(CAMPAIGNS.CID.in(campaignIds));
        }
        if (clientId != null) {
            whereCondition = whereCondition.and(CAMPAIGNS.CLIENT_ID.eq(clientId));
        }

        return ppcDslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID, CAMP_OPTIONS.MEANINGFUL_GOALS, CAMPAIGNS.STRATEGY_DATA)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(whereCondition)
                .fetchMap(CAMP_OPTIONS.CID, record -> {
                    var strategyGoalData = CampaignMappings.strategyDataFromDb(record.get(CAMPAIGNS.STRATEGY_DATA));
                    var strategyGoalId = ifNotNull(strategyGoalData, StrategyData::getGoalId);
                    Set<Long> strategyGoalIds;
                    if (strategyGoalId != null
                            && !Objects.equals(strategyGoalId, CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID)
                            && !Objects.equals(strategyGoalId, CampaignConstants.BY_ALL_GOALS_GOAL_ID)) {
                        strategyGoalIds = Set.of(strategyGoalId);
                    } else {
                        strategyGoalIds = Set.of();
                    }

                    var meaningfulGoals =
                            CampaignConverter.meaningfulGoalsFromDb(record.get(CAMP_OPTIONS.MEANINGFUL_GOALS));
                    Set<Long> meaningfulGoalIds = meaningfulGoals == null ? Set.of() : StreamEx.of(meaningfulGoals)
                            .nonNull()
                            .map(MeaningfulGoal::getGoalId)
                            .filter(id -> id != CampaignConstants.ENGAGED_SESSION_GOAL_ID)
                            .toSet();
                    return new GoalsByUsageType(strategyGoalIds, meaningfulGoalIds);
                });
    }

    /**
     * Возвращает словарь используемых целей клиента по кампаниям
     *
     * @param shard       шард
     * @param clientId    id клиента
     * @param campaignIds список идентификаторов кампаний
     * @return список идентификаторов целей
     */
    public Map<Long, Set<Long>> getUsedGoalIdsByCampaignId(int shard, long clientId, Collection<Long> campaignIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID, CAMP_OPTIONS.MEANINGFUL_GOALS, CAMP_OPTIONS.BROAD_MATCH_GOAL_ID,
                        CAMPAIGNS.STRATEGY_DATA)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId)
                        .and(CAMPAIGNS.CID.in(campaignIds)))
                .fetchMap(CAMP_OPTIONS.CID, new UsedCampaignGoalIdsRecordMapper());
    }

    private static class UsedCampaignGoalIdsRecordMapper implements RecordMapper<Record, Set<Long>> {
        @Override
        public Set<Long> map(Record record) {
            var strategyGoalId = CampaignMappings.strategyDataFromDb(record.get(CAMPAIGNS.STRATEGY_DATA)).getGoalId();
            var broadMatchGoalId = record.get(CAMP_OPTIONS.BROAD_MATCH_GOAL_ID);
            var meaningfulGoals = CampaignConverter.meaningfulGoalsFromDb(record.get(CAMP_OPTIONS.MEANINGFUL_GOALS));
            var resultGoalIds = meaningfulGoals == null ? new HashSet<Long>() : StreamEx.of(meaningfulGoals)
                    .nonNull()
                    .map(MeaningfulGoal::getGoalId)
                    .toSet();
            resultGoalIds.add(broadMatchGoalId);
            resultGoalIds.add(strategyGoalId);
            resultGoalIds.remove(null);
            return resultGoalIds;
        }
    }

    /**
     * @deprecated Таблица ppc.camp_turbolanding_metrika_counters не используется и будет удалена
     * https://st.yandex-team.ru/DIRECT-103395
     * Информацию по счетчикам турболендинга нужно получать из таблицы turbolanding_metrika_counters
     */
    @Deprecated(since = "2019-12-19")
    public Map<Long, Set<Long>> getTurbolandingCounters(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return Collections.emptyMap();
        }
        Map<Long, List<Long>> turbolandingCounters = ppcDslContextProvider.ppc(shard)
                .select(CAMP_TURBOLANDING_METRIKA_COUNTERS.CID, CAMP_TURBOLANDING_METRIKA_COUNTERS.METRIKA_COUNTER)
                .from(CAMP_TURBOLANDING_METRIKA_COUNTERS)
                .join(BANNER_TURBOLANDINGS).on(BANNER_TURBOLANDINGS.CID.eq(CAMP_TURBOLANDING_METRIKA_COUNTERS.CID))
                .join(TURBOLANDINGS).on(TURBOLANDINGS.TL_ID.eq(BANNER_TURBOLANDINGS.TL_ID))
                .where(CAMP_TURBOLANDING_METRIKA_COUNTERS.CID.in(campaignIds))
                .fetchGroups(CAMP_TURBOLANDING_METRIKA_COUNTERS.CID,
                        CAMP_TURBOLANDING_METRIKA_COUNTERS.METRIKA_COUNTER);
        return EntryStream.of(turbolandingCounters)
                .mapValues(counterIds -> StreamEx.of(counterIds)
                        .toSet())
                .toMap();
    }

    /**
     * Получить цели от технического счетчика турболендингов используемых в конкретной кампании кампании
     * Технический счетчик для клиента должен быть только один. Это следует проверить на уровне сервиса
     */
    public Map<Long, Set<Long>> getTurbolandingInternalCounterGoals(int shard, Collection<Long> campaignId,
                                                                    Set<TurbolandingGoalType> goalTypes) {
        Map<Long, List<Long>> turbolandingGoalsByCounter = ppcDslContextProvider.ppc(shard)
                .select(TURBOLANDING_METRIKA_COUNTERS.COUNTER, TURBOLANDING_METRIKA_GOALS.GOAL_ID)
                .from(BANNER_TURBOLANDINGS)
                .join(TURBOLANDING_METRIKA_COUNTERS).on(BANNER_TURBOLANDINGS.TL_ID.eq(TURBOLANDING_METRIKA_COUNTERS.TL_ID))
                .join(TURBOLANDING_METRIKA_GOALS).on(TURBOLANDING_METRIKA_COUNTERS.TL_ID.eq(TURBOLANDING_METRIKA_GOALS.TL_ID))
                .where(BANNER_TURBOLANDINGS.CID.in(campaignId)
                        .and(TURBOLANDING_METRIKA_COUNTERS.IS_USER_COUNTER.eq(0L))
                        .and(TURBOLANDING_METRIKA_GOALS.IS_CONVERSION_GOAL.in(mapList(goalTypes,
                                TurbolandingGoalType::getValue))))
                .fetchGroups(TURBOLANDING_METRIKA_COUNTERS.COUNTER, TURBOLANDING_METRIKA_GOALS.GOAL_ID);
        return EntryStream.of(turbolandingGoalsByCounter)
                .mapValues(goalIds -> listToSet(goalIds, Function.identity()))
                .toMap();
    }

    /**
     * Получить цели от технического счетчика турболендингов опубликованных клиентом
     * Технический счетчик для клиента может быть только один. Это следует проверить на уровне сервиса
     */
    public Map<Long, Set<Long>> getTurbolandingInternalCounterGoals(int shard, ClientId clientId,
                                                                    Set<TurbolandingGoalType> goalTypes) {
        Map<Long, List<Long>> turbolandingGoalsByCounter = ppcDslContextProvider.ppc(shard)
                .select(TURBOLANDING_METRIKA_COUNTERS.COUNTER, TURBOLANDING_METRIKA_GOALS.GOAL_ID)
                .from(TURBOLANDINGS)
                .join(TURBOLANDING_METRIKA_COUNTERS).on(TURBOLANDINGS.TL_ID.eq(TURBOLANDING_METRIKA_COUNTERS.TL_ID))
                .join(TURBOLANDING_METRIKA_GOALS).on(TURBOLANDING_METRIKA_COUNTERS.TL_ID.eq(TURBOLANDING_METRIKA_GOALS.TL_ID))
                .where(TURBOLANDINGS.CLIENT_ID.in(clientId.asLong())
                        .and(TURBOLANDING_METRIKA_COUNTERS.IS_USER_COUNTER.eq(0L))
                        .and(TURBOLANDING_METRIKA_GOALS.IS_CONVERSION_GOAL.in(
                                mapList(goalTypes, TurbolandingGoalType::getValue))))
                .fetchGroups(TURBOLANDING_METRIKA_COUNTERS.COUNTER, TURBOLANDING_METRIKA_GOALS.GOAL_ID);
        return EntryStream.of(turbolandingGoalsByCounter)
                .mapValues(goalIds -> listToSet(goalIds, Function.identity()))
                .toMap();
    }

    public void addGoalIds(int shard, long campaignId, Set<Long> goalIds) {
        if (goalIds.isEmpty()) {
            return;
        }

        addGoalIds(shard, Map.of(campaignId, goalIds));
    }

    public void addGoalIds(int shard, Map<Long, Set<Long>> goalIdsByCampaignId) {
        if (goalIdsByCampaignId.isEmpty()) {
            return;
        }

        InsertHelper<CampMetrikaGoalsRecord> insertHelper =
                new InsertHelper<>(ppcDslContextProvider.ppc(shard), CAMP_METRIKA_GOALS);

        EntryStream.of(goalIdsByCampaignId)
                .flatMapValues(Collection::stream)
                .forKeyValue((campaignId, goalId) -> insertHelper
                        .set(CAMP_METRIKA_GOALS.CID, campaignId)
                        .set(CAMP_METRIKA_GOALS.GOAL_ID, goalId)
                        .newRecord());

        int rowsInserted = insertHelper
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();

        logger.debug("Rows inserted in addGoalIds: {}", rowsInserted);
    }

    /**
     * Получает список целей метрики на кампании со статистикой по их первичным ключам - парам (cid, goal_id)
     */
    public List<CampMetrikaGoal> getCampMetrikaGoalsByIds(int shard, ClientId clientId,
                                                          List<CampMetrikaGoalId> goalIds) {
        if (goalIds.isEmpty()) {
            return List.of();
        }

        var idCondition = StreamEx.of(goalIds)
                .map(id -> CAMP_METRIKA_GOALS.CID.eq(id.getCampaignId())
                        .and(CAMP_METRIKA_GOALS.GOAL_ID.eq(id.getGoalId())))
                .reduce(Condition::or)
                .orElseThrow();

        var goals = ppcDslContextProvider.ppc(shard)
                .select(campMetrikaGoalMapper.getFieldsToRead())
                .from(CAMP_METRIKA_GOALS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(CAMP_METRIKA_GOALS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(idCondition)
                .fetch(campMetrikaGoalMapper::fromDb);

        return StreamEx.of(goals)
                .map(goal -> goal.withId(new CampMetrikaGoalId()
                        .withCampaignId(goal.getCampaignId())
                        .withGoalId(goal.getGoalId())))
                .toList();
    }

    /**
     * Добавляет в базу новые записи camp_metrika_goals. Поле id (которое объект - составной ключ) не учитывается,
     * вставляются значения из отдельных полей campaignId и goalId.
     */
    public void addCampMetrikaGoals(int shard, List<CampMetrikaGoal> goals) {
        if (goals.isEmpty()) {
            return;
        }

        InsertHelper<CampMetrikaGoalsRecord> insertHelper =
                new InsertHelper<>(ppcDslContextProvider.ppc(shard), CAMP_METRIKA_GOALS);
        insertHelper
                .addAll(campMetrikaGoalMapper, goals)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }

    public Set<Long> getCampaignIdsWithCombinedGoals(int shard, Collection<Long> campaignIds) {
        return getCampaignIdsWithCombinedGoals(ppcDslContextProvider.ppc(shard), campaignIds);
    }

    public Set<Long> getCampaignIdsWithCombinedGoals(DSLContext context, Collection<Long> campaignIds) {
        return context
                .selectDistinct(CAMP_METRIKA_GOALS.CID)
                .from(CAMP_METRIKA_GOALS)
                .where(CAMP_METRIKA_GOALS.CID.in(campaignIds))
                .and(CAMP_METRIKA_GOALS.LINKS_COUNT.gt(0L))
                .and(SqlUtils.findInSet(GoalRole.COMBINED.getTypedValue(), CAMP_METRIKA_GOALS.GOAL_ROLE).gt(0L))
                .fetchSet(CAMP_METRIKA_GOALS.CID);
    }

    public Set<CampMetrikaGoalId> getCampMetrikaGoalIdsByCampaignIds(int shard, ClientId clientId,
                                                                     Collection<Long> campaignIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(CAMP_METRIKA_GOALS.GOAL_ID, CAMP_METRIKA_GOALS.CID)
                .from(CAMPAIGNS)
                .join(CAMP_METRIKA_GOALS).on(CAMP_METRIKA_GOALS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(CAMP_METRIKA_GOALS.CID.in(campaignIds))
                .fetchSet(campMetrikaGoalIdMapper::fromDb);
    }

    private static JooqMapperWithSupplierBuilder<CampaignForMetrika> commonMapper() {
        return JooqMapperWithSupplierBuilder.builder(CampaignForMetrika::new)
                .map(property(CampaignForMetrika.CLIENT_ID, CAMPAIGNS.CLIENT_ID))
                .map(property(CampaignForMetrika.ID, CAMPAIGNS.CID))
                .map(property(CampaignForMetrika.ORDER_ID, CAMPAIGNS.ORDER_ID))
                .map(property(CampaignForMetrika.NAME, CAMPAIGNS.NAME))
                .map(convertibleProperty(CampaignForMetrika.CURRENCY, CAMPAIGNS.CURRENCY,
                        CampaignMappings::currencyCodeFromDb, CampaignMappings::currencyCodeToDb))
                .map(convertibleProperty(
                        CampaignForMetrika.STATUS_CLICK_TRACK, CAMP_OPTIONS.STATUS_CLICK_TRACK,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::booleanToLong));
    }

    private static JooqMapperWithSupplier<CampaignForMetrika> mapperWithTurbolandingMetrikaCounter() {
        return commonMapper()
                .readProperty(CampaignForMetrika.METRIKA_COUNTERS,
                        fromField(TURBOLANDING_METRIKA_COUNTERS.COUNTER).by(Arrays::asList))
                .build();
    }

    private static JooqMapperWithSupplier<CampaignForMetrika> mapperWithCampaignMetrikaCounters() {
        return commonMapper()
                .map(convertibleProperty(
                        CampaignForMetrika.METRIKA_COUNTERS, CAMP_METRIKA_COUNTERS.METRIKA_COUNTERS,
                        CampaignMappings::metrikaCountersFromDb, CampaignMappings::metrikaCountersToDb)
                )
                .build();
    }

    /**
     * Используется в тестах
     */
    public static JooqMapperWithSupplier<CampMetrikaGoal> createCampMetrikaGoalsMapper() {
        return JooqMapperWithSupplierBuilder.builder(CampMetrikaGoal::new)
                .map(property(CAMPAIGN_ID, CAMP_METRIKA_GOALS.CID))
                .map(property(GOAL_ID, CAMP_METRIKA_GOALS.GOAL_ID))
                .map(property(GOALS_COUNT, CAMP_METRIKA_GOALS.GOALS_COUNT))
                .map(property(CONTEXT_GOALS_COUNT, CAMP_METRIKA_GOALS.CONTEXT_GOALS_COUNT))
                .map(property(STAT_DATE, CAMP_METRIKA_GOALS.STAT_DATE))
                .map(convertibleEnumSet(GOAL_ROLE, CAMP_METRIKA_GOALS.GOAL_ROLE,
                        GoalRole::fromTypedValue,
                        GoalRole::getTypedValue))
                .map(property(LINKS_COUNT, CAMP_METRIKA_GOALS.LINKS_COUNT))
                .build();
    }

}
