package ru.yandex.direct.core.entity.metrika.repository;

import java.util.List;

import org.jooq.Field;
import org.jooq.tools.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.metrika.model.MetrikaCounterByDomain;
import ru.yandex.direct.grid.schema.yt.Tables;
import ru.yandex.direct.grid.schema.yt.tables.AllCountersByDomain;
import ru.yandex.direct.grid.schema.yt.tables.CounterByDomain;
import ru.yandex.direct.grid.schema.yt.tables.CountersByDomainHitLog;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.ytcomponents.service.AllCountersByDomainDynContextProvider;
import ru.yandex.direct.ytcomponents.service.CounterByDomainDynContextProvider;
import ru.yandex.direct.ytcomponents.service.CountersByDomainHitLogDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.YtTableUtils.longValueGetter;

@Repository
public class MetrikaCounterByDomainRepository {

    private static final CounterByDomain COUNTER_BY_DOMAIN = Tables.COUNTER_BY_DOMAIN.as("C");
    private static final AllCountersByDomain ALL_COUNTERS_BY_DOMAIN = Tables.ALL_COUNTERS_BY_DOMAIN.as("C");
    private static final CountersByDomainHitLog COUNTERS_BY_DOMAIN_HIT_LOG = Tables.COUNTERS_BY_DOMAIN_HIT_LOG.as("C");

    private static final Field<Long> COUNTER_ID = aliased(COUNTER_BY_DOMAIN.COUNTER_ID);
    private static final Field<Long> OWNER_ID = aliased(COUNTER_BY_DOMAIN.OWNER_UID);
    private static final Field<Long> TIMESTAMP = aliased(COUNTER_BY_DOMAIN.TIMESTAMP);

    private final CounterByDomainDynContextProvider counterByDomainDynContextProvider;
    private final AllCountersByDomainDynContextProvider allCountersByDomainDynContextProvider;
    private final CountersByDomainHitLogDynContextProvider countersByDomainHitLogDynContextProvider;

    @Autowired
    public MetrikaCounterByDomainRepository(
            CounterByDomainDynContextProvider counterByDomainDynContextProvider,
            AllCountersByDomainDynContextProvider allCountersByDomainDynContextProvider,
            CountersByDomainHitLogDynContextProvider countersByDomainHitLogDynContextProvider) {
        this.counterByDomainDynContextProvider = counterByDomainDynContextProvider;
        this.allCountersByDomainDynContextProvider = allCountersByDomainDynContextProvider;
        this.countersByDomainHitLogDynContextProvider = countersByDomainHitLogDynContextProvider;
    }

    /**
     * Получить счетчики по домену, удовлетворяющие ограничениям:
     * <li> людей на счетчике должно быть хотя бы 10% от людей на домене;</li>
     * <li> хотя бы 10 уникальных людей на домене</li>
     */
    public List<MetrikaCounterByDomain> getRestrictedCountersByDomain(String domain, boolean useHitLogTable) {
        if (StringUtils.isEmpty(domain)) {
            return emptyList();
        }

        UnversionedRowset rowset;
        if (useHitLogTable) {
            try (TraceProfile ignored =
                         Trace.current().profile("counterbydomain_hitlog:getCountersByDomain", "yt", 1)) {
                var query = YtDSL.ytContext()
                        .select(COUNTER_ID, OWNER_ID, TIMESTAMP)
                        .from(COUNTERS_BY_DOMAIN_HIT_LOG)
                        .where(COUNTERS_BY_DOMAIN_HIT_LOG.DOMAIN.eq(domain)
                                .and(COUNTERS_BY_DOMAIN_HIT_LOG.NUM_DOMAIN_USERS.ge(10L))
                                .and(COUNTERS_BY_DOMAIN_HIT_LOG.NUM_DOMAIN_USERS.le(
                                        COUNTERS_BY_DOMAIN_HIT_LOG.NUM_COUNTER_USERS.multiply(10))));
                rowset = countersByDomainHitLogDynContextProvider.getContext().executeTimeoutSafeSelect(query);
            }
        } else {
            try (TraceProfile ignored = Trace.current().profile("counterbydomain:getCountersByDomain", "yt", 1)) {
                var query = YtDSL.ytContext()
                        .select(COUNTER_ID, OWNER_ID, TIMESTAMP)
                        .from(COUNTER_BY_DOMAIN)
                        .where(COUNTER_BY_DOMAIN.DOMAIN.eq(domain));
                rowset = counterByDomainDynContextProvider.getContext().executeTimeoutSafeSelect(query);
            }
        }
        TableSchema tableSchema = rowset.getSchema();
        return mapList(rowset.getRows(), row -> convertToModel(row, tableSchema));
    }

    /**
     * Получить счетчики по домену с хотя бы одним уникальным человеком на домене
     */
    public List<MetrikaCounterByDomain> getAllCountersByDomain(String domain, boolean useHitLogTable) {
        if (StringUtils.isEmpty(domain)) {
            return emptyList();
        }

        UnversionedRowset rowset;
        if (useHitLogTable) {
            try (TraceProfile ignored =
                         Trace.current().profile("counterbydomain_hitlog:getAllCountersByDomain", "yt", 1)) {
                var query = YtDSL.ytContext()
                        .select(COUNTER_ID, OWNER_ID, TIMESTAMP)
                        .from(COUNTERS_BY_DOMAIN_HIT_LOG)
                        .where(COUNTERS_BY_DOMAIN_HIT_LOG.DOMAIN.eq(domain)
                                .and(COUNTERS_BY_DOMAIN_HIT_LOG.NUM_DOMAIN_USERS.ge(1L)));
                rowset = countersByDomainHitLogDynContextProvider.getContext().executeTimeoutSafeSelect(query);
            }
        } else {
            try (TraceProfile ignored = Trace.current().profile("counterbydomain:getAllCountersByDomain", "yt", 1)) {
                var query = YtDSL.ytContext()
                        .select(COUNTER_ID, OWNER_ID, TIMESTAMP)
                        .from(ALL_COUNTERS_BY_DOMAIN)
                        .where(ALL_COUNTERS_BY_DOMAIN.DOMAIN.eq(domain));
                rowset = allCountersByDomainDynContextProvider.getContext().executeTimeoutSafeSelect(query);
            }
        }
        TableSchema tableSchema = rowset.getSchema();
        return mapList(rowset.getRows(), row -> convertToModel(row, tableSchema));
    }

    private static MetrikaCounterByDomain convertToModel(UnversionedRow row, TableSchema tableSchema) {
        return new MetrikaCounterByDomain()
                .withCounterId(longValueGetter(tableSchema, COUNTER_ID).apply(row))
                .withOwnerUid(longValueGetter(tableSchema, OWNER_ID).apply(row))
                .withTimestamp(longValueGetter(tableSchema, TIMESTAMP).apply(row))
                .withSource(MetrikaCounterSource.UNKNOWN);
    }
}
