package ru.yandex.direct.core.entity.metrika.repository;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.Record;
import org.jooq.Result;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.metrika.model.objectinfo.CreativeInfoForMetrika;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Arrays.asList;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class MetrikaCreativeRepository {

    public static final int ADVANCE_MINUTES = 10;

    private final DslContextProvider dslContextProvider;

    @Autowired
    public MetrikaCreativeRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public List<CreativeInfoForMetrika> getCreativesInfo(int shard, @Nullable LimitOffset limitOffset) {
        return getCreativesInfo(shard, LocalDateTime.MIN,
                0L, limitOffset);

    }


    public List<CreativeInfoForMetrika> getCreativesInfo(int shard, LocalDateTime lastChange,
                                                         Long lastCreativeId, @Nullable LimitOffset limitOffset) {
        limitOffset = limitOffset != null ? limitOffset : limited(ObjectInfoConstants.DEFAULT_LIMIT);

        LocalDateTime gapTimestamp = LocalDateTime
                .now()
                .plusMinutes(ADVANCE_MINUTES)
                .minusSeconds(ObjectInfoConstants.GAP_SECONDS);

        Condition afterTime = PERF_CREATIVES.MODERATE_SEND_TIME.greaterThan(lastChange);
        Condition equalTimeAndGreaterId = PERF_CREATIVES.MODERATE_SEND_TIME.equal(lastChange)
                .and(PERF_CREATIVES.CREATIVE_ID.greaterThan(lastCreativeId));

        Result<Record> result = dslContextProvider.ppc(shard)
                .select(asList(PERF_CREATIVES.CREATIVE_ID,
                        PERF_CREATIVES.NAME,
                        PERF_CREATIVES.GROUP_NAME,
                        PERF_CREATIVES.MODERATE_SEND_TIME,
                        CAMPAIGNS.ORDER_ID))
                .from(PERF_CREATIVES
                        .join(BANNERS_PERFORMANCE).on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                        .join(PHRASES).on(PHRASES.PID.eq(BANNERS_PERFORMANCE.PID))
                        .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID)))
                .where(afterTime.or(equalTimeAndGreaterId))
                .and(PERF_CREATIVES.MODERATE_SEND_TIME.lessThan(gapTimestamp))
                .orderBy(PERF_CREATIVES.MODERATE_SEND_TIME, PERF_CREATIVES.CREATIVE_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch();

        return mapList(result, record -> new CreativeInfoForMetrika()
                .withId(record.getValue(PERF_CREATIVES.CREATIVE_ID))
                .withOrderId(record.getValue(CAMPAIGNS.ORDER_ID))
                .withName(record.getValue(PERF_CREATIVES.NAME))
                .withGroupName(record.getValue(PERF_CREATIVES.GROUP_NAME))
                .withModerateSendTime(record.getValue(PERF_CREATIVES.MODERATE_SEND_TIME)));
    }


}
