package ru.yandex.direct.core.entity.metrika.repository;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.Record;
import org.jooq.SelectJoinStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.metrika.model.DynamicForMetrika;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static ru.yandex.direct.dbschema.ppc.Tables.DYNAMIC_CONDITIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CommonUtils.mapByKey;

@Repository
@ParametersAreNonnullByDefault
public class MetrikaDynamicRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<DynamicForMetrika> jooqMapper;

    @Autowired
    public MetrikaDynamicRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        jooqMapper = JooqMapperWithSupplierBuilder.builder(DynamicForMetrika::new)
                .map(property(DynamicForMetrika.CONDITION_ID, DYNAMIC_CONDITIONS.DYN_COND_ID))
                .map(property(DynamicForMetrika.CONDITION_NAME, DYNAMIC_CONDITIONS.CONDITION_NAME))
                .build();
    }

    /**
     * Получение мапы условий из dynamic_conditions по order_id:dyn_cond_id. Ключ мапы - dyn_cond_id
     *
     * @param keys список пар: ключ order_id значение dyn_cond_id.
     *             Не может быть null
     */
    public Map<Long, DynamicForMetrika> getConditions(int shard, List<Pair<Long, Long>> keys) {
        if (keys.isEmpty()) {
            return Collections.emptyMap();
        }

        SelectJoinStep<Record> selectJoinStep = dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(DYNAMIC_CONDITIONS)
                .join(PHRASES).on(PHRASES.PID.eq(DYNAMIC_CONDITIONS.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID));

        Condition condition = StreamEx.of(keys)
                .map(key -> CAMPAIGNS.ORDER_ID.eq(key.getKey())
                        .and(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(key.getValue())))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());

        return StreamEx.of(selectJoinStep
                .where(condition)
                .fetch())
                .map(jooqMapper::fromDb)
                .collect(mapByKey(DynamicForMetrika::getConditionId));
    }
}
