package ru.yandex.direct.core.entity.metrika.repository;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.Record;
import org.jooq.Record2;
import org.jooq.Result;
import org.jooq.SelectJoinStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.metrika.model.PerformanceFilterForMetrika;
import ru.yandex.direct.core.entity.metrika.model.objectinfo.PerformanceFilterInfoForMetrika;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Arrays.asList;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.utils.CommonUtils.mapByKey;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class MetrikaPerformanceFilterRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<PerformanceFilterForMetrika> jooqMapper;

    @Autowired
    public MetrikaPerformanceFilterRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        jooqMapper = JooqMapperWithSupplierBuilder.builder(PerformanceFilterForMetrika::new)
                .map(property(PerformanceFilterForMetrika.FILTER_ID, BIDS_PERFORMANCE.PERF_FILTER_ID))
                .map(property(PerformanceFilterForMetrika.FILTER_NAME, BIDS_PERFORMANCE.NAME))
                .build();
    }

    /**
     * Получение мапы фильтров из bids_performance по order_id:perf_filter_id. Ключ мапы - perf_filter_id
     *
     * @param keys список пар: ключ order_id значение perf_filter_id.
     *             Не может быть null или пустым
     */
    public Map<Long, PerformanceFilterForMetrika> getFilters(int shard, List<Pair<Long, Long>> keys) {
        if (keys.isEmpty()) {
            return Collections.emptyMap();
        }
        SelectJoinStep<Record2<Long, String>> selectJoinStep = dslContextProvider.ppc(shard)
                .select(BIDS_PERFORMANCE.PERF_FILTER_ID, BIDS_PERFORMANCE.NAME)
                .from(BIDS_PERFORMANCE)
                .join(PHRASES).on(PHRASES.PID.eq(BIDS_PERFORMANCE.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID));

        Condition condition = StreamEx.of(keys)
                .map(key -> CAMPAIGNS.ORDER_ID.eq(key.getKey())
                        .and(BIDS_PERFORMANCE.PERF_FILTER_ID.eq(key.getValue())))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());

        return StreamEx.of(selectJoinStep
                .where(condition)
                .fetch())
                .map(jooqMapper::fromDb)
                .collect(mapByKey(PerformanceFilterForMetrika::getFilterId));
    }

    public List<PerformanceFilterInfoForMetrika> getFiltersInfo(int shard, @Nullable LimitOffset limitOffset) {
        return getFiltersInfo(shard, LocalDateTime.MIN, 0L, limitOffset);
    }

    public List<PerformanceFilterInfoForMetrika> getFiltersInfo(int shard, LocalDateTime lastChange,
                                                                Long lastFilterId, @Nullable LimitOffset limitOffset) {
        limitOffset = limitOffset != null ? limitOffset : limited(ObjectInfoConstants.DEFAULT_LIMIT);
        LocalDateTime gapTimestamp = LocalDateTime.now().minusSeconds(ObjectInfoConstants.GAP_SECONDS);
        Condition afterTime = BIDS_PERFORMANCE.LAST_CHANGE.greaterThan(lastChange);
        Condition equalTimeAndGreaterId = BIDS_PERFORMANCE.LAST_CHANGE.equal(lastChange)
                .and(BIDS_PERFORMANCE.PERF_FILTER_ID.greaterThan(lastFilterId));

        Result<Record> result = dslContextProvider.ppc(shard)
                .select(asList(BIDS_PERFORMANCE.PERF_FILTER_ID,
                        BIDS_PERFORMANCE.NAME,
                        BIDS_PERFORMANCE.LAST_CHANGE,
                        CAMPAIGNS.ORDER_ID))
                .from(BIDS_PERFORMANCE
                        .join(PHRASES).on(PHRASES.PID.eq(BIDS_PERFORMANCE.PID))
                        .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID)))
                .where(afterTime.or(equalTimeAndGreaterId))
                .and(BIDS_PERFORMANCE.LAST_CHANGE.lessThan(gapTimestamp))
                .and(PHRASES.PRIORITY_ID.greaterThan(0L))
                .orderBy(BIDS_PERFORMANCE.LAST_CHANGE, BIDS_PERFORMANCE.PERF_FILTER_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch();

        return mapList(result, record -> new PerformanceFilterInfoForMetrika()
                .withId(record.getValue(BIDS_PERFORMANCE.PERF_FILTER_ID))
                .withOrderId(record.getValue(CAMPAIGNS.ORDER_ID))
                .withName(record.getValue(BIDS_PERFORMANCE.NAME))
                .withLastChange(record.getValue(BIDS_PERFORMANCE.LAST_CHANGE)));
    }
}
