package ru.yandex.direct.core.entity.metrika.repository;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.Record;
import org.jooq.Result;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.metrika.model.objectinfo.RetargetingConditionInfoForMetrika;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Arrays.asList;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_CONDITIONS;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class MetrikaRetargetingConditionRepository {

    private static final int DEFAULT_LIMIT = 100_000;

    @Autowired
    private DslContextProvider dslContextProvider;

    public List<RetargetingConditionInfoForMetrika> getRetargetingConditionsInfo(int shard,
                                                                                 @Nullable LimitOffset limitOffset) {
        return getRetargetingConditionsInfo(shard, LocalDateTime.MIN, 0L, limitOffset);
    }

    public List<RetargetingConditionInfoForMetrika> getRetargetingConditionsInfo(int shard, LocalDateTime lastChange,
                                                                                 long lastId, @Nullable LimitOffset limitOffset) {
        limitOffset = limitOffset != null ? limitOffset : limited(DEFAULT_LIMIT);

        Condition afterTime = RETARGETING_CONDITIONS.MODTIME.greaterThan(lastChange);
        Condition equalTimeAndGreaterId =
                RETARGETING_CONDITIONS.MODTIME.equal(lastChange).and(
                        RETARGETING_CONDITIONS.RET_COND_ID.greaterThan(lastId));

        Result<Record> result = dslContextProvider.ppc(shard)
                .select(asList(RETARGETING_CONDITIONS.RET_COND_ID,
                        RETARGETING_CONDITIONS.CONDITION_NAME,
                        RETARGETING_CONDITIONS.CONDITION_DESC,
                        RETARGETING_CONDITIONS.MODTIME))
                .from(RETARGETING_CONDITIONS)
                .where(afterTime.or(equalTimeAndGreaterId))
                .orderBy(RETARGETING_CONDITIONS.MODTIME, RETARGETING_CONDITIONS.RET_COND_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch();

        return mapList(result, r -> new RetargetingConditionInfoForMetrika()
                .withId(r.getValue(RETARGETING_CONDITIONS.RET_COND_ID))
                .withName(r.getValue(RETARGETING_CONDITIONS.CONDITION_NAME))
                .withDescription(r.getValue(RETARGETING_CONDITIONS.CONDITION_DESC))
                .withLastChange(r.getValue(RETARGETING_CONDITIONS.MODTIME)));
    }

}
