package ru.yandex.direct.core.entity.metrika.service;

import java.time.Duration;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.common.lettuce.LettuceExecuteException;
import ru.yandex.direct.core.entity.metrika.model.GoalsConversionVisitsCountCacheRecord;
import ru.yandex.direct.utils.HashingUtils;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE;
import static ru.yandex.direct.utils.FunctionalUtils.setUnion;

@Service
@ParametersAreNonnullByDefault
public class MetrikaGoalsConversionCacheService {
    private static final Logger logger = LoggerFactory.getLogger(MetrikaGoalsConversionCacheService.class);

    private final LettuceConnectionProvider lettuceCache;

    public static final Duration EXPIRATION_TIME = Duration.ofHours(1);

    public MetrikaGoalsConversionCacheService(@Qualifier(LETTUCE) LettuceConnectionProvider lettuceCache) {
        this.lettuceCache = lettuceCache;
    }

    private static String generateKey(Long clientId, Set<Long> counterIds) {
        Set<Long> keySet = setUnion(Set.of(clientId), counterIds);
        String md5Hash = HashingUtils.getMd5HashUtf8AsHexString(StreamEx.of(keySet).sorted().joining("_"));
        return String.format("goals-metrika-conversions-cache-%s", md5Hash);
    }

    public GoalsConversionVisitsCountCacheRecord getFromCache(Long clientId, Set<Long> counterIds) {
        String key = generateKey(clientId, counterIds);
        String jsonValue = redisGet(key);

        return Optional.ofNullable(jsonValue)
                .map(v -> JsonUtils.fromJson(v, GoalsConversionVisitsCountCacheRecord.class))
                .orElse(null);
    }

    public void saveToCache(GoalsConversionVisitsCountCacheRecord cacheRecord) {
        String key = generateKey(cacheRecord.getClientId(), cacheRecord.getCounterIds());

        String jsonValue = JsonUtils.toJson(cacheRecord);
        redisSetex(key, jsonValue);
    }

    @Nullable
    private String redisGet(String key) {
        try {
            return lettuceCache.call("redis:get", cmd -> cmd.get(key));
        } catch (LettuceExecuteException e) {
            logger.warn("error in redisGet", e);
            return null;
        }
    }

    private void redisSetex(String key, String value) {
        try {
            lettuceCache.call("redis:setex", cmd -> cmd.setex(key, EXPIRATION_TIME.getSeconds(), value));
        } catch (LettuceExecuteException e) {
            logger.warn("error in redisSetex", e);
        }
    }

}
