package ru.yandex.direct.core.entity.metrika.service;

import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.model.response.GoalConversionInfo;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.feature.FeatureName.ENABLE_GOAL_CONVERSION_STATISTICS_FOR_7_DAYS;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
@ParametersAreNonnullByDefault
public class MetrikaGoalsConversionService {
    public static final int WRONG_DAYS_WITH_CONVERSION_VISITS = 14; //todo(pashkus) delete after feature setting
    public static final int DAYS_WITH_CONVERSION_VISITS = 7;

    private final MetrikaClient metrikaClient;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final FeatureService featureService;

    @Autowired
    public MetrikaGoalsConversionService(MetrikaClient metrikaClient,
                                         CampMetrikaCountersService campMetrikaCountersService,
                                         FeatureService featureService) {
        this.metrikaClient = metrikaClient;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.featureService = featureService;
    }

    /**
     * Получение информации о конверсиях и наличии дохода у целей по переданным счетчикам за прошедшие две недели
     *
     * @param clientId   - id клиента
     * @param counterIds - набор счетчиков клиента, для которого требуется статистика
     */
    public Map<Long, GoalConversionInfo> getGoalsConversion(ClientId clientId, Set<Long> counterIds) {
        Set<Long> counterIdsToRequest = computeCounterIdsToGoalsConversion(clientId, counterIds);

        if (counterIdsToRequest.isEmpty()) {
            return emptyMap();
        }
        var statisticsPeriod = featureService.isEnabledForClientId(clientId,
                ENABLE_GOAL_CONVERSION_STATISTICS_FOR_7_DAYS)
                ? DAYS_WITH_CONVERSION_VISITS
                : WRONG_DAYS_WITH_CONVERSION_VISITS;

        return metrikaClient.getGoalsConversionInfoByCounterIds(mapSet(counterIdsToRequest, Long::intValue),
                statisticsPeriod);
    }

    private Set<Long> computeCounterIdsToGoalsConversion(ClientId clientId, Set<Long> counterIds) {
        if (counterIds.isEmpty()) {
            return campMetrikaCountersService.getAvailableCounterIdsForGoals(clientId);
        }
        Set<Long> availableCounterIds =
                campMetrikaCountersService.getAvailableAndFilterInputCounterIdsInMetrikaForGoals(clientId, counterIds);
        return filterToSet(counterIds, availableCounterIds::contains);
    }

    /**
     * Получить конверсии по целям для заданных счетчиков без проверки прав на счетчики и цели.
     */
    public Map<Long, GoalConversionInfo> getGoalsConversion(Set<Long> counterIds) {
        if (counterIds.isEmpty()) {
            return emptyMap();
        }

        return metrikaClient.getGoalsConversionInfoByCounterIds(mapSet(counterIds, Long::intValue),
                DAYS_WITH_CONVERSION_VISITS);
    }

    public Map<Long, GoalConversionInfo> getGoalsConversion(ClientId clientId, Set<Long> counterIds,
                                                            boolean withUnavailableGoals) {
        return withUnavailableGoals ? getGoalsConversion(counterIds) : getGoalsConversion(clientId, counterIds);
    }
}
