package ru.yandex.direct.core.entity.metrika.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.container.CreateCounterGoalContainer;
import ru.yandex.direct.core.entity.metrika.model.GoalCampUsages;
import ru.yandex.direct.core.entity.metrika.repository.LalSegmentRepository;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository;
import ru.yandex.direct.core.entity.metrika.utils.GoalsForSuggestionComparator;
import ru.yandex.direct.core.entity.metrika.utils.MetrikaGoalsUtils;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformation;
import ru.yandex.direct.core.entity.retargeting.converter.GoalConverter;
import ru.yandex.direct.core.entity.retargeting.model.ConversionLevel;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalStatus;
import ru.yandex.direct.core.entity.retargeting.model.GoalsSuggestion;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingGoalsPpcDictRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.metrika.client.model.request.GoalType;
import ru.yandex.direct.metrika.client.model.response.CounterGoal;
import ru.yandex.direct.metrika.client.model.response.CreateCounterGoal;
import ru.yandex.direct.metrika.client.model.response.RetargetingCondition;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.metrika.utils.MetrikaGoalsUtils.countersHasGoal;
import static ru.yandex.direct.core.entity.metrika.utils.MetrikaGoalsUtils.getGoalsByCampaignIds;
import static ru.yandex.direct.core.entity.retargeting.model.Goal.MOBILE_GOAL_IDS;
import static ru.yandex.direct.core.security.tvm.TvmUtils.retrieveUserTicket;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
public class MetrikaGoalsService {
    private static final Logger logger = LoggerFactory.getLogger(MetrikaGoalsService.class);

    private static final Comparator<Goal> GOALS_FOR_SUGGESTION_COMPARATOR = new GoalsForSuggestionComparator();

    @Autowired
    private CampaignTypedRepository campaignTypedRepository;
    @Autowired
    private RbacService rbacService;
    @Autowired
    private MetrikaClient metrikaClient;
    @Autowired
    private ShardHelper shardHelper;
    @Autowired
    private MetrikaCampaignRepository metrikaCampaignRepository;
    @Autowired
    private RetargetingGoalsPpcDictRepository retargetingGoalsPpcDictRepository;
    @Autowired
    private CampMetrikaCountersService campMetrikaCountersService;
    @Autowired
    private LalSegmentRepository lalSegmentRepository;
    @Autowired
    private FeatureService featureService;
    @Autowired
    private MobileGoalsService mobileGoalsService;
    @Autowired
    private CampaignRepository campaignRepository;

    /**
     * Возвращает список целей по клиенту и кампаниям для оператора
     *
     * @param operatorUid идентификатор оператора
     * @param clientId    id клиента
     * @param campaignIds список идентификаторов кампаний
     * @return список целей
     */
    public List<Goal> getGoals(Long operatorUid, ClientId clientId, @Nullable Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        var campaignIdToCampaignType = ifNotNull(campaignIds,
                ids -> campaignRepository.getCampaignsTypeMap(shard, ids));
        Map<Long, List<Goal>> goalsByCampaignId = getGoalsByCampaignId(shard, operatorUid, clientId,
                campaignIdToCampaignType, false);
        List<Goal> resultGoals = goalsByCampaignId.values()
                .stream()
                .flatMap(Collection::stream)
                .distinct() // выше приходят дублирующиеся цели
                .collect(toList());

        // проставляем цели id счетчика
        Set<Long> counterIds = getAvailableAndExistingCounterIds(clientId, goalsByCampaignId.keySet());
        Map<Long, Set<Goal>> goalsFromMetrikaByCounterIds =
                getMetrikaGoalsByCounterIds(clientId, counterIds, false);
        Map<Integer, Integer> counterIdByGoalId = getCounterIdByGoalId(goalsFromMetrikaByCounterIds);
        resultGoals.forEach(goal -> setCounterId(counterIdByGoalId, goal));
        return resultGoals;
    }

    private Set<Long> getAvailableAndExistingCounterIds(ClientId clientId, Set<Long> campaignIds) {
        Set<Long> availableCounterIdsForClient =
                campMetrikaCountersService.getAvailableCounterIdsFromCampaignIdsForGoals(clientId, campaignIds);
        Set<Long> existingCounterIds = flatMapToSet(
                campMetrikaCountersService.getCounterByCampaignIds(clientId, campaignIds).values(),
                Function.identity());
        return Sets.union(availableCounterIdsForClient, existingCounterIds);
    }

    /**
     * Получить цели доступные для редактирования в ДРФ в кампаниях
     * https://st.yandex-team.ru/DIRECT-73912#5a9d3b9aec9e8d001a9c4494
     */
    public Map<Long, Set<Goal>> getAvailableBroadMatchesForCampaignId(
            Long operatorUid, ClientId clientId, Map<Long, CampaignType> campaignIdToCampaignType) {
        //для кампаний с OrderId == 0 возвращаем пустой список, смотри: DIRECT-105107
        int shard = shardHelper.getShardByClientId(clientId);
        var campaignIds = campaignIdToCampaignType.keySet();
        List<? extends BaseCampaign> typedCampaigns = campaignTypedRepository.getTypedCampaigns(shard, campaignIds);

        Map<Long, Set<Goal>> availableBroadMatchGoalsForCampaignId = StreamEx.of(typedCampaigns)
                .select(CommonCampaign.class)
                .filter(campaign -> campaign.getOrderId() == 0)
                .mapToEntry(CommonCampaign::getId, campaign -> Set.<Goal>of())
                .toMap();

        Set<Long> campaignIdsWithOrderId = StreamEx.of(typedCampaigns)
                .select(CommonCampaign.class)
                .filter(campaign -> campaign.getOrderId() != 0)
                .map(CommonCampaign::getId)
                .toSet();
        Map<Long, CampaignType> campaignIdWithOrderIdToCampaignType = EntryStream.of(campaignIdToCampaignType)
                .filterKeys(campaignIdsWithOrderId::contains)
                // DIRECT-124837 - toMap выбрасывает исключение в случае null
                .collect(HashMap::new, (m, v) -> m.put(v.getKey(), v.getValue()), HashMap::putAll);

        var campaignGoalsByCampaignId = getCampaignGoals(shard, operatorUid, clientId,
                campaignIdWithOrderIdToCampaignType);
        availableBroadMatchGoalsForCampaignId.putAll(campaignGoalsByCampaignId);

        Set<Long> cidsWithGoals = availableBroadMatchGoalsForCampaignId.keySet();
        Map<Long, MetrikaCounterWithAdditionalInformation> availableMetrikaCounterWithAdditionalInformationById =
                listToMap(campMetrikaCountersService.getAvailableCountersFromCampaignsForGoals(clientId, cidsWithGoals),
                        MetrikaCounterWithAdditionalInformation::getId, identity());

        Set<Integer> availableMetrikaCounterIds =
                listToSet(availableMetrikaCounterWithAdditionalInformationById.values(),
                        counter -> counter.getId().intValue());

        Set<Long> ecommerceGoalIds = getEcommerceGoalIds(campaignGoalsByCampaignId, availableMetrikaCounterIds);

        Map<Integer, List<CounterGoal>> massCountersGoalsFromMetrika =
                metrikaClient.getMassCountersGoalsFromMetrika(availableMetrikaCounterIds);

        Set<Long> availableGoalIdsForClient =
                getAvailableMetrikaAndMobileGoalIdsForClient(massCountersGoalsFromMetrika);

        boolean campaignModificationGoalsValidationTighteningEnabled = featureService.isEnabledForClientId(clientId,
                FeatureName.CAMPAIGN_MODIFICATION_GOALS_VALIDATION_TIGHTENING);

        Map<Long, Set<Goal>> result = EntryStream.of(removeDeletedGoals(availableBroadMatchGoalsForCampaignId))
                .mapValues(goals -> removeUnavailableGoalsIfNeeded(goals,
                        campaignModificationGoalsValidationTighteningEnabled,
                        availableGoalIdsForClient,
                        ecommerceGoalIds))
                .toMap();

        addCounterIdToGoals(massCountersGoalsFromMetrika, result);

        return EntryStream.of(result)
                .mapValues(goals -> expandGoalData(availableMetrikaCounterWithAdditionalInformationById, goals))
                .toMap();
    }

    /**
     * Получить все доступные ab_segments
     */
    public Map<Long, Set<Goal>> getAbSegments(ClientId clientId, Map<Long, Set<Long>> countersByCampaignId) {
        Collection<Long> clientRepresentativesUids =
                rbacService.getClientRepresentativesUidsForGetMetrikaCounters(clientId);
        Map<Long, List<RetargetingCondition>> goalsByUids = metrikaClient
                .getGoals(clientRepresentativesUids, GoalType.AB_SEGMENT).getUidToConditions();
        Set<RetargetingCondition> metrikaGoals = StreamEx.of(goalsByUids.values())
                .flatMap(Collection::stream)
                .toSet();
        return getCountersGoals(countersByCampaignId, metrikaGoals);
    }

    /**
     * Получить все доступные ab_segments
     */
    public Map<Long, Set<Goal>> getAbSegmentsByCounterIds(ClientId clientId, Set<Long> counterIds) {
        Collection<Long> clientRepresentativesUids =
                rbacService.getClientRepresentativesUidsForGetMetrikaCounters(clientId);
        Map<Long, List<RetargetingCondition>> goalsByUids = metrikaClient
                .getGoals(clientRepresentativesUids, GoalType.AB_SEGMENT).getUidToConditions();

        List<Goal> goals = StreamEx.of(goalsByUids.values())
                .flatMap(Collection::stream)
                .filter(goal -> counterIds.isEmpty() || countersHasGoal(counterIds, goal))
                .map(GoalConverter::fromMetrikaRetargetingCondition)
                .toList();

        return StreamEx.of(goals)
                .mapToEntry(MetrikaGoalsUtils::counterIdsFromGoal, Function.identity())
                .flatMapKeys(Collection::stream)
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .filterKeys(id -> counterIds.isEmpty() || counterIds.contains(id))
                .mapValues(Set::copyOf)
                .toMap();
    }

    /**
     * Возвращает доступные для редактирования ключевые цели в кампаниях
     * https://st.yandex-team.ru/DIRECT-73912#5a9d3b9aec9e8d001a9c4494
     *
     * @param operatorUid              uid оператора
     * @param clientId                 id клиента
     * @param campaignIdToCampaignType словарь соответствий типа кампании её id
     * @param countersByCampaignId     словарь счетчиков по кампаниям
     * @return Словарь целей по кампаниям
     * @deprecated замена {@link ru.yandex.direct.core.entity.metrika.service.campaigngoals.CampaignGoalsService#getAvailableGoalsForCampaignId}
     */
    public Map<Long, Set<Goal>> getAvailableMeaningfulGoalsForCampaignId(
            Long operatorUid, ClientId clientId,
            Map<Long, CampaignType> campaignIdToCampaignType,
            Map<Long, Set<Long>> countersByCampaignId) {
        int shard = shardHelper.getShardByClientId(clientId);
        Map<Long, Set<Goal>> countersGoalsByCampaignId = getMetrikaGoalsByCampaignId(countersByCampaignId);

        var campaignIds = campaignIdToCampaignType.keySet();
        Map<Long, Set<Goal>> turbolandingGoalsByCampaignId = getTurboLandingCountersGoals(shard, campaignIds);
        var campaignGoalsByCampaignId = getCampaignGoals(shard, operatorUid, clientId, campaignIdToCampaignType);

        Set<Long> inputCounterIds = StreamEx.ofValues(countersByCampaignId).flatMap(StreamEx::of).toSet();
        Map<Long, MetrikaCounterWithAdditionalInformation> availableMetrikaCounterWithAdditionalInformationById =
                listToMap(campMetrikaCountersService.getAvailableAndFilterInputCountersInMetrikaForGoals(clientId,
                        inputCounterIds),
                        MetrikaCounterWithAdditionalInformation::getId, identity());

        Set<Integer> availableMetrikaCounterIds =
                listToSet(availableMetrikaCounterWithAdditionalInformationById.values(),
                        counter -> counter.getId().intValue());

        boolean coldStartForEcommerceEnabled = featureService.isEnabledForClientId(clientId,
                FeatureName.COLD_START_FOR_ECOMMERCE_GOALS);
        Set<Long> ecommerceGoalIds;
        Map<Long, Set<Goal>> ecommerceGoalsByCampaignId;
        if (coldStartForEcommerceEnabled) {
            Set<Goal> ecommerceGoalsFromMetrika =
                    getAllEcommerceGoalsFromMetrikaCounters(availableMetrikaCounterWithAdditionalInformationById);

            ecommerceGoalIds = mapSet(ecommerceGoalsFromMetrika, GoalBase::getId);

            Map<Long, Goal> ecommerceGoalsFromMetrikaByCounterId = listToMap(ecommerceGoalsFromMetrika,
                    goal -> goal.getCounterId().longValue());

            ecommerceGoalsByCampaignId = EntryStream.of(countersByCampaignId)
                    .mapValues(campaignCounterIds -> filterAndMapToSet(campaignCounterIds,
                            ecommerceGoalsFromMetrikaByCounterId::containsKey,
                            ecommerceGoalsFromMetrikaByCounterId::get))
                    .toMap();
        } else {
            ecommerceGoalIds = getEcommerceGoalIds(campaignGoalsByCampaignId,
                    availableMetrikaCounterIds);
            ecommerceGoalsByCampaignId = emptyMap();
        }

        Map<Integer, List<CounterGoal>> massCountersGoalsFromMetrika =
                metrikaClient.getMassCountersGoalsFromMetrika(availableMetrikaCounterIds);

        Set<Long> availableGoalIdsForClient =
                getAvailableMetrikaAndMobileGoalIdsForClient(massCountersGoalsFromMetrika);

        boolean campaignModificationGoalsValidationTighteningEnabled = featureService.isEnabledForClientId(clientId,
                FeatureName.CAMPAIGN_MODIFICATION_GOALS_VALIDATION_TIGHTENING);

        Map<Long, Set<Goal>> result = StreamEx.of(campaignIds)
                .mapToEntry(campaignId -> concatGoals(countersGoalsByCampaignId.get(campaignId),
                        turbolandingGoalsByCampaignId.get(campaignId),
                        campaignGoalsByCampaignId.get(campaignId),
                        ecommerceGoalsByCampaignId.getOrDefault(campaignId, null)))
                .mapValues(goals -> removeUnavailableGoalsIfNeeded(goals,
                        campaignModificationGoalsValidationTighteningEnabled,
                        availableGoalIdsForClient,
                        ecommerceGoalIds))
                .toMap();


        addCounterIdToGoals(massCountersGoalsFromMetrika, result);

        Map<Long, Set<Goal>> expandedResult = EntryStream.of(result)
                .mapValues(goals -> expandGoalData(availableMetrikaCounterWithAdditionalInformationById, goals))
                .toMap();

        return removeDeletedGoals(expandedResult);
    }

    private void addCounterIdToGoals(Map<Integer, List<CounterGoal>> massCountersGoalsFromMetrika,
                                     Map<Long, Set<Goal>> goalsByCampaignId) {

        Set<Goal> goals = flatMapToSet(goalsByCampaignId.values(), Function.identity());

        addCounterIdToGoals(massCountersGoalsFromMetrika, goals);
    }

    private void addCounterIdToGoals(Map<Integer, List<CounterGoal>> massCountersGoalsFromMetrika,
                                     Set<Goal> goals) {
        Map<Integer, Integer> counterIdByGoalId = EntryStream.of(massCountersGoalsFromMetrika)
                .flatMapValues(Collection::stream)
                .mapValues(CounterGoal::getId)
                .invert()
                .toMap();
        goals.forEach(goal -> setCounterId(counterIdByGoalId, goal));
    }

    /**
     * Возвращает цели, которые используются в настройках стратегий или как ключевые.
     */
    public Map<Long, GoalCampUsages> getGoalsUsedInCampaignsByClientId(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return metrikaCampaignRepository.getGoalsUsedInCampaignsByClientId(shard, clientId);
    }

    private void setCounterId(Map<Integer, Integer> counterIdByGoalId, Goal goal) {
        if (MetrikaCounterGoalType.ECOMMERCE == goal.getMetrikaCounterGoalType()) {
            goal.setCounterId(getCounterIdForEcommerceGoal(goal));
            return;
        }

        Integer goalCounterId = counterIdByGoalId.get(goal.getId().intValue());
        Integer parentGoalCounterId = counterIdByGoalId.get(ifNotNull(goal.getParentId(), Long::intValue));
        goal.setCounterId(nullableNvl(goalCounterId, parentGoalCounterId));
    }

    private Set<Long> getEcommerceGoalIds(Map<Long, Set<Goal>> campaignGoalsByCampaignId,
                                          Set<Integer> availableCounterIdsForClient) {
        return getEcommerceGoalIds(flatMapToSet(campaignGoalsByCampaignId.values(),
                Function.identity()),
                availableCounterIdsForClient);
    }

    private Set<Long> getEcommerceGoalIds(Set<Goal> campaignGoals,
                                          Set<Integer> availableCounterIds) {
        return filterAndMapToSet(campaignGoals,
                g -> g.getMetrikaCounterGoalType() == MetrikaCounterGoalType.ECOMMERCE &&
                        availableCounterIds.contains(getCounterIdForEcommerceGoal(g)),
                GoalBase::getId);
    }

    public static Integer getCounterIdForEcommerceGoal(Goal goal) {
        return Integer.valueOf(goal.getName());
    }

    private Map<Long, Set<Goal>> removeDeletedGoals(Map<Long, Set<Goal>> campaignToGoals) {
        return EntryStream.of(campaignToGoals)
                .mapValues(set -> StreamEx.of(set)
                        .filter(goal -> goal.getStatus() != GoalStatus.DELETED)
                        .toSet())
                .toMap();
    }

    private Set<Goal> expandGoalData(
            Map<Long, MetrikaCounterWithAdditionalInformation> metrikaCounterWithAdditionalInformationById,
            Collection<Goal> goals) {
        return StreamEx.of(goals)
                .map(goal -> {
                    MetrikaCounterWithAdditionalInformation counterInfo = metrikaCounterWithAdditionalInformationById
                            .get(ifNotNull(goal.getCounterId(), Integer::longValue));
                    return (Goal) goal
                            .withDomain(ifNotNull(counterInfo, MetrikaCounterWithAdditionalInformation::getDomain));
                })
                .map(MetrikaGoalsService::setMobileFlagForGoal)
                .toSet();
    }

    private static Goal setMobileFlagForGoal(Goal goal) {
        boolean isMobileGoal = MOBILE_GOAL_IDS.contains(goal.getId());
        return (Goal) goal.withIsMobileGoal(isMobileGoal);
    }

    /**
     * Возвращает все доступные цели клиента по доступным счетчикам, включая составные цели и ecommerce-цели.
     * В случае недоступности Метрики вернет null.
     */
    @Nullable
    public Set<Long> getAvailableMetrikaGoalIdsForClientWithExceptionHandling(Long operatorUid, ClientId clientId) {
        try {
            Set<MetrikaCounterWithAdditionalInformation> availableCounters =
                    campMetrikaCountersService.getAvailableCountersForGoals(clientId);
            return mapSet(getAvailableMetrikaGoalsForClient(operatorUid, clientId, availableCounters), Goal::getId);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika goals for clientId: {}", clientId, e);
            return null;
        }
    }

    /**
     * Возвращает все доступные цели клиента по доступным счетчикам, включая составные цели и ecommerce-цели
     */
    public Set<Goal> getAvailableMetrikaGoalsForClient(Long operatorUid, ClientId clientId) {
        Set<MetrikaCounterWithAdditionalInformation> availableCounters =
                campMetrikaCountersService.getAvailableCountersForGoals(clientId);
        return getAvailableMetrikaGoalsForClient(operatorUid, clientId, availableCounters);
    }

    /**
     * Возвращает все доступные цели клиента по указанным счетчикам, включая составные цели и ecommerce-цели
     */
    public Set<Goal> getAvailableMetrikaGoalsForClient(Long operatorUid, ClientId clientId,
                                                       Collection<MetrikaCounterWithAdditionalInformation> counters) {

        Map<Long, MetrikaCounterWithAdditionalInformation> countersById =
                listToMap(counters, MetrikaCounterWithAdditionalInformation::getId, identity());

        Set<Integer> counterIdSet = listToSet(counters, c -> c.getId().intValue());

        Map<Integer, List<CounterGoal>> goalsByCounterId =
                metrikaClient.getMassCountersGoalsFromMetrika(counterIdSet);

        Set<Goal> metrikaGoals = getAvailableGoalsForStrategies(goalsByCounterId, true);

        //получаем все ecommerce цели клиента
        List<Goal> existedClientGoals = getGoalsWithCounters(operatorUid, clientId,
                countersById.keySet(), goalsByCounterId);

        boolean coldStartForEcommerceEnabled = featureService.isEnabledForClientId(clientId,
                FeatureName.COLD_START_FOR_ECOMMERCE_GOALS);
        Set<Goal> ecommerceGoalsFromMetrika = null;
        Set<Long> ecommerceGoalIds = null;
        if (coldStartForEcommerceEnabled) {
            ecommerceGoalsFromMetrika = getAllEcommerceGoalsFromMetrikaCounters(countersById);
            ecommerceGoalIds = mapSet(ecommerceGoalsFromMetrika, GoalBase::getId);
        } else {
            ecommerceGoalIds = getEcommerceGoalIds(listToSet(existedClientGoals), counterIdSet);
        }

        Set<Long> availableGoalIdsForClient =
                getAvailableMetrikaAndMobileGoalIdsForClient(goalsByCounterId);

        Set<Goal> goals = removeUnavailableGoalsIfNeeded(concatGoals(metrikaGoals, Set.of(),
                listToSet(existedClientGoals), ecommerceGoalsFromMetrika),
                featureService.isEnabledForClientId(clientId,
                        FeatureName.CAMPAIGN_MODIFICATION_GOALS_VALIDATION_TIGHTENING),
                availableGoalIdsForClient, ecommerceGoalIds);

        addCounterIdToGoals(goalsByCounterId, goals);
        addConversionLevelToGoals(goals);

        return expandGoalData(countersById, goals);
    }

    public Set<Goal> getMetrikaGoalsByCounter(Long operatorUid, ClientId clientId, Collection<Long> counterIds,
                                              @Nullable Long campaignId, @Nullable CampaignType campaignType) {
        boolean campaignModificationGoalsValidationTighteningEnabled = featureService.isEnabledForClientId(clientId,
                FeatureName.CAMPAIGN_MODIFICATION_GOALS_VALIDATION_TIGHTENING);
        Map<Long, CampaignType> campaignTypeMap = Map.of();
        MobileGoalsFilter mobileGoalsFilter = null;
        if (campaignId != null && campaignType != null) {
            campaignTypeMap = Map.of(campaignId, campaignType);
        } else if (campaignType != null) {
            mobileGoalsFilter = new ForCampaignType(campaignType);
        }
        return getMetrikaGoalsByCounters(operatorUid, clientId, counterIds, Set.of(), campaignTypeMap,
                mobileGoalsFilter, campaignModificationGoalsValidationTighteningEnabled, false);
    }

    /**
     * Аналог ручки cmd_ajaxGetMetrikaGoalsByCounter из перла
     * <p>
     * Получает цели по заданным счетчикам из метрики и записывает их в ppcdict.metrika_goals, ppc.camp_metrika_goals.
     * Отдает цели из метрики по counterIds и из ppc.camp_metrika_goals по campaignId
     * В завимиости от FeatureName.ALLOW_STEP_GOALS_IN_STRATEGIES отдаем составные цели
     *
     * @param clientId          клиент
     * @param counterIds        cписок счетчиков
     * @param campaignTypeById  список идентификаторов кампаний с их типами
     * @param mobileGoalsFilter фильтр для получения списка доступных мобильных целей
     * @return список целей
     */
    public Set<Goal> getMetrikaGoalsByCounters(Long operatorUid,
                                               ClientId clientId,
                                               Collection<Long> counterIds,
                                               Set<Long> unavailableEcommerceCounterIds,
                                               Map<Long, CampaignType> campaignTypeById,
                                               @Nullable MobileGoalsFilter mobileGoalsFilter,
                                               boolean removeUnavailableGoals,
                                               boolean isStrictFilterByInputCounters) {
        int shard = shardHelper.getShardByClientId(clientId);
        Set<Goal> existedCampaignGoals = emptySet();
        if (!campaignTypeById.isEmpty()) {
            existedCampaignGoals = getCampaignGoals(shard, operatorUid, clientId, campaignTypeById)
                    .values()
                    .stream()
                    .flatMap(Collection::stream)
                    .collect(Collectors.toSet());
        } else if (mobileGoalsFilter != null) {
            // для создаваемых автобюджетных кампаний внутренней рекламы ТГО кампаний добавляем цели из РМП
            existedCampaignGoals = listToSet(
                    mobileGoalsService.getMobileGoalsForFilter(
                            operatorUid,
                            clientId,
                            mobileGoalsFilter)
            );
        }
        return getMetrikaGoalsByCounter(
                clientId,
                counterIds,
                unavailableEcommerceCounterIds,
                existedCampaignGoals,
                removeUnavailableGoals,
                isStrictFilterByInputCounters
        );
    }

    private Set<Goal> getMetrikaGoalsByCounter(ClientId clientId,
                                               Collection<Long> counterIds,
                                               Set<Long> unavailableEcommerceCounterIds,
                                               Set<Goal> existedCampaignGoals,
                                               boolean removeUnavailableGoals,
                                               boolean isStrictFilterByInputCounters) {

        Set<Integer> counterIdSet = listToSet(counterIds, Long::intValue);
        Map<Integer, List<CounterGoal>> counterToGoal =
                metrikaClient.getMassCountersGoalsFromMetrika(counterIdSet);
        Set<String> clientFeatures = featureService.getEnabledForClientId(clientId);
        boolean stepGoalsIsAllowed = clientFeatures.contains(FeatureName.ALLOW_STEP_GOALS_IN_STRATEGIES.getName());
        Set<Goal> metrikaGoals = getAvailableGoalsForStrategies(counterToGoal, stepGoalsIsAllowed);

        Map<Long, MetrikaCounterWithAdditionalInformation> availableMetrikaCounterWithAdditionalInformationById =
                listToMap(campMetrikaCountersService.getAvailableAndFilterInputCountersInMetrikaForGoals(clientId,
                        counterIds, isStrictFilterByInputCounters), MetrikaCounterWithAdditionalInformation::getId,
                        identity());

        Set<Integer> availableMetrikaCounterIds =
                listToSet(availableMetrikaCounterWithAdditionalInformationById.values(),
                        counter -> counter.getId().intValue());

        // получаем ecom из счетчиков метрики, если холодный старт включен
        boolean coldStartForEcommerceEnabled = featureService.isEnabledForClientId(clientId,
                FeatureName.COLD_START_FOR_ECOMMERCE_GOALS);
        Set<Goal> ecommerceGoalsAvailableForCampaign = null;
        Set<Long> ecommerceGoalIds = null;
        if (coldStartForEcommerceEnabled) {
            Set<Goal> allEcommerceGoals = getAllEcommerceGoalsFromMetrikaCounters(
                    availableMetrikaCounterWithAdditionalInformationById);
            if (!removeUnavailableGoals && !unavailableEcommerceCounterIds.isEmpty()) {
                Set<Goal> unavailableEcommerceGoals =
                        mapSet(unavailableEcommerceCounterIds, MetrikaGoalsUtils::ecommerceGoalFromCounterId);
                allEcommerceGoals = Sets.union(allEcommerceGoals, unavailableEcommerceGoals);
            }
            ecommerceGoalsAvailableForCampaign = filterToSet(allEcommerceGoals,
                    goal -> counterIdSet.contains(goal.getCounterId()));
            ecommerceGoalIds = mapSet(allEcommerceGoals, GoalBase::getId);
        }

        // если холодный старт выключен, получаем ecom из camp_metrika_goals
        if (!coldStartForEcommerceEnabled) {
            ecommerceGoalIds = getEcommerceGoalIds(existedCampaignGoals, availableMetrikaCounterIds);
        }

        Map<Integer, List<CounterGoal>> massCountersGoalsFromMetrika =
                metrikaClient.getMassCountersGoalsFromMetrika(availableMetrikaCounterIds);

        Set<Long> availableGoalIdsForClient =
                getAvailableMetrikaAndMobileGoalIdsForClient(massCountersGoalsFromMetrika);

        Set<Goal> goals = concatGoals(metrikaGoals, Set.of(), existedCampaignGoals,
                ecommerceGoalsAvailableForCampaign);
        addCounterIdToGoals(counterToGoal, goals);
        addConversionLevelToGoals(goals);

        goals = removeUnavailableGoalsIfNeeded(goals, removeUnavailableGoals,
                availableGoalIdsForClient, ecommerceGoalIds);

        //todo: сейчас для некоторых клиентов "мягкая" валидация -- поэтому недоступные составные цели
        //не отфильтровались в removeUnavailableGoalsIfNeeded, удалим их из ответа, если нужно
        Set<Goal> goalsWithoutStepGoals = filterToSet(goals,
                goal -> MetrikaCounterGoalType.STEP != goal.getMetrikaCounterGoalType() &&
                        (goal.getParentId() == null || goal.getParentId() == 0));

        return expandGoalData(availableMetrikaCounterWithAdditionalInformationById, stepGoalsIsAllowed ?
                goals : goalsWithoutStepGoals);
    }

    public Set<Goal> getAllEcommerceGoalsFromMetrikaCounters(Map<Long,
            MetrikaCounterWithAdditionalInformation> availableMetrikaCounterWithAdditionalInformationById) {
        return EntryStream.of(availableMetrikaCounterWithAdditionalInformationById)
                .filterValues(MetrikaCounterWithAdditionalInformation::getHasEcommerce)
                .keys()
                .map(MetrikaGoalsUtils::ecommerceGoalFromCounterId)
                .toSet();
    }

    private Set<Goal> getAvailableGoalsForStrategies(Map<Integer, List<CounterGoal>> counterToGoal,
                                                     boolean stepGoalsIsAllowed) {
        if (stepGoalsIsAllowed) {
            return counterToGoal.values().stream()
                    .flatMap(List::stream)
                    .map(GoalConverter::fromCounterGoal)
                    .collect(toSet());
        } else {
            return counterToGoal.values().stream()
                    .flatMap(List::stream)
                    .filter(counterGoal -> counterGoal.getType() != CounterGoal.Type.STEP)
                    .filter(counterGoal -> counterGoal.getParentId() == null || counterGoal.getParentId() == 0)
                    .map(GoalConverter::fromCounterGoal)
                    .collect(toSet());
        }
    }

    public void addMetrikaGoals(Set<Goal> metrikaGoals) {
        Set<Long> metrikaGoalIds = mapSet(metrikaGoals, Goal::getId);

        Set<Long> metrikaGoalIdsInPpcDict =
                listToSet(retargetingGoalsPpcDictRepository.getMetrikaGoalsFromPpcDict(metrikaGoalIds), Goal::getId);

        Set<Goal> goalsToAddInPpcDict = filterAndMapToSet(metrikaGoals,
                goal -> !metrikaGoalIdsInPpcDict.contains(goal.getId()), identity());

        retargetingGoalsPpcDictRepository.addMetrikaGoalsToPpcDict(goalsToAddInPpcDict);
    }

    public void addCampMetrikaGoals(int shard, ClientId clientId, Map<Long, Set<Long>> metrikaGoalsByCampaignId) {
        Set<Long> metrikaGoalIdsInPpc = metrikaCampaignRepository.getGoalIds(shard, clientId.asLong(),
                metrikaGoalsByCampaignId.keySet());

        Map<Long, Set<Long>> metrikaGoalsByCampaignIdToAddInPpc = EntryStream.of(metrikaGoalsByCampaignId)
                .mapValues(goalIds -> filterToSet(goalIds,
                        goalId -> !metrikaGoalIdsInPpc.contains(goalId)))
                .toMap();

        metrikaCampaignRepository.addGoalIds(shard, metrikaGoalsByCampaignIdToAddInPpc);
    }

    private Map<Long, Set<Goal>> getMetrikaGoalsByCampaignId(@Nullable Map<Long, Set<Long>> countersByCampaignId) {
        if (countersByCampaignId == null) {
            return Collections.emptyMap();
        }

        Set<Integer> counterIds = EntryStream.of(countersByCampaignId)
                .flatMapValues(Collection::stream)
                .mapValues(Long::intValue)
                .values()
                .toSet();

        Map<Integer, List<CounterGoal>> countersGoals =
                getCountersGoalsWithExceptionHandling(counterIds);

        return EntryStream.of(countersByCampaignId)
                .mapValues(counters -> getGoalsForCampaignCounters(countersGoals, counters))
                .toMap();
    }

    private Set<Goal> getGoalsForCampaignCounters(Map<Integer, List<CounterGoal>> countersGoals,
                                                  Set<Long> counters) {
        return StreamEx.of(counters)
                .map(Long::intValue)
                .filter(countersGoals::containsKey)
                .map(countersGoals::get)
                .map(GoalConverter::fromCounterGoals)
                .flatMap(Collection::stream)
                .toSet();
    }

    /**
     * Если по какой-либо причине счетчик удален в метрике при запросе целей метрика вернет нам ошибку
     * Чтобы не падать в этом месте делаем обертку
     */
    private Map<Integer, List<CounterGoal>> getCountersGoalsWithExceptionHandling(Set<Integer> counterIds) {
        try {
            return metrikaClient.getMassCountersGoalsFromMetrika(counterIds);
        } catch (MetrikaClientException e) {
            logger.warn("Counters with ids: " + StreamEx.of(counterIds).joining() + " is unavailable", e);
            return emptyMap();
        }
    }

    public Map<Long, Set<Goal>> getMetrikaGoalsByCounterIds(ClientId clientId, Set<Long> counterIds) {
        return getMetrikaGoalsByCounterIds(clientId, counterIds, true);
    }

    /**
     * Получить цели по счетчикам метрики
     *
     * @param clientId       id клиента
     * @param counterIds     список счетчиков
     * @param expandGoalData нужно ли дополнять цели информацией из метрики
     * @return словарь соответствий "id счетчика метрики" - "цели по счетчику"*
     * @implNote {@code expandGoalData == true} влечет поход в метрику
     */
    public Map<Long, Set<Goal>> getMetrikaGoalsByCounterIds(ClientId clientId, Set<Long> counterIds,
                                                            boolean expandGoalData) {
        Set<Integer> counterIntegerIds = listToSet(counterIds, Long::intValue);
        Map<Integer, List<CounterGoal>> counterGoals = getCountersGoalsWithExceptionHandling(counterIntegerIds);
        return getMetrikaGoalsByCounterIds(clientId, counterIds, counterGoals, expandGoalData);
    }

    /**
     * Получить цели по счетчикам метрики
     *
     * @implNote {@code expandGoalData == true} влечет поход в метрику
     */
    private Map<Long, Set<Goal>> getMetrikaGoalsByCounterIds(ClientId clientId, Set<Long> counterIds,
                                                             Map<Integer, List<CounterGoal>> counterGoals,
                                                             boolean expandGoalData) {
        EntryStream<Long, Set<Goal>> counterIdToGoalStream = EntryStream.of(counterGoals)
                .mapValues(GoalConverter::fromCounterGoals)
                .mapValues(Set::copyOf)
                .mapToValue((counterId, goals) -> mapSet(goals, goal -> (Goal) goal.withCounterId(counterId)))
                .mapKeys(Integer::longValue);

        if (expandGoalData) {
            Map<Long, MetrikaCounterWithAdditionalInformation> availableMetrikaCounterWithAdditionalInformationById =
                    listToMap(campMetrikaCountersService.getAvailableAndFilterInputCountersInMetrikaForGoals(clientId,
                            counterIds),
                            MetrikaCounterWithAdditionalInformation::getId, identity());
            counterIdToGoalStream = counterIdToGoalStream
                    .mapValues(goals -> expandGoalData(availableMetrikaCounterWithAdditionalInformationById, goals));
        }

        Map<Long, Set<Goal>> counterIdToGoal = counterIdToGoalStream.toMap();
        if (!featureService.isEnabledForClientId(clientId, FeatureName.GOALS_ONLY_WITH_CAMPAIGN_COUNTERS_USED)) {
            StreamEx.of(counterIds)
                    .distinct()
                    .filter(id -> !counterIdToGoal.containsKey(id))
                    .forEach(id -> counterIdToGoal.put(id, Set.of()));
        }
        return counterIdToGoal;
    }

    private static Set<Goal> concatGoals(Set<Goal> countersGoals, Set<Goal> turbolandingGoals,
                                         Set<Goal> campaignGoals, Set<Goal> ecommerceGoals) {
        Set<Goal> result = new HashSet<>();
        if (countersGoals != null) {
            result.addAll(countersGoals);
        }
        if (turbolandingGoals != null) {
            result.addAll(turbolandingGoals);
        }
        if (campaignGoals != null) {
            result.addAll(campaignGoals);
        }
        if (ecommerceGoals != null) {
            result.addAll(ecommerceGoals);
        }
        return StreamEx.of(result)
                .distinct(GoalBase::getId)
                .toSet();
    }

    /**
     * Получить все цели, привязанные к счётчикам турболендингов кампании
     */
    private Map<Long, Set<Goal>> getTurboLandingCountersGoals(int shard, Collection<Long> campaignIds) {
        Map<Long, Set<Long>> turbolandingCountersByCampaignId =
                metrikaCampaignRepository.getTurbolandingCounters(shard, campaignIds);
        return getMetrikaGoalsByCampaignId(turbolandingCountersByCampaignId);
    }

    /**
     * Получить все цели, привязанные к обычным счётчикам кампании
     */
    private Map<Long, Set<Goal>> getCountersGoals(@Nullable Map<Long, Set<Long>> countersByCampaignId,
                                                  Set<RetargetingCondition> metrikaGoals) {
        if (countersByCampaignId == null) {
            return Map.of();
        }
        return getGoalsByCampaignIds(metrikaGoals, countersByCampaignId);
    }

    /**
     * Получить все цели, по которым есть статистика кампании
     */
    private Map<Long, Set<Goal>> getCampaignGoals(int shard, Long operatorUid, ClientId clientId,
                                                  Map<Long, CampaignType> campaignIdToCampaignType) {
        var goalsByCampaignId = getGoalsByCampaignId(shard, operatorUid, clientId, campaignIdToCampaignType, true);
        return EntryStream.of(goalsByCampaignId)
                .mapValues(Set::copyOf)
                .toMap();
    }

    public List<Goal> getGoalsWithCounters(Long operatorUid, ClientId clientId, @Nullable Set<Long> campaignIds) {
        return getGoalsWithCounters(operatorUid, clientId, campaignIds, true);
    }

    /**
     * https://st.yandex-team.ru/DIRECT-108651
     * Получить все актуальные (доступные и разрешенные Метрикой недоступные) цели,
     * счетчики которых привязаны к кампаниям
     * с предварительной проверкой их наличия в метрике
     */
    public List<Goal> getGoalsWithCounters(Long operatorUid, ClientId clientId, @Nullable Set<Long> campaignIds,
                                           boolean expandGoalData) {
        var goalsByCampaignId = getGoalsByCampaignId(operatorUid, clientId, campaignIds);
        Set<Goal> setOfGoalsFromCampMetrikaGoals = flatMapToSet(goalsByCampaignId.values(), identity());

        //эта ручка используется для статистики, тут важно, что не берутся системные счетчики метрики
        Set<Long> cidsWithGoals = goalsByCampaignId.keySet();
        Set<Long> counterIds = getAvailableAndAllowedUnavailableCounterIds(clientId, cidsWithGoals);
        Map<Long, Set<Goal>> goalsFromMetrikaByCounterIds =
                getMetrikaGoalsByCounterIds(clientId, counterIds, expandGoalData);
        return getGoalsWithCounters(counterIds, setOfGoalsFromCampMetrikaGoals, goalsFromMetrikaByCounterIds);
    }

    /**
     * Получить доступные и разрешенные недоступные счетчики на переданных кампаниях
     */
    private Set<Long> getAvailableAndAllowedUnavailableCounterIds(ClientId clientId, Set<Long> campaignIds) {
        Set<Long> availableCounterIdsForClient =
                campMetrikaCountersService.getAvailableCounterIdsFromCampaignIdsForGoals(clientId, campaignIds);
        Set<Long> existingCounterIds = flatMapToSet(
                campMetrikaCountersService.getCounterByCampaignIds(clientId, campaignIds).values(),
                Function.identity());
        Set<Long> unavailableCounterIds = Sets.difference(existingCounterIds, availableCounterIdsForClient);
        Set<Long> allowedByMetrikaUnavailableCounterIds =
                campMetrikaCountersService.getAllowedInaccessibleCounterIds(unavailableCounterIds);
        return Sets.union(availableCounterIdsForClient, allowedByMetrikaUnavailableCounterIds);
    }

    /**
     * Получить актуальные цели для указанных счетчиков
     * с предварительной проверкой их наличия в метрике
     */
    private List<Goal> getGoalsWithCounters(Long operatorUid, ClientId clientId,
                                            Set<Long> counterIdsForClient,
                                            Map<Integer, List<CounterGoal>> goalsByCounterId) {
        var goalsByCampaignId = getGoalsByCampaignId(operatorUid, clientId, null);
        Set<Goal> setOfGoalsFromCampMetrikaGoals = flatMapToSet(goalsByCampaignId.values(), identity());

        Map<Long, Set<Goal>> goalsFromMetrikaByCounterIds = getMetrikaGoalsByCounterIds(clientId,
                counterIdsForClient, goalsByCounterId, true);
        return getGoalsWithCounters(counterIdsForClient, setOfGoalsFromCampMetrikaGoals,
                goalsFromMetrikaByCounterIds);
    }

    private Map<Long, List<Goal>> getGoalsByCampaignId(Long operatorUid, ClientId clientId,
                                                       @Nullable Set<Long> campaignIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        var campaignIdToCampaignType = ifNotNull(campaignIds,
                ids -> campaignRepository.getCampaignsTypeMap(shard, ids));
        return getGoalsByCampaignId(shard, operatorUid, clientId, campaignIdToCampaignType, false);
    }

    private List<Goal> getGoalsWithCounters(Set<Long> counterIdsForClient,
                                            Set<Goal> setOfGoalsFromCampMetrikaGoals,
                                            Map<Long, Set<Goal>> goalsFromMetrikaByCounterIds) {
        List<Goal> ecommerceGoals = filterList(setOfGoalsFromCampMetrikaGoals,
                g -> g.getMetrikaCounterGoalType() == MetrikaCounterGoalType.ECOMMERCE);

        ecommerceGoals = filterList(ecommerceGoals,
                g -> counterIdsForClient.contains(Long.valueOf(g.getName())));
        Set<Goal> setOfGoalsFromMetrikaByCounterIds = flatMapToSet(goalsFromMetrikaByCounterIds.values(), identity());
        Set<Long> goalIdsFromMetrika = mapSet(setOfGoalsFromMetrikaByCounterIds, Goal::getId);
        List<Goal> goalsFromMetrika = new ArrayList<>(removeUnavailableGoalsIfNeeded(setOfGoalsFromCampMetrikaGoals,
                true, goalIdsFromMetrika, MOBILE_GOAL_IDS));
        List<Goal> resultGoals = ListUtils.union(goalsFromMetrika, ecommerceGoals);

        Map<Integer, Integer> counterIdByGoalId = getCounterIdByGoalId(goalsFromMetrikaByCounterIds);
        resultGoals.forEach(goal -> setCounterId(counterIdByGoalId, goal));
        return resultGoals;
    }

    private Map<Integer, Integer> getCounterIdByGoalId(Map<Long, Set<Goal>> goalsByCounterIds) {
        return EntryStream.of(goalsByCounterIds)
                .flatMapValues(Collection::stream)
                .mapKeys(Long::intValue)
                .mapValues(goal -> goal.getId().intValue())
                .invert()
                .toMap();
    }

    /**
     * Возвращает словарь целей для клиента по кампаниям
     *
     * @param operatorUid              uid оператора
     * @param clientId                 id клиента
     * @param campaignIdToCampaignType словарь соответствий типа кампании её id
     * @param isUsedMobileGoalsNeeded  нужны ли используемые цели
     * @return список целей
     */
    private Map<Long, List<Goal>> getGoalsByCampaignId(int shard, Long operatorUid, ClientId clientId,
                                                       @Nullable Map<Long, CampaignType> campaignIdToCampaignType,
                                                       boolean isUsedMobileGoalsNeeded) {
        var campaignIds = ifNotNull(campaignIdToCampaignType, Map::keySet);
        Map<Long, List<Long>> goalIdsByCampaignId = metrikaCampaignRepository
                .getGoalIdsByCampaignId(shard, clientId.asLong(), campaignIds);

        //в базе могут храниться невалидные цели с мобильными goal_id
        var goalIdsWithoutMobileByCampaignId = EntryStream.of(goalIdsByCampaignId)
                .mapValues(goalIds -> filterList(goalIds, goalId -> !MOBILE_GOAL_IDS.contains(goalId)))
                .toMap();

        Map<Long, List<Goal>> goalsWithoutMobileByCampaignId = getGoalsByCampaignId(goalIdsWithoutMobileByCampaignId);

        Map<Long, List<Long>> parentGoalIdsByCampaignId = EntryStream.of(goalsWithoutMobileByCampaignId)
                .mapValues(MetrikaGoalsUtils::getParentIds)
                .toMap();
        Map<Long, List<Goal>> parentGoalsByCampaignId = getGoalsByCampaignId(parentGoalIdsByCampaignId);

        Map<Long, List<Goal>> mobileGoalsByCampaignId = mobileGoalsService.getMobileGoalsByCampaignId(shard,
                operatorUid, clientId,
                calculateCampaignTypesByIdIfNeed(shard, campaignIdToCampaignType, goalIdsByCampaignId.keySet()),
                isUsedMobileGoalsNeeded);

        return EntryStream.of(goalsWithoutMobileByCampaignId)
                .append(parentGoalsByCampaignId)
                .append(mobileGoalsByCampaignId)
                .toMap(ListUtils::union);
    }

    /**
     * Возвращает словарь id целей для клиента по кампаниям
     *
     * @param operatorUid              идентификатор оператора
     * @param clientId                 id клиента
     * @param campaignIdToCampaignType словарь соответствий типа кампании её id
     * @return список идентификаторов целей
     */
    public Map<Long, List<Long>> getGoalIdsByCampaignId(Long operatorUid, ClientId clientId,
                                                        Map<Long, CampaignType> campaignIdToCampaignType,
                                                        boolean isUsedMobileGoalsNeeded) {
        final int shard = shardHelper.getShardByClientId(clientId);
        var campaignIds = ifNotNull(campaignIdToCampaignType, Map::keySet);
        Map<Long, List<Long>> goalIdsByCampaignId = metrikaCampaignRepository.getGoalIdsByCampaignId(shard,
                clientId.asLong(), campaignIds);

        var mobileGoalIdsByCampaignId = mobileGoalsService.getMobileGoalIdsByCampaignId(shard, operatorUid, clientId,
                campaignIdToCampaignType, isUsedMobileGoalsNeeded);

        return EntryStream.of(goalIdsByCampaignId)
                .append(mobileGoalIdsByCampaignId)
                .toMap(ListUtils::union);
    }

    private Map<Long, CampaignType> calculateCampaignTypesByIdIfNeed(
            int shard, Map<Long, CampaignType> campaignIdToCampaignType, Collection<Long> campaignIds) {
        return campaignIdToCampaignType != null
                ? campaignIdToCampaignType
                : campaignRepository.getCampaignsTypeMap(shard, campaignIds);
    }

    private Map<Long, List<Goal>> getGoalsByCampaignId(Map<Long, List<Long>> goalIdsByCampaignId) {
        List<Long> goalIds = StreamEx.of(goalIdsByCampaignId.values())
                .flatMap(Collection::stream)
                .distinct()
                .toList();
        List<Goal> goals = retargetingGoalsPpcDictRepository.getMetrikaGoalsFromPpcDict(goalIds,
                false);

        Map<Long, Goal> goalsById = StreamEx.of(goals)
                .mapToEntry(GoalBase::getId, identity())
                .toMap();

        return EntryStream.of(goalIdsByCampaignId)
                .mapValues(ids -> getGoalsWithoutNull(goalsById, ids))
                .toMap();
    }

    private void addConversionLevelToGoals(Collection<Goal> goals) {
        List<Long> goalIdsWithoutConversionLevel = filterAndMapList(goals, goal -> goal.getConversionLevel() == null,
                Goal::getId);

        Map<Long, ConversionLevel> conversionLevelByGoalId = listToMap(
                retargetingGoalsPpcDictRepository.getMetrikaGoalsFromPpcDict(goalIdsWithoutConversionLevel, false),
                Goal::getId, GoalBase::getConversionLevel);

        for (Goal goal : goals) {
            ConversionLevel newConversionLevel = conversionLevelByGoalId.get(goal.getId());
            if (newConversionLevel != null) {
                goal.setConversionLevel(newConversionLevel);
            }
        }
    }

    private List<Goal> getGoalsWithoutNull(Map<Long, Goal> goalsById, List<Long> ids) {
        return StreamEx.of(ids)
                .map(goalsById::get)
                .nonNull()
                .toList();
    }

    public CreateCounterGoal createMetrikaGoals(Integer counterId, CreateCounterGoal goal) {
        var userTicket = retrieveUserTicket();
        return metrikaClient.createGoal(counterId, goal, userTicket);
    }

    public List<CreateCounterGoal> createMetrikaGoals(Integer counterId, List<CreateCounterGoal> goals) {
        var userTicket = retrieveUserTicket();

        List<CreateCounterGoal> createdGoalsInfo = new ArrayList<>();
        goals.forEach(goal -> {
            var createdGoal = metrikaClient.createGoal(counterId, goal, userTicket);
            createdGoalsInfo.add(createdGoal);
        });
        return createdGoalsInfo;
    }

    public List<CreateCounterGoalContainer> createMetrikaGoals(List<CreateCounterGoalContainer> goalsToCreate) {
        var userTicket = retrieveUserTicket();

        List<CreateCounterGoalContainer> createGoalsResult = new ArrayList<>();
        goalsToCreate.forEach(createGoalContainer -> {
            int counterId = createGoalContainer.getCounterId().intValue();
            var createdGoal = metrikaClient.createGoal(counterId,
                    createGoalContainer.getCounterGoal(), userTicket);
            createGoalsResult.add(new CreateCounterGoalContainer()
                    .withCounterId((long) counterId)
                    .withCounterGoal(createdGoal));
        });
        return createGoalsResult;
    }

    /**
     * Если {@code removeUnavailableGoals == false}, то недоступные цели берем только с заполненным счетчиком,
     * так как может быть случай, когда цель без доступа успели сохранить на кампанию, а потом цель стало
     * невозможно использовать (удалили или запретили через Метрику), но в базе она осталась.
     */
    private static Set<Goal> removeUnavailableGoalsIfNeeded(
            Collection<Goal> goals,
            boolean removeUnavailableGoals,
            Set<Long> availableGoalIdsForClient,
            Set<Long> ecommerceGoalIds) {
        Predicate<Goal> goalIsAvailablePredicate =
                goalIsAvailablePredicate(availableGoalIdsForClient, ecommerceGoalIds);
        if (removeUnavailableGoals) {
            return filterToSet(goals, goalIsAvailablePredicate);
        }
        return filterToSet(goals, goalIsAvailablePredicate.or(goal -> goal.getCounterId() != null));
    }

    private static Predicate<Goal> goalIsAvailablePredicate(Set<Long> availableGoalIdsForClient,
                                                            Set<Long> ecommerceGoalIds) {
        return goal -> availableGoalIdsForClient.contains(goal.getId()) ||
                availableGoalIdsForClient.contains(goal.getParentId()) ||
                ecommerceGoalIds.contains(goal.getId()) ||
                goal.getMobileAppId() != null;
    }

    /**
     * Метод возвращает id целей, к которым есть доступ у клиента
     */
    private Set<Long> getAvailableMetrikaAndMobileGoalIdsForClient(
            Map<Integer, List<CounterGoal>> massCountersGoalsFromMetrika) {
        return StreamEx.of(massCountersGoalsFromMetrika.values())
                .flatMap(Collection::stream)
                .map(CounterGoal::getId)
                .map(Integer::longValue)
                .append(MOBILE_GOAL_IDS)
                .toSet();
    }

    public List<Goal> createLalSegments(List<Long> parentIds) {
        return lalSegmentRepository.createLalSegments(parentIds);
    }

    /**
     * Получить список целей для их отрисовки, при выборе цели в стратегии, отсортированных в порядке рекомендации;
     * а так же получить id цели для автопроставления в стратегии (top1)
     */
    public GoalsSuggestion getGoalsSuggestion(ClientId clientId, Collection<Goal> goals) {
        if (!featureService.isEnabledForClientId(clientId, FeatureName.ENABLE_SUGGESTION_FOR_RECOMMENDED_GOALS) ||
                goals.isEmpty()) {
            return new GoalsSuggestion()
                    .withSortedGoalsToSuggestion(List.copyOf(goals))
                    .withTop1GoalId(null);
        }

        List<Goal> sortedGoalsForSuggestion = StreamEx.of(goals)
                .sorted(GOALS_FOR_SUGGESTION_COMPARATOR)
                .toList();

        return new GoalsSuggestion()
                .withSortedGoalsToSuggestion(sortedGoalsForSuggestion)
                .withTop1GoalId(getTop1GoalId(sortedGoalsForSuggestion));
    }

    /**
     * Возвращает топ1 цель для автоподстановки в стратегию.
     * В топ1 попадает первая из отсортированного списка рекомендуемых целей ({@link GoalsForSuggestionComparator}),
     * с конверсиями >= 20. При этом для ACTION или URL учитываются только perfect цели
     */
    @Nullable
    private static Long getTop1GoalId(List<Goal> sortedGoalsForSuggestion) {
        if (sortedGoalsForSuggestion.isEmpty()) {
            return null;
        }

        Goal firstGoal = sortedGoalsForSuggestion.get(0);
        if (GoalsForSuggestionComparator.getGoalOrderDescription(firstGoal).compareTo(
                GoalsForSuggestionComparator.GoalOrderDescription.GOAL_WITH_PERFECT_CONVERSION_LEVEL) <= 0) {
            return firstGoal.getId();
        }
        return null;
    }

    public List<Goal> getGoalsByCounterId(Integer counterId) {
        List<CounterGoal> goals = metrikaClient.getMassCountersGoalsFromMetrika(Set.of(counterId)).getOrDefault(
                counterId, List.of());
        return StreamEx.of(GoalConverter.fromCounterGoals(goals))
                .peek(goal -> goal.setCounterId(counterId))
                .toList();
    }

}
