package ru.yandex.direct.core.entity.metrika.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaSegmentPreset;
import ru.yandex.direct.core.entity.retargeting.model.RawMetrikaSegmentPreset;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingGoalsPpcDictRepository;
import ru.yandex.direct.core.util.ReflectionTranslator;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.metrika.client.model.response.Counter;
import ru.yandex.direct.metrika.client.model.response.Segment;
import ru.yandex.direct.metrika.client.model.response.UserCounters;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static ru.yandex.direct.common.db.PpcPropertyNames.METRIKA_SEGMENT_PRESETS_COUNTERS_LIMIT;
import static ru.yandex.direct.core.security.tvm.TvmUtils.retrieveUserTicket;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class MetrikaSegmentService {
    private static final Logger logger = LoggerFactory.getLogger(MetrikaSegmentService.class);

    public enum SegmentType {
        // только «неотказная аудитория сайта»
        NOT_BOUNCE,
        // совершившие покупку
        ECOM_PURCHASE,
        // брошенные корзины
        ECOM_ABANDONED_CART,
        // смотрели товары, но не купили
        ECOM_VIEWED_WITHOUT_PURCHASE
    }

    public static final String NOT_BOUNCE_EXPRESSION = "ym:s:bounce=='No'";
    public static final String ECOM_PURCHASE_EXPRESSION =
            "EXISTS ym:u:userID WITH (ym:u:productsBought>0)";
    public static final String ECOM_ABANDONED_CART_EXPRESSION =
            "EXISTS ym:u:userID WITH ((ym:u:productsAdded>0) AND (ym:u:productsBought==0))";
    public static final String ECOM_VIEWED_WITHOUT_PURCHASE_EXPRESSION =
            "EXISTS ym:u:userID WITH ((ym:u:productsBought==0) AND (ym:u:productViews>0))";

    private static final Map<SegmentType, String> SEGMENT_PRESET_EXPRESSION_BY_TYPE = Map.of(
            SegmentType.NOT_BOUNCE, NOT_BOUNCE_EXPRESSION
    );

    private static final Map<SegmentType, String> SEGMENT_CREATION_EXPRESSION_BY_TYPE = Map.of(
            SegmentType.ECOM_PURCHASE, ECOM_PURCHASE_EXPRESSION,
            SegmentType.ECOM_ABANDONED_CART, ECOM_ABANDONED_CART_EXPRESSION,
            SegmentType.ECOM_VIEWED_WITHOUT_PURCHASE, ECOM_VIEWED_WITHOUT_PURCHASE_EXPRESSION
    );

    // переводы для ключей находятся в MetrikaSegmentPresetTranslations
    protected static final Map<SegmentType, String> SEGMENT_TANKER_NAME_BY_TYPE = Map.of(
            SegmentType.ECOM_PURCHASE, "ecomPurchase",
            SegmentType.ECOM_ABANDONED_CART, "ecomAbandonedCart",
            SegmentType.ECOM_VIEWED_WITHOUT_PURCHASE, "ecomViewedWithoutPurchase"
    );

    //TODO make one map an invert of another
    private static final Map<String, SegmentType> SEGMENT_TYPE_BY_FILTER_EXPRESSION = Map.of(
            NOT_BOUNCE_EXPRESSION, SegmentType.NOT_BOUNCE,
            String.format("(%s)", NOT_BOUNCE_EXPRESSION), SegmentType.NOT_BOUNCE,
            ECOM_PURCHASE_EXPRESSION, SegmentType.ECOM_PURCHASE, //TODO check if () variant is needed
            ECOM_ABANDONED_CART_EXPRESSION, SegmentType.ECOM_ABANDONED_CART,
            ECOM_VIEWED_WITHOUT_PURCHASE_EXPRESSION, SegmentType.ECOM_VIEWED_WITHOUT_PURCHASE
    );

    private static final Map<SegmentType, Set<String>> SEGMENT_FILTER_EXPRESSIONS_BY_TYPE = Map.of(
            SegmentType.NOT_BOUNCE, Set.of(NOT_BOUNCE_EXPRESSION, String.format("(%s)", NOT_BOUNCE_EXPRESSION)),
            SegmentType.ECOM_PURCHASE, Set.of(ECOM_PURCHASE_EXPRESSION),
            SegmentType.ECOM_ABANDONED_CART, Set.of(ECOM_ABANDONED_CART_EXPRESSION),
            SegmentType.ECOM_VIEWED_WITHOUT_PURCHASE, Set.of(ECOM_VIEWED_WITHOUT_PURCHASE_EXPRESSION)
    );

    private static final Set<SegmentType> ECOM_SEGMENTS = Set.of(SegmentType.ECOM_PURCHASE,
            SegmentType.ECOM_ABANDONED_CART, SegmentType.ECOM_VIEWED_WITHOUT_PURCHASE);

    /**
     * Значение по умолчанию для ограничения по количеству счётчиков
     * <p>
     * Строить пресеты дорого, потому что при N доступных пользователю счётчиках нам
     * надо сделать {@code N + 1} запрорсов к метрике — один запрос за счётчиками
     * и по запросу для каждого счётчика, чтобы получить соответствующие сегменты.
     * Поэтому ограничиваем максимальное количество счётчиков, по которым строим
     * пресеты. Делаем это либо с помощью {@code ppc_properties}, либо данным значением.
     */
    private static final Integer DEFAULT_METRIKA_SEGMENT_COUNTERS_LIMIT = 100;

    private final RbacService rbacService;
    private final MetrikaClient metrikaClient;
    private final ReflectionTranslator reflectionTranslator;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final RetargetingGoalsPpcDictRepository retargetingGoalsPpcDictRepository;

    public MetrikaSegmentService(RbacService rbacService,
                                 MetrikaClient metrikaClient,
                                 ReflectionTranslator reflectionTranslator,
                                 PpcPropertiesSupport ppcPropertiesSupport,
                                 RetargetingGoalsPpcDictRepository retargetingGoalsPpcDictRepository) {
        this.rbacService = rbacService;
        this.metrikaClient = metrikaClient;
        this.reflectionTranslator = reflectionTranslator;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.retargetingGoalsPpcDictRepository = retargetingGoalsPpcDictRepository;
    }

    public List<MetrikaSegmentPreset> getSegmentPresets(ClientId clientId) {
        var userTicket = retrieveUserTicket();
        var rawPresets = retargetingGoalsPpcDictRepository.getSegmentPresets();
        List<MetrikaSegmentPreset> result = new ArrayList<>();
        getFilteredEditableCounters(clientId, userTicket).forEach(counter -> {
            var existingSegmentsNames = mapList(metrikaClient.getSegments(counter.getId(), userTicket),
                    segment -> segment.getName().toLowerCase());
            StreamEx.of(rawPresets)
                    .map(preset -> {
                        var presetName = translatePresetName(preset.getTankerNameKey());
                        if (existingSegmentsNames.contains(presetName.toLowerCase())) {
                            return null;
                        }
                        return toMetrikaSegmentPreset(counter, preset.getPresetId(), presetName);
                    })
                    .nonNull()
                    .forEach(result::add);
        });
        return result;
    }

    public List<Segment> createMetrikaSegmentsByPresets(Map<Integer, List<MetrikaSegmentPreset>> presetsByCounter) {
        var userTicket = retrieveUserTicket();

        var presetsByIds = listToMap(retargetingGoalsPpcDictRepository.getSegmentPresets(),
                RawMetrikaSegmentPreset::getPresetId, Function.identity());
        List<Segment> createdSegments = new ArrayList<>();
        Map<Integer, String> segmentDataToSave = new HashMap<>();
        presetsByCounter.forEach((counterId, presets) ->
                presets.forEach(preset -> {
                    var dbPreset = presetsByIds.get(preset.getPresetId());
                    var segment = metrikaClient.createSegment(counterId,
                            translatePresetName(dbPreset.getTankerNameKey()), dbPreset.getExpression(),
                            userTicket);
                    segmentDataToSave.put(segment.getId(), preset.getCounterOwner());
                    createdSegments.add(segment);
                })
        );
        retargetingGoalsPpcDictRepository.addMetrikaSegmentsCreatedByPresets(segmentDataToSave);
        return createdSegments;
    }

    /**
     * Получение или создание сегментов вида:
     * <li>только «неотказная аудитория сайта»</li>
     *
     * @param clientId   id клиента
     * @param counterIds id счетчиков, для которых требуется вернуть сегменты
     * @return id сегментов
     */
    public Set<Long> getOrCreateNotBounceSegmentIds(ClientId clientId, List<Integer> counterIds) {
        List<Segment> notBounceSegments = getOrCreatePresetSegments(clientId, counterIds, SegmentType.NOT_BOUNCE);
        return StreamEx.of(notBounceSegments)
                .map(Segment::getId)
                .map(Integer::longValue)
                .toSet();
    }

    /**
     * Получение из Метрики сегментов указанного типа, если они уже существуют,
     * либо создание таких сегментов в Метрике на основе пресета из ppcdict базы.
     * <p>
     * Если у клиента нет доступа на редактирование счетчика, то сегмент для него не будет создан,
     * но будет возвращен в случае его существования.
     *
     * @param clientId    id клиента
     * @param counterIds  id счетчиков, для которых требуется вернуть сегменты
     * @param segmentType тип сегмента
     * @return список сегментов
     */
    public List<Segment> getOrCreatePresetSegments(ClientId clientId, List<Integer> counterIds,
                                                   SegmentType segmentType) {
        Set<Integer> uniqueCounterIds = listToSet(counterIds);
        List<Segment> existingSegments = getExistingSegments(uniqueCounterIds, segmentType);
        Set<Integer> counterIdsWithSegment = listToSet(existingSegments, Segment::getCounterId);
        Set<Integer> counterIdsWithoutSegment = Sets.difference(uniqueCounterIds, counterIdsWithSegment);
        if (counterIdsWithoutSegment.isEmpty()) {
            return existingSegments;
        }

        var segmentPresetExpression = SEGMENT_PRESET_EXPRESSION_BY_TYPE.get(segmentType);
        Optional<RawMetrikaSegmentPreset> specificSegmentPreset =
                StreamEx.of(retargetingGoalsPpcDictRepository.getSegmentPresets())
                        .filter(s -> segmentPresetExpression.equals(s.getExpression()))
                        .findFirst();
        if (specificSegmentPreset.isEmpty()) {
            logger.error("Segment preset not found in ppcdict by expression='{}'", segmentPresetExpression);
            return existingSegments;
        }

        String userTicket = retrieveUserTicket();
        Map<Integer, Counter> editableCounters = StreamEx.of(getFilteredEditableCounters(clientId, userTicket))
                .mapToEntry(Counter::getId, Function.identity())
                .filterKeys(counterIdsWithoutSegment::contains)
                .toMap();
        Set<Integer> notEditableCounterIds = Sets.difference(counterIdsWithoutSegment, editableCounters.keySet());
        if (!notEditableCounterIds.isEmpty()) {
            logger.info("Client {} does not have permissions to create segment for counters {}",
                    clientId, notEditableCounterIds);
        }
        if (editableCounters.isEmpty()) {
            return existingSegments;
        }
        logger.info("Creating {} segments for counters {}", segmentType, editableCounters.keySet());
        List<Segment> createdSegments = createPresetSegments(specificSegmentPreset.get(), editableCounters);
        logger.info("Created {} segments", createdSegments.size());
        return StreamEx.of(existingSegments)
                .append(createdSegments)
                .toList();
    }

    /**
     * Получение из Метрики сегментов указанных типов, если они уже существуют,
     * либо создание таких сегментов в Метрике на основе пресетов из SEGMENT_CREATION_EXPRESSION_BY_TYPE.
     * <p>
     * Если у клиента нет доступа на редактирование счетчика, то сегмент для него не будет создан,
     * но будет возвращен в случае его существования.
     *
     * @param clientId     id клиента
     * @param counterIds   id счетчиков, для которых требуется вернуть сегменты
     * @param segmentTypes типы сегмента
     * @return словарь id сегментов по типам
     */
    public Map<SegmentType, Set<Long>> getOrCreateSegmentIdsBySegmentType(ClientId clientId, List<Integer> counterIds,
                                                                          Set<SegmentType> segmentTypes) {
        var segmentsBySegmentType = getOrCreateSegmentsBySegmentType(clientId, counterIds, segmentTypes);
        return EntryStream.of(segmentsBySegmentType)
                .mapValues(segments -> segments.stream()
                        .map(Segment::getId)
                        .map(Integer::longValue)
                        .collect(Collectors.toSet()))
                .toMap();
    }

    /**
     * Получение из Метрики сегментов указанных типов, если они уже существуют,
     * либо создание таких сегментов в Метрике на основе пресетов из SEGMENT_CREATION_EXPRESSION_BY_TYPE.
     * <p>
     * Если у клиента нет доступа на редактирование счетчика, то сегмент для него не будет создан,
     * но будет возвращен в случае его существования.
     *
     * @param clientId     id клиента
     * @param counterIds   id счетчиков, для которых требуется вернуть сегменты
     * @param segmentTypes типы сегмента
     * @return словарь сегментов по типам
     */
    protected Map<SegmentType, List<Segment>> getOrCreateSegmentsBySegmentType(ClientId clientId,
                                                                               List<Integer> counterIds,
                                                                               Set<SegmentType> segmentTypes) {
        Set<Integer> uniqueCounterIds = listToSet(counterIds);
        var existingSegmentsBySegmentType =
                getExistingSegmentsBySegmentType(uniqueCounterIds, segmentTypes);

        String userTicket = retrieveUserTicket();
        var allClientEditableCounters = getFilteredEditableCounters(clientId, userTicket);

        for (var segmentType : segmentTypes) {
            var existingSegments = new ArrayList(existingSegmentsBySegmentType.getOrDefault(segmentType, List.of()));
            Set<Integer> counterIdsWithSegment = listToSet(existingSegments, Segment::getCounterId);
            Set<Integer> counterIdsWithoutSegment = Sets.difference(uniqueCounterIds, counterIdsWithSegment);
            if (counterIdsWithoutSegment.isEmpty()) {
                continue;
            }

            // проверка нужна только в том случае, если сегмент данного типа нужно создавать (не для всех счетчиков
            // существует данный тип сегмента), поэтому переносить ее выше по циклу не стоит
            if (!SEGMENT_CREATION_EXPRESSION_BY_TYPE.containsKey(segmentType)) {
                logger.error("Segment preset not found by type='{}'", segmentType);
                continue;
            }

            Map<Integer, Counter> editableCountersById = StreamEx.of(allClientEditableCounters)
                    .mapToEntry(Counter::getId, Function.identity())
                    .filterKeys(counterIdsWithoutSegment::contains)
                    .toMap();
            Set<Integer> notEditableCounterIds =
                    Sets.difference(counterIdsWithoutSegment, editableCountersById.keySet());
            if (!notEditableCounterIds.isEmpty()) {
                logger.info("Client {} does not have permissions to create segment for counters {}",
                        clientId, notEditableCounterIds);
            }
            if (editableCountersById.isEmpty()) {
                continue;
            }

            logger.info("Creating {} segments for counters {}", segmentType, editableCountersById.keySet());
            List<Segment> createdSegments = createSegments(segmentType, editableCountersById);
            logger.info("Created {} segments", createdSegments.size());

            existingSegments.addAll(createdSegments);
            existingSegmentsBySegmentType.put(segmentType, existingSegments);
        }

        return existingSegmentsBySegmentType;
    }

    /**
     * Проверка наличия или возможности создания хотя бы одного сегмента указанного типа
     *
     * @param clientId   id клиента
     * @param counterIds id счетчиков, для которых требуется проверить сегменты
     * @return доступность сегментов для данных счетчиков
     */
    public boolean checkAvailabilityOfPresetSegments(ClientId clientId, List<Integer> counterIds,
                                                     SegmentType segmentType) {
        Set<Integer> uniqueCounterIds = listToSet(counterIds);
        List<Segment> existingSegments;
        try {
            existingSegments = getExistingSegments(uniqueCounterIds, segmentType);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.error("Got an exception when querying for metrika segments for client " + clientId
                    + " and counters " + uniqueCounterIds, e);
            return false;
        }

        if (!existingSegments.isEmpty()) {
            return true;
        } else {
            // проверка, можно ли создать нужные сегменты
            var segmentPresetExpression = SEGMENT_PRESET_EXPRESSION_BY_TYPE.get(segmentType);
            Optional<RawMetrikaSegmentPreset> specificSegmentPreset =
                    StreamEx.of(retargetingGoalsPpcDictRepository.getSegmentPresets())
                            .filter(s -> segmentPresetExpression.equals(s.getExpression()))
                            .findFirst();
            if (specificSegmentPreset.isEmpty()) {
                return false;
            }

            String userTicket = retrieveUserTicket();
            Map<Integer, Counter> editableCounters = StreamEx.of(getFilteredEditableCounters(clientId, userTicket))
                    .mapToEntry(Counter::getId, Function.identity())
                    .filterKeys(uniqueCounterIds::contains)
                    .toMap();

            return !editableCounters.isEmpty();
        }
    }

    public List<Segment> getExistingSegments(Set<Integer> counterIds, SegmentType segmentType) {
        var filterSegmentExpressions = SEGMENT_FILTER_EXPRESSIONS_BY_TYPE.get(segmentType);
        return StreamEx.of(counterIds)
                .map(counterId -> metrikaClient.getSegments(counterId, null))
                .map(segments -> StreamEx.of(segments)
                        .filter(s -> filterSegmentExpressions.contains(s.getExpression()))
                        .findFirst()
                        .orElse(null))
                .nonNull()
                .toList();
    }

    /**
     * Проверка наличия или возможности создания типов еком-сегментов
     *
     * @param clientId       id клиента
     * @param ecomCounterIds id счетчиков, для которых требуется проверить еком-сегменты
     * @return еком-сегменты, которые доступны для данных счетчиков
     */
    public Set<SegmentType> getAvailableEcomSegments(ClientId clientId, List<Integer> ecomCounterIds) {
        Set<Integer> uniqueCounterIds = listToSet(ecomCounterIds);
        Map<SegmentType, List<Segment>> existingEcomSegmentsBySegmentType;
        try {
            existingEcomSegmentsBySegmentType = getExistingSegmentsBySegmentType(uniqueCounterIds, ECOM_SEGMENTS);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.error("Got an exception when querying for metrika segments for client " + clientId
                    + " and counters " + uniqueCounterIds, e);
            return Set.of();
        }

        if (existingEcomSegmentsBySegmentType.keySet().containsAll(ECOM_SEGMENTS)) {
            return ECOM_SEGMENTS;
        } else {
            // проверка, можно ли создать нужные сегменты
            String userTicket = retrieveUserTicket();
            Map<Integer, Counter> editableCounters = StreamEx.of(getFilteredEditableCounters(clientId, userTicket))
                    .mapToEntry(Counter::getId, Function.identity())
                    .filterKeys(uniqueCounterIds::contains)
                    .toMap();

            return !editableCounters.isEmpty()
                    ? ECOM_SEGMENTS
                    : existingEcomSegmentsBySegmentType.keySet();
        }
    }

    /**
     * Получение существующих сегментов определенных типов у счетчиков.
     * Для типов должны быть заданы соответствия условиям в SEGMENT_FILTER_EXPRESSIONS_BY_TYPE
     *
     * @param counterIds   id счетчиков
     * @param segmentTypes типы сегментов
     * @return словарь сегментов по типам
     */
    private Map<SegmentType, List<Segment>> getExistingSegmentsBySegmentType(Set<Integer> counterIds,
                                                                             Set<SegmentType> segmentTypes) {
        var filterExpressionsBySegmentType = listToMap(segmentTypes, type -> type,
                SEGMENT_FILTER_EXPRESSIONS_BY_TYPE::get);
        var allFilterExpressions = filterExpressionsBySegmentType.values().stream()
                .flatMap(Set::stream)
                .collect(Collectors.toList());
        return StreamEx.of(counterIds)
                .map(counterId -> metrikaClient.getSegments(counterId, null))
                .map(segments -> StreamEx.of(segments)
                        .filter(s -> allFilterExpressions.contains(s.getExpression())
                                && SEGMENT_TYPE_BY_FILTER_EXPRESSION.containsKey(s.getExpression()))
                        .collect(Collectors.toList()))
                .flatMap(List::stream)
                .groupingBy(s -> SEGMENT_TYPE_BY_FILTER_EXPRESSION.get(s.getExpression()));
    }

    private List<Segment> createPresetSegments(RawMetrikaSegmentPreset segmentPreset,
                                               Map<Integer, Counter> counters) {
        String presetName = translatePresetName(segmentPreset.getTankerNameKey());
        Map<Integer, List<MetrikaSegmentPreset>> presets = EntryStream.of(counters)
                .mapValues(counter ->
                        toMetrikaSegmentPreset(counter, segmentPreset.getPresetId(), presetName))
                .mapValues(List::of)
                .toMap();
        return createMetrikaSegmentsByPresets(presets);
    }

    private List<Segment> createSegments(SegmentType segmentType,
                                         Map<Integer, Counter> countersById) {
        var userTicket = retrieveUserTicket();

        var segmentName = translatePresetName(SEGMENT_TANKER_NAME_BY_TYPE.get(segmentType));
        var segmentExpression = SEGMENT_CREATION_EXPRESSION_BY_TYPE.get(segmentType);

        List<Segment> createdSegments = new ArrayList<>();
        Map<Integer, String> segmentDataToSave = new HashMap<>();
        countersById.forEach((counterId, counter) -> {
            var segment = metrikaClient.createSegment(counterId, segmentName, segmentExpression, userTicket);
            segmentDataToSave.put(segment.getId(), counter.getOwnerLogin());
            createdSegments.add(segment);
        });
        retargetingGoalsPpcDictRepository.addMetrikaSegmentsCreatedByPresets(segmentDataToSave);

        return createdSegments;
    }

    /**
     * Возвращает все счётчики клиента и его представителей, на которые у текущего оператора есть
     * права на редактирование.
     * <p>
     * Возвращает счётчиков не более, чем {@link PpcPropertyNames#METRIKA_SEGMENT_PRESETS_COUNTERS_LIMIT},
     * или, если это свойство не установлено, не более, чем {@link #DEFAULT_METRIKA_SEGMENT_COUNTERS_LIMIT}.
     *
     * @param clientId   {@link ClientId Идентификатор клиента}
     * @param userTicket TVM2 user тикет текущего пользователя (оператора)
     * @return Список {@link Counter счётчиков}
     */
    private List<Counter> getFilteredEditableCounters(ClientId clientId, String userTicket) {
        var uids = rbacService.getClientRepresentativesUids(clientId);
        var editableCounters = metrikaClient.getEditableCounters(userTicket);

        var counterIds = mapList(editableCounters, c -> (long) c.getId());
        var clientCountersSet = StreamEx.of(metrikaClient
                        .getUsersCountersNum2(List.copyOf(uids), counterIds).getUsers())
                .map(UserCounters::getCounterIds)
                .flatMap(List::stream)
                .toSet();

        var limit = ppcPropertiesSupport.get(METRIKA_SEGMENT_PRESETS_COUNTERS_LIMIT)
                .getOrDefault(DEFAULT_METRIKA_SEGMENT_COUNTERS_LIMIT);
        var allCounters = StreamEx.of(editableCounters)
                .filter(counter -> clientCountersSet.contains(counter.getId()))
                .sortedBy(Counter::getId)
                .toList();
        if (allCounters.size() > limit) {
            logger.warn("Client {} has {} counters, which exceeds the {} limit", clientId, allCounters.size(), limit);
            return allCounters.stream()
                    .limit(limit)
                    .collect(Collectors.toList());
        } else {
            return allCounters;
        }
    }

    private static MetrikaSegmentPreset toMetrikaSegmentPreset(Counter counter, Integer presetId, String presetName) {
        return new MetrikaSegmentPreset()
                .withCounterId(counter.getId())
                .withPresetId(presetId)
                .withName(presetName)
                .withCounterOwner(counter.getOwnerLogin())
                .withDomain(counter.getDomain());
    }

    private String translatePresetName(String name) {
        return reflectionTranslator.translate(name, MetrikaSegmentPresetTranslations.INSTANCE);
    }
}
