package ru.yandex.direct.core.entity.metrika.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.function.Function;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppService;
import ru.yandex.direct.core.entity.mobilegoals.MobileAppGoalsService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.rbac.RbacClientsRelations;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.utils.FunctionalUtils;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.core.entity.retargeting.model.Goal.MOBILE_GOAL_IDS;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис отдающий мобильные цели
 * Сейчас есть два вида таких целей:
 * - фиксированный набор на все приложения из MobileGoalsPermissionService
 * - сгенерированные цели под каждое приложение из таблицы mobile_app_goals_external_tracker
 */
@Service
public class MobileGoalsService {

    // дополнительные доступные цели для автобюджетных кампаний внутренней рекламы, рекламирующих мобильные приложения
    private static final Map<Long, String> METRIKA_GOALS_FOR_MOBILE_INTERNAL_AUTOBUDGET_CAMPAIGN = Map.ofEntries(
            Map.entry(3L, "Оптимизация по установкам мобильного приложения")
    );

    @Autowired
    private MobileGoalsPermissionService mobileGoalsPermissionService;
    @Autowired
    private RbacService rbacService;
    @Autowired
    private RbacClientsRelations rbacClientsRelations;
    @Autowired
    private MetrikaCampaignRepository metrikaCampaignRepository;
    @Autowired
    private FeatureService featureService;
    @Autowired
    private MobileAppService mobileAppService;
    @Autowired
    private MobileAppGoalsService mobileAppGoalsService;

    /**
     * Возвращает словарь целей по кампаниям
     *
     * @param shard                    шард
     * @param operatorUid              uid оператора
     * @param clientId                 id клиента
     * @param campaignIdToCampaignType мапа ид кампаний на их типы
     * @param isUsedMobileGoalsNeeded  нужны ли используемые цели
     * @return список целей
     */
    public Map<Long, List<Goal>> getMobileGoalsByCampaignId(
            int shard, Long operatorUid, ClientId clientId,
            Map<Long, CampaignType> campaignIdToCampaignType,
            boolean isUsedMobileGoalsNeeded) {
        var typeToCampaignIds = toCampaignIdsByCampaignType(campaignIdToCampaignType);

        Map<Long, List<Goal>> goalsByTextCampaignId = getMobileGoalsForTgoByCampaignId(shard, operatorUid, clientId,
                typeToCampaignIds.getOrDefault(CampaignType.TEXT, emptyList()), isUsedMobileGoalsNeeded);

        List<Goal> inAppMobileGoals = getAllAvailableInAppMobileGoals(clientId);
        Map<Long, List<Goal>> goalsByDynamicCampaignId = FunctionalUtils.listToMap(
                typeToCampaignIds.getOrDefault(CampaignType.DYNAMIC, List.of()), Function.identity(),
                (id) -> inAppMobileGoals
        );

        Map<Long, List<Goal>> goalsBySmartCampaignId = FunctionalUtils.listToMap(
                typeToCampaignIds.getOrDefault(CampaignType.PERFORMANCE, List.of()), Function.identity(),
                (id) -> inAppMobileGoals
        );

        Map<Long, List<Goal>> goalsByInternalAutobudgetCampaignId = FunctionalUtils.listToMap(
                typeToCampaignIds.getOrDefault(CampaignType.INTERNAL_AUTOBUDGET, emptyList()),
                Function.identity(), (id) -> getMobileGoalsForInternalAutobudgetCampaign());

        Map<Long, List<Goal>> goalsForMobileContent = FunctionalUtils.listToMap(
                typeToCampaignIds.getOrDefault(CampaignType.MOBILE_CONTENT, List.of()), Function.identity(),
                (id) -> getAllowedFixedMobileGoals(operatorUid, clientId)
        );

        Map<Long, List<Goal>> goalsByCampaignId = new HashMap<>();
        goalsByCampaignId.putAll(goalsByTextCampaignId);
        goalsByCampaignId.putAll(goalsByInternalAutobudgetCampaignId);
        goalsByCampaignId.putAll(goalsForMobileContent);
        goalsByCampaignId.putAll(goalsByDynamicCampaignId);
        goalsByCampaignId.putAll(goalsBySmartCampaignId);

        return goalsByCampaignId;
    }

    /**
     * Возвращает словарь id целей по кампаниям
     *
     * @param shard                    шард
     * @param operatorUid              uid оператора
     * @param clientId                 id клиента
     * @param campaignIdToCampaignType мапа ид кампаний на их типы
     * @param isUsedMobileGoalsNeeded  нужны ли используемые цели
     * @return список id целей
     */
    public Map<Long, List<Long>> getMobileGoalIdsByCampaignId(int shard, Long operatorUid, ClientId clientId,
                                                              Map<Long, CampaignType> campaignIdToCampaignType,
                                                              boolean isUsedMobileGoalsNeeded) {
        return getMobileGoalsByCampaignId(shard, operatorUid, clientId,
                campaignIdToCampaignType, isUsedMobileGoalsNeeded)
                .entrySet()
                .stream()
                .collect(toMap(Map.Entry::getKey, e -> mapList(e.getValue(), GoalBase::getId)));
    }

    /**
     * Возвращает словарь целей по текстовым кампаниям.
     *
     * @param shard                   шард
     * @param operatorUid             uid оператора
     * @param clientId                id клиента
     * @param campaignIds             список id кампаний, для которых нужны мобильные цели
     * @param isUsedMobileGoalsNeeded нужны ли используемые цели
     * @return список целей
     */
    public Map<Long, List<Goal>> getMobileGoalsForTgoByCampaignId(int shard, Long operatorUid, ClientId clientId,
                                                                  Collection<Long> campaignIds,
                                                                  boolean isUsedMobileGoalsNeeded) {
        var mobileGoalsByCampaignId = new HashMap<Long, List<Goal>>(campaignIds.size());
        Set<Goal> allowedMobileGoals = getAllowedMobileGoalsForTgo(operatorUid, clientId);
        Set<Goal> allFixedMobileGoals = featureService.isEnabledForClientId(clientId,
                FeatureName.MOBILE_APP_GOALS_FOR_TEXT_CAMPAIGN_STRATEGY_ENABLED)
                ? mobileGoalsPermissionService.getMobileGoalsWithPermissions().keySet()
                : Set.of();

        Map<Long, Set<Long>> usedGoalIdsByCampaignId
                = getUsedMobileGoalIds(shard, clientId.asLong(), campaignIds, isUsedMobileGoalsNeeded);

        for (var campaignId : campaignIds) {
            Set<Long> usedGoalIds = usedGoalIdsByCampaignId.getOrDefault(campaignId, Set.of());
            Set<Goal> mobileGoals = filterToSet(allFixedMobileGoals, g -> usedGoalIds.contains(g.getId()));
            mobileGoals.addAll(allowedMobileGoals);
            mobileGoalsByCampaignId.put(campaignId, new ArrayList<>(mobileGoals));
        }
        return mobileGoalsByCampaignId;
    }

    /**
     * Возвращает словарь используемых мобильных целей (ключевых, стратегий или дрф) по кампаниям.
     *
     * @param shard                   шард
     * @param clientId                id клиента
     * @param campaignIds             список id кампаний
     * @param isUsedMobileGoalsNeeded нужны ли используемые цели
     * @return Используемые мобильные цели
     */
    private Map<Long, Set<Long>> getUsedMobileGoalIds(int shard, Long clientId, Collection<Long> campaignIds,
                                                      boolean isUsedMobileGoalsNeeded) {
        if (!isUsedMobileGoalsNeeded || campaignIds == null || campaignIds.isEmpty()) {
            return Map.of();
        }

        var usedGoalIdsByCampaignId = metrikaCampaignRepository.getUsedGoalIdsByCampaignId(shard,
                clientId, campaignIds);
        return EntryStream.of(usedGoalIdsByCampaignId)
                .nonNullValues()
                .mapValues(goalIds -> StreamEx.of(goalIds)
                        .filter(MOBILE_GOAL_IDS::contains)
                        .toSet())
                .toMap();
    }

    private List<Goal> getMobileGoalsForNewCampaign(Long operatorUid,
                                                    ClientId clientId,
                                                    CampaignType campaignType) {
        if (campaignType == CampaignType.MOBILE_CONTENT) {
            return getAllowedFixedMobileGoals(operatorUid, clientId);
        } else if (campaignType == CampaignType.TEXT) {
            return new ArrayList<>(getAllowedMobileGoalsForTgo(operatorUid, clientId));
        } else if (campaignType == CampaignType.DYNAMIC || campaignType == CampaignType.PERFORMANCE) {
            return new ArrayList<>(getAllAvailableInAppMobileGoals(clientId));
        } else if (campaignType == CampaignType.INTERNAL_AUTOBUDGET) {
            return getMobileGoalsForInternalAutobudgetCampaign();
        }

        return emptyList();
    }

    /**
     * Возвращает доступные оператору мобильные цели в соответствии с переданным фильтром мобильных целей.
     *
     * @param operatorUid       uid оператора
     * @param clientId          id клиента
     * @param mobileGoalsFilter фильтр для получения доступных мобильных целей
     * @return Список целей
     */
    public List<Goal> getMobileGoalsForFilter(Long operatorUid,
                                              ClientId clientId,
                                              MobileGoalsFilter mobileGoalsFilter) {
        if (mobileGoalsFilter instanceof ForCampaignType) {
            return getMobileGoalsForNewCampaign(operatorUid, clientId,
                    ((ForCampaignType) mobileGoalsFilter).getCampaignType());
        } else if (mobileGoalsFilter instanceof ForStrategy) {
            return getMobileGoalsForNewStrategy(operatorUid, clientId);
        }
        return emptyList();
    }

    public List<Goal> getMobileGoalsForNewStrategy(Long operatorUid, ClientId clientId) {
        var mobileGoals = getAllowedFixedMobileGoals(operatorUid, clientId);
        mobileGoals.addAll(getAllAvailableInAppMobileGoals(clientId));
        return mobileGoals;
    }

    private Set<Goal> getAllowedMobileGoalsForTgo(Long operatorUid, ClientId clientId) {
        Set<Goal> allowedMobileGoals = new HashSet<>(getAllAvailableInAppMobileGoals(clientId));
        allowedMobileGoals.addAll(getAllowedFixedMobileGoalsWhenFeatureIsEnabled(operatorUid, clientId));
        return allowedMobileGoals;
    }

    /**
     * Получить разрешенные фиксированные мобильные цели для ТГО кампании при включенной фиче
     */
    public List<Goal> getAllowedFixedMobileGoalsWhenFeatureIsEnabled(Long operatorUid, ClientId clientId) {
        return featureService.isEnabledForClientId(clientId,
                FeatureName.MOBILE_APP_GOALS_FOR_TEXT_CAMPAIGN_STRATEGY_ENABLED)
                ? getAllowedFixedMobileGoals(operatorUid, clientId,
                mobileGoalsPermissionService.getMobileGoalsWithPermissions())
                : emptyList();
    }

    private List<Goal> getAllowedFixedMobileGoals(Long operatorUid, ClientId clientId) {
        return getAllowedFixedMobileGoals(operatorUid, clientId,
                mobileGoalsPermissionService.getMobileGoalsWithPermissions());
    }

    private List<Goal> getAllowedFixedMobileGoals(Long operatorUid, ClientId clientId,
                                                  Map<Goal, BiPredicate<RbacRole, ClientId>> getMobileGoalsWithPermissions) {
        var operatorRole = rbacService.getUidRole(operatorUid);
        return EntryStream.of(getMobileGoalsWithPermissions)
                .filterValues(permissions -> permissions.test(operatorRole, clientId))
                .keys()
                .toList();
    }

    /**
     * Получить мобильные in-app цели доступные клиенту -- это цели приложений клиента + цели, к которым ему
     * может предоставить доступ другой клиент
     */
    public List<Goal> getAllAvailableInAppMobileGoals(ClientId clientId) {
        if (!featureService.isEnabledForClientId(clientId, FeatureName.IN_APP_MOBILE_TARGETING)) {
            return emptyList();
        }
        return StreamEx.of(getClientInAppMobileGoals(clientId))
                .append(getSharedInAppMobileGoals(clientId))
                .toList();
    }

    /**
     * Получить мобильные in-app цели другого клиента, к которым предоставлен доступ
     */
    private List<Goal> getSharedInAppMobileGoals(ClientId clientId) {
        List<ClientId> owners = rbacClientsRelations.getOwnersOfMobileGoals(clientId);
        // вообще, в owners должен быть ровно один элемент -- сейчас это гарантируется валидацией ручек по добавлению
        // доступа к мобильным целям клиента,
        // но пусть здесь будет сформулировано получение целей для более общего случая
        return StreamEx.of(owners).flatCollection(this::getClientInAppMobileGoals).toList();
    }

    /**
     * Получить мобильные in-app цели приложений клиента
     */
    private List<Goal> getClientInAppMobileGoals(ClientId clientId) {
        List<MobileApp> mobileApps = mobileAppService.getMobileApps(clientId);
        List<Goal> commonMobileGoals = mobileAppGoalsService.getGoalsByApps(clientId, mobileApps);

        Map<Long, MobileApp> mobileAppsById = listToMap(mobileApps, MobileApp::getId);
        commonMobileGoals.forEach(g -> g.setMobileAppName(mobileAppsById.get(g.getMobileAppId()).getName()));
        return commonMobileGoals;
    }

    /**
     * Получить цели для автобюджетных кампаний внутренней рекламы
     */
    private List<Goal> getMobileGoalsForInternalAutobudgetCampaign() {
        return EntryStream.of(METRIKA_GOALS_FOR_MOBILE_INTERNAL_AUTOBUDGET_CAMPAIGN)
                .mapKeyValue((id, name) -> (Goal) new Goal()
                        .withId(id)
                        .withName(name)
                        .withType(GoalType.GOAL)
                        .withMetrikaCounterGoalType(MetrikaCounterGoalType.ACTION)
                        .withIsMobileGoal(true)
                ).toList();
    }

    private Map<CampaignType, List<Long>> toCampaignIdsByCampaignType(
            Map<Long, CampaignType> campaignIdToCampaignType) {
        return EntryStream.of(campaignIdToCampaignType)
                .nonNullValues()
                .invert()
                .grouping();
    }
}
