package ru.yandex.direct.core.entity.metrika.service.campaigngoals;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.campaign.service.RequestBasedMetrikaClientAdapter;
import ru.yandex.direct.core.entity.metrika.container.CampaignTypeWithCounterIds;
import ru.yandex.direct.core.entity.metrika.container.GoalsGettingForCampaignTypeContainer;
import ru.yandex.direct.core.entity.metrika.container.MetrikaInformationForCampaignGoals;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.metrika.client.model.response.CounterGoal;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.metrika.client.model.response.GetExistentCountersResponseItem;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.validation.Predicates.ignore;

/**
 * Класс содержит методы получения списка целей доступных для редактирования различных блоков кампании
 */
@Service
@ParametersAreNonnullByDefault
public class CampaignGoalsService {
    private final CommonCampaignGoalsService commonCampaignGoalsService;
    private final CommonCampaignCountersService commonCampaignCountersService;
    private final CampMetrikaCountersService campMetrikaCountersService;

    @Autowired
    public CampaignGoalsService(CommonCampaignGoalsService commonCampaignGoalsService,
                                CommonCampaignCountersService commonCampaignCountersService,
                                CampMetrikaCountersService campMetrikaCountersService) {
        this.commonCampaignGoalsService = commonCampaignGoalsService;
        this.commonCampaignCountersService = commonCampaignCountersService;
        this.campMetrikaCountersService = campMetrikaCountersService;
    }

    /**
     * Метод получения целей для связки id кампании + счетчики
     *
     * @param operatorUid                               uid оператора
     * @param clientId                                  id клиента
     * @param campaignTypeWithCounterIdsByCampaignId    мапа campaignId -> объект с типом и
     *                                                  пришедшими от клиента счетчиками, установленными на кампанию
     * @param metrikaData                               полученные заранее данные из метрики
     * @return мапа доступные цели для выбора в ключевых целях по id кампании
     */
    public Map<Long, Set<Goal>> getAvailableGoalsForCampaignId(
            Long operatorUid, ClientId clientId,
            Map<Long, CampaignTypeWithCounterIds> campaignTypeWithCounterIdsByCampaignId,
            @Nullable RequestBasedMetrikaClientAdapter metrikaData
    ) {
        List<Long> campaignsCounterIds = StreamEx.of(campaignTypeWithCounterIdsByCampaignId.values())
                .flatCollection(CampaignTypeWithCounterIds::getCounterIds)
                .distinct()
                .toList();

        boolean shouldFetchUnavailableGoals = Optional.ofNullable(metrikaData)
                .map(RequestBasedMetrikaClientAdapter::isShouldFetchUnavailableGoals)
                .orElseGet(() -> StreamEx.of(campaignTypeWithCounterIdsByCampaignId.values())
                        .anyMatch(campaign -> campaign.isUnavailableAutoGoalsAllowed()
                                || campaign.isUnavailableGoalsAllowed())
                );
        var metrikaInformation = calculateMetrikaInformationIfNeeded(
                clientId, campaignsCounterIds, shouldFetchUnavailableGoals, metrikaData
        );
        Set<Long> unAllowedInaccessibleCounterIds = getUnAllowedInaccessibleCounterIds(
                shouldFetchUnavailableGoals, clientId,
                campaignTypeWithCounterIdsByCampaignId.values(), metrikaInformation);
        var container =
                new GoalsGettingForCampaignTypeContainer(operatorUid, clientId, unAllowedInaccessibleCounterIds);
        return commonCampaignGoalsService.getAvailableGoalsByCampaignId(container,
                campaignTypeWithCounterIdsByCampaignId, metrikaInformation);
    }

    /**
     * Метод получения целей для связки тип кампании + счетчики
     * Можно использовать для массовых запросов данных по целям, для запросов о еще не созданных кампаниях
     *
     * @param operatorUid                   uid оператора
     * @param clientId                      id клиента
     * @param campaignTypeWithCounterIds    список типов кампаний и их счетчиков
     * @param metrikaData                   полученные заранее данные из метрики
     * @return мапа доступные цели для выбора в ключевых целях по типу кампании + счетчикам
     */
    public Map<CampaignTypeWithCounterIds, Set<Goal>> getAvailableGoalsForCampaignType(
            Long operatorUid, ClientId clientId,
            Set<CampaignTypeWithCounterIds> campaignTypeWithCounterIds,
            @Nullable RequestBasedMetrikaClientAdapter metrikaData
    ) {
        List<Long> campaignsCounterIds = StreamEx.of(campaignTypeWithCounterIds)
                .flatCollection(CampaignTypeWithCounterIds::getCounterIds)
                .distinct()
                .toList();

        boolean shouldFetchUnavailableGoals = Optional.ofNullable(metrikaData)
                .map(RequestBasedMetrikaClientAdapter::isShouldFetchUnavailableGoals)
                .orElseGet(() -> StreamEx.of(campaignTypeWithCounterIds)
                        .anyMatch(campaign -> campaign.isUnavailableAutoGoalsAllowed()
                                || campaign.isUnavailableGoalsAllowed())
                );
        var metrikaInformation = calculateMetrikaInformationIfNeeded(
                clientId, campaignsCounterIds, shouldFetchUnavailableGoals, metrikaData
        );
        Set<Long> unAllowedInaccessibleCounterIds = getUnAllowedInaccessibleCounterIds(
                shouldFetchUnavailableGoals, clientId, campaignTypeWithCounterIds, metrikaInformation);
        var container =
                new GoalsGettingForCampaignTypeContainer(operatorUid, clientId, unAllowedInaccessibleCounterIds);
        return commonCampaignGoalsService.getAvailableGoalsForCampaignType(container,
                campaignTypeWithCounterIds, metrikaInformation);
    }

    /**
     * Получить недоступные счетчики с кампаний, цели с которых нельзя использовать в стратегии.
     * Нельзя, т.к. домен с этих счетчиков может быть в блэклисте.
     */
    private Set<Long> getUnAllowedInaccessibleCounterIds(
            boolean unavailableGoalsOrAutoGoalsAllowed,
            ClientId clientId,
            Collection<CampaignTypeWithCounterIds> campaignTypeWithCounterIds,
            MetrikaInformationForCampaignGoals metrikaInformation) {
        if (!unavailableGoalsOrAutoGoalsAllowed) {
            return Set.of();
        }
        Set<Long> availableCounterIds = mapSet(metrikaInformation.getAvailableCounterIds(), Integer::longValue);
        Set<Long> unavailableCounterIds = StreamEx.of(campaignTypeWithCounterIds)
                .flatMap(c -> c.getCounterIds().stream())
                .remove(availableCounterIds::contains)
                .toSet();
        Set<Long> allowedInaccessibleCounterIds =
                campMetrikaCountersService.getAllowedInaccessibleCounterIds(unavailableCounterIds);
        return Sets.difference(unavailableCounterIds, allowedInaccessibleCounterIds);
    }

    private MetrikaInformationForCampaignGoals calculateMetrikaInformationIfNeeded(
            ClientId clientId,
            List<Long> campaignsCounterIds,
            boolean shouldFetchUnavailableGoals,
            @Nullable RequestBasedMetrikaClientAdapter metrikaData
    ) {
        var availableMetrikaCounters = Optional.ofNullable(metrikaData)
                .map(RequestBasedMetrikaClientAdapter::getUsersCountersNumExtendedByCampaignCounterIds)
                .orElseGet(() -> commonCampaignCountersService
                        .getAvailableCountersFromMetrika(clientId, campaignsCounterIds));

        var campaignsCounters = Optional.ofNullable(metrikaData)
                .map(RequestBasedMetrikaClientAdapter::getCampaignCountersIfNeeded)
                .orElseGet(() -> commonCampaignCountersService.getExistentCountersFromMetrika(
                        clientId, campaignsCounterIds, shouldFetchUnavailableGoals
                ));

        var goalsByCounterId = Optional.ofNullable(metrikaData)
                .map(RequestBasedMetrikaClientAdapter::getCountersGoals)
                .orElseGet(() -> getCounterGoals(
                        availableMetrikaCounters, campaignsCounters, shouldFetchUnavailableGoals));

        return new MetrikaInformationForCampaignGoals(campaignsCounters, availableMetrikaCounters, goalsByCounterId);
    }

    private Map<Integer, List<CounterGoal>> getCounterGoals(List<UserCountersExtended> availableMetrikaCounters,
                                                            List<GetExistentCountersResponseItem> campaignsCounters,
                                                            boolean shouldFetchUnavailableGoals) {
        var availableMetrikaCounterIds = StreamEx.of(availableMetrikaCounters)
                .flatCollection(UserCountersExtended::getCounters)
                .map(CounterInfoDirect::getId)
                .toSet();

        var relevantCounterIds = StreamEx.of(campaignsCounters)
                .mapToEntry(GetExistentCountersResponseItem::getCounterId,
                        GetExistentCountersResponseItem::getCounterSource)
                .mapValues(CampaignConverter::toMetrikaCounterSource)
                .filterValues(shouldFetchUnavailableGoals ? ignore() : MetrikaCounterSource.SPRAV::equals)
                .keys()
                .map(Long::intValue)
                .append(availableMetrikaCounterIds)
                .toSet();

        return isEmpty(relevantCounterIds)
                ? emptyMap()
                : commonCampaignCountersService.getCountersGoalsWithExceptionHandling(relevantCounterIds);
    }
}
