package ru.yandex.direct.core.entity.metrika.service.campaigngoals;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.metrika.container.MetrikaInformationForCampaignGoals;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalStatus;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;
import ru.yandex.direct.metrika.client.model.response.CounterGoal;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.retargeting.converter.GoalConverter.fromCounterGoal;
import static ru.yandex.direct.core.entity.retargeting.model.Goal.METRIKA_ECOMMERCE_BASE;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;


@ParametersAreNonnullByDefault
public class CampaignGoalsUtils {

    /**
     * Из разных источников собираем полный набор целей.
     */
    public static Set<Goal> collectGoals(
            @Nullable Set<Long> countersGoalsIds, @Nullable Set<Long> goalIdsWithStatistic,
            @Nullable Set<Long> ecommerceGoalIds, @Nullable Set<Goal> mobileGoals,
            Map<Long, Goal> knownInDirectGoalById, MetrikaInformationForCampaignGoals metrikaInformation) {
        return collectGoalsWithNonnullArguments(nvl(countersGoalsIds, emptySet()),
                nvl(goalIdsWithStatistic, emptySet()),
                nvl(ecommerceGoalIds, emptySet()),
                nvl(mobileGoals, emptySet()),
                knownInDirectGoalById, metrikaInformation);
    }

    private static Set<Goal> collectGoalsWithNonnullArguments(
            Set<Long> countersGoalsIds, Set<Long> goalIdsWithStatistic,
            Set<Long> ecommerceGoalIds, Set<Goal> mobileGoals,
            Map<Long, Goal> knownInDirectGoalById, MetrikaInformationForCampaignGoals metrikaInformation) {

        Set<Long> mobileGoalIds = listToSet(mobileGoals, GoalBase::getId);

        Set<Goal> countersGoals = StreamEx.of(countersGoalsIds)
                //в метрике хранятся цели c id мобильных целей, но заполненные некорректными данными, приходится
                // фильтровать
                .remove(mobileGoalIds::contains)
                .map(goalId -> getGoal(metrikaInformation.getGoals(), knownInDirectGoalById, goalId))
                .nonNull()
                .toSet();

        Set<Goal> goalWithStatistic = StreamEx.of(goalIdsWithStatistic)
                .remove(countersGoalsIds::contains)
                .remove(mobileGoalIds::contains)
                .map(goalId -> getGoal(metrikaInformation.getGoals(), knownInDirectGoalById, goalId))
                .toSet();

        Set<Goal> ecommerceGoals = listToSet(ecommerceGoalIds, goalId -> constructEcommerceGoal(metrikaInformation,
                goalId));

        return StreamEx.of(countersGoals)
                .append(goalWithStatistic)
                .map(goal -> CampaignGoalsUtils.addCounterInfoToGoal(goal,
                        metrikaInformation.getCounterByGoalId(goal.getId())))
                .removeBy(GoalBase::getStatus, GoalStatus.DELETED)
                .append(mobileGoals)
                .append(ecommerceGoals)
                .toSet();
    }

    private static Goal constructEcommerceGoal(MetrikaInformationForCampaignGoals metrikaInformation, Long goalId) {
        int counterId = (int) (goalId - METRIKA_ECOMMERCE_BASE);
        return (Goal) new Goal()
                .withId(goalId)
                .withName(String.valueOf(counterId))
                .withType(GoalType.ECOMMERCE)
                .withMetrikaCounterGoalType(MetrikaCounterGoalType.ECOMMERCE)
                .withCounterId(counterId)
                .withDomain(metrikaInformation.getCounterById(counterId).getSitePath());
    }

    private static Goal getGoal(Map<Long, CounterGoal> metrikaGoalById, Map<Long, Goal> knownInDirectGoalById,
                                Long goalId) {
        //если цель есть и в метрике и в нашей базе, название из метрики может быть свежее
        if (knownInDirectGoalById.containsKey(goalId) && metrikaGoalById.containsKey(goalId)) {
            Goal goalFromMetrika = fromCounterGoal(metrikaGoalById.get(goalId));
            return (Goal) knownInDirectGoalById.get(goalId)
                    .withName(goalFromMetrika.getName());
        }

        return knownInDirectGoalById.containsKey(goalId) ? knownInDirectGoalById.get(goalId) :
                fromCounterGoal(metrikaGoalById.get(goalId));
    }

    //мобильные цели и цели ecomm заполняем сами
    private static Goal addCounterInfoToGoal(Goal goal, @Nullable CounterInfoDirect counter) {
        Integer counterId = Optional.ofNullable(counter)
                .map(CounterInfoDirect::getId)
                .orElse(null);
        Long parentId = goal.getParentId();
        return (Goal) goal
                .withParentId(parentId != null && parentId == 0L ? null : parentId)
                .withCounterId(counterId)
                .withCounterName(ifNotNull(counter, CounterInfoDirect::getName))
                .withDomain(ifNotNull(counter, CounterInfoDirect::getSitePath));

    }

    public static Map<Integer, CounterInfoDirect> convertToCountersById(List<UserCountersExtended> userCounters) {
        return StreamEx.of(userCounters)
                .flatCollection(UserCountersExtended::getCounters)
                .distinct(CounterInfoDirect::getId)
                .mapToEntry(CounterInfoDirect::getId, Function.identity())
                .toMap();
    }
}
