package ru.yandex.direct.core.entity.metrika.service.campaigngoals;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.service.commongoals.CommonCountersService;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformation;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.metrika.client.model.request.GetExistentCountersRequest;
import ru.yandex.direct.metrika.client.model.response.CounterGoal;
import ru.yandex.direct.metrika.client.model.response.GetExistentCountersResponseItem;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.toMetrikaCountersWithAdditionalInformation;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
@ParametersAreNonnullByDefault
public class CommonCampaignCountersService {
    private static final Logger logger = LoggerFactory.getLogger(CommonCampaignCountersService.class);

    private final MetrikaClient metrikaClient;
    private final FeatureService featureService;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final CommonCountersService commonCountersService;

    @Autowired
    public CommonCampaignCountersService(MetrikaClient metrikaClient,
                                         FeatureService featureService,
                                         CampMetrikaCountersService campMetrikaCountersService,
                                         CommonCountersService commonCountersService) {
        this.metrikaClient = metrikaClient;
        this.featureService = featureService;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.commonCountersService = commonCountersService;
    }

    /**
     * Преобразует исходный набор счетчиков в набор счетчиков по которым можем показать цели.
     * фильтрует пользовательские счетчики, на которые нет прав
     * добавляет счетчики, по которым нужно показывать цели безотносительно прав (например счетчики справочника)
     *
     * @param clientId                          id клиента
     * @param counterIdsMap                     пришедшие счетчики по ключу, которые нужно отфильтровать
     * @param availableCountersPreCalculated    доступные клиенту счетчики
     * @param campaignCountersDataPreCalculated информация о фильтруемых счетчиках
     * @param <T>                               тип ключа
     * @return сет счетчиков по ключу
     */
    public <T> Map<T, Set<Long>> getRelevantCounterIds(
            ClientId clientId,
            Map<T, Set<Long>> counterIdsMap,
            List<UserCountersExtended> availableCountersPreCalculated,
            List<GetExistentCountersResponseItem> campaignCountersDataPreCalculated
    ) {
        var relevantCounters = getRelevantCounters(
                clientId, counterIdsMap, availableCountersPreCalculated, campaignCountersDataPreCalculated
        );

        return EntryStream.of(relevantCounters)
                .mapValues(counterInfo -> mapSet(counterInfo, MetrikaCounterWithAdditionalInformation::getId))
                .toMap();
    }

    /**
     * Преобразует исходный набор счетчиков в набор счетчиков по которым можем показать цели.
     * Каждый счетчик содержит в себе подробную информацию из метрики.
     * фильтрует пользовательские счетчики, на которые нет прав
     * добавляет счетчики, по которым нужно показывать цели безотносительно прав (например счетчики справочника)
     *
     * @param clientId                          id клиента
     * @param counterIdsMap                     пришедшие счетчики по ключу, которые нужно отфильтровать
     * @param availableCountersPreCalculated    доступные клиенту счетчики
     * @param campaignCountersDataPreCalculated информация о фильтруемых счетчиках
     * @param <T>                               тип ключа
     * @return сет счетчиков с информацией по ним
     */
    public <T> Map<T, Set<MetrikaCounterWithAdditionalInformation>> getRelevantCounters(
            ClientId clientId,
            Map<T, Set<Long>> counterIdsMap,
            List<UserCountersExtended> availableCountersPreCalculated,
            List<GetExistentCountersResponseItem> campaignCountersDataPreCalculated
    ) {

        var countersById = listToMap(
                toMetrikaCountersWithAdditionalInformation(availableCountersPreCalculated),
                MetrikaCounterWithAdditionalInformation::getId
        );

        var spravCounterIds =
                campMetrikaCountersService.filterSpravCounterIdsInMetrika(campaignCountersDataPreCalculated);
        var spravCountersById = StreamEx.of(spravCounterIds)
                .filter(id -> !countersById.containsKey(id))
                .toMap(id -> id, id -> new MetrikaCounterWithAdditionalInformation()
                        .withId(id)
                        .withHasEcommerce(false)
                        .withSource(MetrikaCounterSource.SPRAV)
                );
        countersById.putAll(spravCountersById);


        return EntryStream.of(counterIdsMap)
                .mapValues(counterIds -> filterAndMapToSet(counterIds, countersById::containsKey, countersById::get))
                .toMap();
    }

    public List<UserCountersExtended> getAvailableCountersFromMetrika(ClientId clientId, List<Long> counterIds) {
        return commonCountersService.getAvailableCountersFromMetrika(clientId, counterIds);
    }

    public List<GetExistentCountersResponseItem> getExistentCountersFromMetrika(ClientId clientId,
                                                                                List<Long> campaignsCounterIds,
                                                                                boolean shouldFetchUnavailableAutoGoals) {
        var goalsFromAllOrgsAllowed = featureService.isEnabledForClientId(
                clientId, FeatureName.GOALS_FROM_ALL_ORGS_ALLOWED
        );

        boolean shouldNotFetchCampaignCounters = !shouldFetchUnavailableAutoGoals && !goalsFromAllOrgsAllowed;
        return isEmpty(campaignsCounterIds) || shouldNotFetchCampaignCounters
                ? emptyList() : metrikaClient.getExistentCounters(
                        new GetExistentCountersRequest().withCounterIds(campaignsCounterIds)).getResponseItems();
    }

    /**
     * Если по какой-либо причине счетчик удален в метрике при запросе целей метрика вернет нам ошибку
     * Чтобы не падать в этом месте делаем обертку
     * метод дубликат, чтобы не тащить зависимость из MetrikaGoalsService
     */
    public Map<Integer, List<CounterGoal>> getCountersGoalsWithExceptionHandling(Set<Integer> counterIds) {
        try {
            return metrikaClient.getMassCountersGoalsFromMetrika(counterIds);
        } catch (MetrikaClientException e) {
            logger.warn("Counters with ids: " + StreamEx.of(counterIds).joining() + " is unavailable", e);
            return emptyMap();
        }
    }

}
