package ru.yandex.direct.core.entity.metrika.service.campaigngoals;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.container.CampaignTypeWithCounterIds;
import ru.yandex.direct.core.entity.metrika.container.GoalsGettingForCampaignTypeContainer;
import ru.yandex.direct.core.entity.metrika.container.MetrikaInformationForCampaignGoals;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository;
import ru.yandex.direct.core.entity.metrika.service.MobileGoalsService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingGoalsPpcDictRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.metrika.client.model.response.CounterGoal;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.metrika.client.model.response.GetExistentCountersResponseItem;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService.TECHNICAL_COUNTER_SOURCES;
import static ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService.getCounterIdForEcommerceGoal;
import static ru.yandex.direct.core.entity.retargeting.model.Goal.METRIKA_ECOMMERCE_BASE;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Service
@ParametersAreNonnullByDefault
public class CommonCampaignGoalsService {
    private final ShardHelper shardHelper;

    private final FeatureService featureService;
    private final MobileGoalsService mobileGoalsService;

    private final MetrikaCampaignRepository metrikaCampaignRepository;
    private final RetargetingGoalsPpcDictRepository retargetingGoalsPpcDictRepository;

    @Autowired
    public CommonCampaignGoalsService(ShardHelper shardHelper, FeatureService featureService,
                                      MobileGoalsService mobileGoalsService,
                                      MetrikaCampaignRepository metrikaCampaignRepository,
                                      RetargetingGoalsPpcDictRepository retargetingGoalsPpcDictRepository) {
        this.shardHelper = shardHelper;

        this.featureService = featureService;
        this.mobileGoalsService = mobileGoalsService;

        this.metrikaCampaignRepository = metrikaCampaignRepository;
        this.retargetingGoalsPpcDictRepository = retargetingGoalsPpcDictRepository;
    }

    /**
     * Получение доступных целей для счетчиков в еще не созданных кампаниях
     * В качестве ключа используется CampaignTypeWithCounterIds
     * <p>
     * Метод package private.
     * Вместо него в сервисах лучше использовать {@link CampaignGoalsService#getAvailableGoalsForCampaignType}
     *
     * @param container                  необходимая информация для получения целей
     * @param campaignTypeWithCounterIds счетчики кампаний
     * @param metrikaInformation         информация полученная из метрики
     * @return мапа подходящих целей по CampaignTypeWithCounterIds
     */
    Map<CampaignTypeWithCounterIds, Set<Goal>> getAvailableGoalsForCampaignType(
            GoalsGettingForCampaignTypeContainer container,
            Set<CampaignTypeWithCounterIds> campaignTypeWithCounterIds,
            MetrikaInformationForCampaignGoals metrikaInformation
    ) {
        return getGoalsForCampaignType(container, campaignTypeWithCounterIds, metrikaInformation);
    }

    /**
     * Получение доступных целей для существующих кампаний
     * <p>
     * Метод package private.
     * Вместо него в сервисах лучше использовать
     * {@link CampaignGoalsService#getAvailableGoalsForCampaignId}
     *
     * @param container                                 необходимая информация для получения целей
     * @param campaignTypeWithCounterIdsByCampaignId    счетчики кампаний
     * @param metrikaInformation                        информация полученная из метрики
     * @return мапа подходящих целей по id кампаний
     */
    Map<Long, Set<Goal>> getAvailableGoalsByCampaignId(
            GoalsGettingForCampaignTypeContainer container,
            Map<Long, CampaignTypeWithCounterIds> campaignTypeWithCounterIdsByCampaignId,
            MetrikaInformationForCampaignGoals metrikaInformation
    ) {
        return getGoalsForCampaignId(container, campaignTypeWithCounterIdsByCampaignId, metrikaInformation);
    }

    /**
     * Универсальный метод получения целей для связки тип кампании + счетчики
     */
    private Map<CampaignTypeWithCounterIds, Set<Goal>> getGoalsForCampaignType(
            GoalsGettingForCampaignTypeContainer container,
            Set<CampaignTypeWithCounterIds> campaignTypeWithCounterIds,
            MetrikaInformationForCampaignGoals metrikaInformation
    ) {
        var clientId = container.getClientId();
        var operatorUid = container.getOperatorUid();
        Map<CampaignTypeWithCounterIds, Set<Long>> countersByCampaign = listToMap(campaignTypeWithCounterIds,
                Function.identity(), CampaignTypeWithCounterIds::getCounterIds);
        Map<CampaignTypeWithCounterIds, Set<Long>> countersGoalIdsByCampaign =
                getCounterGoalsForCampaign(container, countersByCampaign, metrikaInformation);

        Set<Long> goalIds = flatMapToSet(countersGoalIdsByCampaign.values(), Function.identity());
        Map<Long, Goal> goalById = getGoalById(goalIds);

        Set<Goal> mobileGoals = StreamEx.of(campaignTypeWithCounterIds)
                .map(CampaignTypeWithCounterIds::getMobileGoalsFilter)
                .nonNull()
                .distinct()
                .flatCollection(filter -> mobileGoalsService.getMobileGoalsForFilter(operatorUid, clientId, filter))
                .toSet();

        boolean coldStartForEcommerceEnabled =
                featureService.isEnabledForClientId(clientId, FeatureName.COLD_START_FOR_ECOMMERCE_GOALS);
        Map<CampaignTypeWithCounterIds, Set<Long>> ecommerceGoalIdsByCampaignId = !coldStartForEcommerceEnabled ?
                emptyMap() :
                getEcommerceGoalIdsByCampaign(container, countersByCampaign, metrikaInformation);


        return fetchNewCampaignsGoals(metrikaInformation, campaignTypeWithCounterIds, goalById,
                countersGoalIdsByCampaign, ecommerceGoalIdsByCampaignId, mobileGoals);
    }

    /**
     * Для каждой кампании и списка её счетчиков получаем их id целей
     */
    private static Map<CampaignTypeWithCounterIds, Set<Long>> getCounterGoalsForCampaign(
            GoalsGettingForCampaignTypeContainer container,
            Map<CampaignTypeWithCounterIds, Set<Long>> countersMap,
            MetrikaInformationForCampaignGoals metrikaInformation) {
        Set<Integer> availableCounterIds = metrikaInformation.getAvailableCounterIds();
        Map<Integer, MetrikaCounterSource> counterSourceByCounterIds =
                getCounterSourceByCampaignCounterIds(metrikaInformation);

        Map<Integer, List<CounterGoal>> goalsByCounterId = metrikaInformation.getGoalsByCounterId();

        return EntryStream.of(countersMap)
                .flatMapValues(Collection::stream)
                .removeValues(container.getUnAllowedInaccessibleCounterIds()::contains)
                .mapValues(Long::intValue)
                .mapToValue((campaign, counterId) ->
                        getAvailableGoals(campaign, counterId, counterSourceByCounterIds.get(counterId),
                                availableCounterIds, goalsByCounterId))
                .nonNullValues()
                .flatMapValues(Collection::stream)
                .mapValues(CounterGoal::getId)
                .mapValues(goalId -> (long) goalId)
                .grouping(toSet());
    }

    /**
     * Для каждой кампании и списка её счетчиков получаем их id целей
     */
    private static Map<Long, Set<Long>> getCounterGoalsForCampaignId(
            GoalsGettingForCampaignTypeContainer container,
            Map<Long, CampaignTypeWithCounterIds> campaignTypeWithCounterIdsByCampaignId,
            MetrikaInformationForCampaignGoals metrikaInformation) {
        Set<Integer> availableCounterIds = metrikaInformation.getAvailableCounterIds();
        Map<Integer, MetrikaCounterSource> counterSourceByCounterIds =
                getCounterSourceByCampaignCounterIds(metrikaInformation);

        Map<Integer, List<CounterGoal>> goalsByCounterId = metrikaInformation.getGoalsByCounterId();

        return EntryStream.of(campaignTypeWithCounterIdsByCampaignId)
                .mapValues(CampaignTypeWithCounterIds::getCounterIds)
                .flatMapValues(Collection::stream)
                .removeValues(container.getUnAllowedInaccessibleCounterIds()::contains)
                .mapValues(Long::intValue)
                .mapToValue((campaignId, counterId) ->
                        getAvailableGoals(campaignTypeWithCounterIdsByCampaignId.get(campaignId),
                                counterId, counterSourceByCounterIds.get(counterId),
                                availableCounterIds, goalsByCounterId))
                .nonNullValues()
                .flatMapValues(Collection::stream)
                .mapValues(CounterGoal::getId)
                .mapValues(goalId -> (long) goalId)
                .grouping(toSet());
    }

    private static Map<Integer, MetrikaCounterSource> getCounterSourceByCampaignCounterIds(
            MetrikaInformationForCampaignGoals metrikaInformation) {
        return StreamEx.of(metrikaInformation.getCampaignsCounters())
                .mapToEntry(GetExistentCountersResponseItem::getCounterId,
                        GetExistentCountersResponseItem::getCounterSource)
                .mapKeys(Long::intValue)
                .mapValues(CampaignConverter::toMetrikaCounterSource)
                .toMap();
    }

    @Nullable
    private static List<CounterGoal> getAvailableGoals(CampaignTypeWithCounterIds campaign,
                                                       Integer counterId,
                                                       @Nullable MetrikaCounterSource counterSource,
                                                       Set<Integer> availableCounterIds,
                                                       Map<Integer, List<CounterGoal>> goalsByCounterId) {

        if (availableCounterIds.contains(counterId)
                || counterSource == MetrikaCounterSource.SPRAV
                || campaign.isUnavailableGoalsAllowed()) {
            return goalsByCounterId.get(counterId);
        }
        if (campaign.isUnavailableAutoGoalsAllowed()) {
            // для технических счетчиков позволяем все цели (такие цели тоже считаем автоцелями)
            if (counterSource != null && TECHNICAL_COUNTER_SOURCES.contains(counterSource)) {
                return goalsByCounterId.get(counterId);
            }
            // для остальных только цели с source=auto
            return filterList(goalsByCounterId.get(counterId), goal -> goal.getSource() == CounterGoal.Source.AUTO);
        }
        return null;
    }

    /**
     * Универсальный метод получения целей по id кампаний
     * <p>
     * !!!Метод может отдавать цели недоступных счетчиков!!! нужно, для того,
     * чтобы снаружи можно было доопределить список счетчиков, например недоступные счетчики организации
     *
     * @param container                                 необходимые параметры для получения целей
     * @param campaignTypeWithCounterIdsByCampaignId    список счетчиков с которых собираются цели
     * @param metrikaInformation                        предварительно полученные данные из метрики
     * @return Map (id цели, цель)
     */
    private Map<Long, Set<Goal>> getGoalsForCampaignId(
            GoalsGettingForCampaignTypeContainer container,
            Map<Long, CampaignTypeWithCounterIds> campaignTypeWithCounterIdsByCampaignId,
            MetrikaInformationForCampaignGoals metrikaInformation
    ) {
        int shard = shardHelper.getShardByClientId(container.getClientId());
        Map<Long, Set<Long>> countersGoalIdsByCampaignId =
                getCounterGoalsForCampaignId(container, campaignTypeWithCounterIdsByCampaignId, metrikaInformation);

        Set<Long> campaignIds = campaignTypeWithCounterIdsByCampaignId.keySet();
        Map<Long, Set<Long>> goalIdsWithStatisticByCampaignId = getGoalIdsWithStatisticByCampaignId(
                shard, container.getClientId(), campaignIds);

        Map<Long, Goal> goalById = getGoalsFromDbById(countersGoalIdsByCampaignId, goalIdsWithStatisticByCampaignId);

        Set<String> enabledFeatures = featureService.getEnabledForClientId(container.getClientId());
        boolean coldStartForEcommerceEnabled =
                enabledFeatures.contains(FeatureName.COLD_START_FOR_ECOMMERCE_GOALS.getName());

        Map<Long, Set<Long>> ecommerceGoalIdsByCampaignId = getEcommerceGoalIdsByCampaignId(
                container, metrikaInformation,
                coldStartForEcommerceEnabled, campaignTypeWithCounterIdsByCampaignId,
                goalIdsWithStatisticByCampaignId, goalById);

        Map<Long, Set<Long>> availableGoalIdsWithStatisticByCampaignId =
                EntryStream.of(goalIdsWithStatisticByCampaignId)
                        .flatMapValues(Collection::stream)
                        .filterValues(metrikaInformation::isGoalAvailableInMetrika)
                        .grouping(toSet());

        Map<Long, CampaignType> campaignIdToCampaignType = EntryStream.of(campaignTypeWithCounterIdsByCampaignId)
                .mapValues(CampaignTypeWithCounterIds::getCampaignType)
                .nonNullValues()
                .toMap();
        Map<Long, List<Goal>> mobileGoalsByCampaignId = mobileGoalsService.getMobileGoalsByCampaignId(
                shard, container.getOperatorUid(), container.getClientId(), campaignIdToCampaignType, true
        );

        return fetchExistentCampaignsGoals(metrikaInformation, campaignIds,
                goalById, countersGoalIdsByCampaignId, availableGoalIdsWithStatisticByCampaignId,
                mobileGoalsByCampaignId, ecommerceGoalIdsByCampaignId);
    }

    /**
     * В нашей базе для каждой кампании лежит список целей, по которым есть статистика.
     * Статистика может быть пустой, если мы сами её создали -- при создании или обновлении кампании
     * мы пишем цели со всех привязанных счетчиков с пустой статистикой.
     * Или заполненной, если пришла из бк.
     * <p>
     * Эти цели важно получить, тк: цели ecomm мы не получаем из метрики(пока), часть целей может быть со статистикой,
     * Но пользователь не указал их счетчики
     */
    private Map<Long, Set<Long>> getGoalIdsWithStatisticByCampaignId(int shard, ClientId clientId,
                                                                     Collection<Long> campaignIds) {
        Map<Long, List<Long>> goalIdsByCampaignId = metrikaCampaignRepository.getGoalIdsByCampaignId(shard,
                clientId.asLong(), campaignIds);

        return EntryStream.of(goalIdsByCampaignId)
                .mapValues(Set::copyOf)
                .toMap();
    }

    private Map<Long, Goal> getGoalsFromDbById(Map<Long, Set<Long>> countersGoalsByCampaignId,
                                               Map<Long, Set<Long>> goalIdsWithStatisticByCampaignId) {
        Set<Long> allCountersGoalIds = flatMapToSet(countersGoalsByCampaignId.values(), Function.identity());

        Set<Long> goalIdsWithStatistic = flatMapToSet(goalIdsWithStatisticByCampaignId.values(), Function.identity());

        return getGoalById(Sets.union(allCountersGoalIds, goalIdsWithStatistic));
    }

    /**
     * Получение информации о целях из ppcdict
     */
    private Map<Long, Goal> getGoalById(Set<Long> goalIds) {
        return listToMap(retargetingGoalsPpcDictRepository.getMetrikaGoalsFromPpcDict(goalIds, true), GoalBase::getId);
    }

    private Map<Long, Set<Long>> getEcommerceGoalIdsByCampaignId(
            GoalsGettingForCampaignTypeContainer container,
            MetrikaInformationForCampaignGoals metrikaInformation, boolean coldStartForEcommerceEnabled,
            Map<Long, CampaignTypeWithCounterIds> campaignTypeWithCounterIdsByCampaignId,
            Map<Long, Set<Long>> goalIdsWithStatisticByCampaignId,
            Map<Long, Goal> goalById) {
        Set<Integer> ecommerceCounterIds = filterAndMapToSet(metrikaInformation.getAvailableCounters(),
                CounterInfoDirect::getEcommerce,
                CounterInfoDirect::getId);
        Map<Long, Set<Long>> ecommerceGoalIdsWithStatisticByCampaignId =
                getEcommerceGoalsWithStatisticByCampaignId(goalIdsWithStatisticByCampaignId,
                        ecommerceCounterIds, goalById);

        if (!coldStartForEcommerceEnabled) {
            return ecommerceGoalIdsWithStatisticByCampaignId;
        }

        Map<Long, Set<Long>> ecommerceGoalIdsFromCountersByCampaignId =
                getEcommerceGoalIdsByCampaignId(container, campaignTypeWithCounterIdsByCampaignId, metrikaInformation);

        return EntryStream.of(ecommerceGoalIdsWithStatisticByCampaignId)
                .append(EntryStream.of(ecommerceGoalIdsFromCountersByCampaignId))
                .toMap(Sets::union);
    }

    private Map<CampaignTypeWithCounterIds, Set<Long>> getEcommerceGoalIdsByCampaign(
            GoalsGettingForCampaignTypeContainer container,
            Map<CampaignTypeWithCounterIds, Set<Long>> countersMap,
            MetrikaInformationForCampaignGoals metrikaInformation) {
        return EntryStream.of(countersMap)
                .flatMapValues(Collection::stream)
                .removeValues(container.getUnAllowedInaccessibleCounterIds()::contains)
                .filterKeyValue((campaign, counterId) ->
                        metrikaInformation.getAvailableCounterIds().contains(counterId.intValue()) ||
                                campaign.isUnavailableGoalsAllowed() ||
                                // ecom-цели считаем автоцелями (DIRECT-147995)
                                campaign.isUnavailableAutoGoalsAllowed())
                .mapValues(counterId -> metrikaInformation.getCounterById(counterId.intValue()))
                .nonNullValues()
                .filterValues(CounterInfoDirect::getEcommerce)
                .mapValues(counter -> METRIKA_ECOMMERCE_BASE + counter.getId())
                .grouping(toSet());
    }

    private Map<Long, Set<Long>> getEcommerceGoalIdsByCampaignId(
            GoalsGettingForCampaignTypeContainer container,
            Map<Long, CampaignTypeWithCounterIds> campaignTypeWithCounterIdsByCampaignId,
            MetrikaInformationForCampaignGoals metrikaInformation) {
        return EntryStream.of(campaignTypeWithCounterIdsByCampaignId)
                .mapValues(CampaignTypeWithCounterIds::getCounterIds)
                .flatMapValues(Collection::stream)
                .removeValues(container.getUnAllowedInaccessibleCounterIds()::contains)
                .filterKeyValue((campaignId, counterId) ->
                        metrikaInformation.getAvailableCounterIds().contains(counterId.intValue()) ||
                                campaignTypeWithCounterIdsByCampaignId.get(campaignId).isUnavailableGoalsAllowed() ||
                                // ecom-цели считаем автоцелями (DIRECT-147995)
                                campaignTypeWithCounterIdsByCampaignId.get(campaignId).isUnavailableAutoGoalsAllowed())
                .mapValues(counterId -> metrikaInformation.getCounterById(counterId.intValue()))
                .nonNullValues()
                .filterValues(CounterInfoDirect::getEcommerce)
                .mapValues(counter -> METRIKA_ECOMMERCE_BASE + counter.getId())
                .grouping(toSet());
    }

    /**
     * Получаем ecomm цели из статистики: если находится счетчик == имени цели и тип цели == ECOMMERCE,
     * то считаем, эту цель доступной и ecomm
     */
    private static Map<Long, Set<Long>> getEcommerceGoalsWithStatisticByCampaignId(Map<Long, Set<Long>> goalIdsWithStatisticByCampaignId,
                                                                                   Set<Integer> availableMetrikaCounterIds,
                                                                                   Map<Long, Goal> goalById) {
        return EntryStream.of(goalIdsWithStatisticByCampaignId)
                .mapValues(goalIds -> StreamEx.of(goalIds).map(goalById::get).nonNull().toSet())
                .mapValues(goals -> getEcommerceGoalIds(availableMetrikaCounterIds, goals))
                .toMap();
    }

    private static Set<Long> getEcommerceGoalIds(Set<Integer> availableCounterIds, Set<Goal> goals) {
        return filterAndMapToSet(goals,
                g -> g.getMetrikaCounterGoalType() == MetrikaCounterGoalType.ECOMMERCE &&
                        availableCounterIds.contains(getCounterIdForEcommerceGoal(g)),
                GoalBase::getId);
    }

    private static Map<CampaignTypeWithCounterIds, Set<Goal>> fetchNewCampaignsGoals(
            MetrikaInformationForCampaignGoals metrikaInformation,
            Collection<CampaignTypeWithCounterIds> campaigns,
            Map<Long, Goal> goalById,
            Map<CampaignTypeWithCounterIds, Set<Long>> countersByCampaign,
            Map<CampaignTypeWithCounterIds, Set<Long>> ecommerceGoalIdsByCampaignId,
            Set<Goal> mobileGoals) {
        return listToMap(campaigns, Function.identity(),
                campaign -> CampaignGoalsUtils.collectGoals(
                        countersByCampaign.get(campaign), emptySet(), ecommerceGoalIdsByCampaignId.get(campaign),
                        mobileGoals, goalById, metrikaInformation));
    }

    private static Map<Long, Set<Goal>> fetchExistentCampaignsGoals(
            MetrikaInformationForCampaignGoals metrikaInformation,
            Collection<Long> campaignIds,
            Map<Long, Goal> goalById,
            Map<Long, Set<Long>> countersGoalsByCampaignId,
            Map<Long, Set<Long>> availableGoalIdsWithStatisticByCampaignId,
            Map<Long, List<Goal>> mobileGoalsByCampaignId,
            Map<Long, Set<Long>> ecommerceGoalIdsByCampaignId) {
        return StreamEx.of(campaignIds)
                .mapToEntry(campaignId -> CampaignGoalsUtils.collectGoals(countersGoalsByCampaignId.get(campaignId),
                        availableGoalIdsWithStatisticByCampaignId.get(campaignId),
                        ecommerceGoalIdsByCampaignId.get(campaignId),
                        Set.copyOf(mobileGoalsByCampaignId.getOrDefault(campaignId, emptyList())),
                        goalById, metrikaInformation))
                .toMap();
    }

}
