package ru.yandex.direct.core.entity.metrika.service.strategygoals

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.service.RequestBasedMetrikaClientAdapter
import ru.yandex.direct.core.entity.metrika.container.CampaignTypeWithCounterIds
import ru.yandex.direct.core.entity.metrika.service.ForCampaignType
import ru.yandex.direct.core.entity.metrika.service.ForStrategy
import ru.yandex.direct.core.entity.metrika.service.MobileGoalsFilter
import ru.yandex.direct.core.entity.metrika.service.campaigngoals.CampaignGoalsService
import ru.yandex.direct.core.entity.metrika.service.strategygoals.StrategyGoalsService.Companion.StrategyAvailableGoalsRequest
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.dbutil.model.ClientId

/**
 * Сервис получения списка доступных целей для стратегий.
 * Источниками целей для стратегий являются:
 *   - счетчики метрики привязанные к стратегии
 *   - тип кампаний привязанных к стратегии
 *   - цели привязанных кампаний провязанные через статистику БК.
 *   @param campaignGoalsService - сервис получения списка мобильных целей кампаний.
 *
 *   В параметраз запроса [StrategyAvailableGoalsRequest] зашито продуктовое ограничение 1 - стратегий 1 - тип кампаний
 *   для всех привязанных кампаний.
 *   Если это требование изменится, то тогда логика получения/фильтрации целей по типу кампаний будет вынесена в [StrategyGoalsService].
 * */
@Service
class StrategyGoalsService @Autowired constructor(val campaignGoalsService: CampaignGoalsService) {

    fun getAvailableForStrategyIds(
        clientId: ClientId,
        operator: Long,
        strategyIdToRequest: Map<Long, StrategyAvailableGoalsRequest>
    ): Map<Long, Set<Goal>> {
        val strategyToGoals = getAvailableGoals(clientId, operator, strategyIdToRequest.values.toSet())
        return strategyIdToRequest
            .mapValues {
                strategyToGoals[it.value] ?: setOf()
            }
    }

    fun getAvailableForStrategies(
        clientId: ClientId,
        operator: Long,
        requests: Set<StrategyAvailableGoalsRequest>
    ): Map<StrategyAvailableGoalsRequest, Set<Goal>> =
        getAvailableGoals(clientId, operator, requests)

    /*
    * Общий массовый метод получения доступных целей стратегий.
    * Разбивает запросы на 2 множества (в которых определен список кампаний, и остальные).
    * И делает по запросу для каждого множества в CampaignGoalsService
    * */
    private fun getAvailableGoals(
        clientId: ClientId,
        operator: Long,
        requests: Set<StrategyAvailableGoalsRequest>,
        metrikaData: RequestBasedMetrikaClientAdapter? = null
    ): Map<StrategyAvailableGoalsRequest, Set<Goal>> {
        val (withCampaignIds, withoutCampaignIds) = requests.partition(StrategyAvailableGoalsRequest::isCampaignIdsDefined)

        val result = mutableMapOf<StrategyAvailableGoalsRequest, Set<Goal>>()
        if (withCampaignIds.isNotEmpty()) {
            result.putAll(getAvailableGoalsByCampaignIds(clientId, operator, withCampaignIds, metrikaData))
        }
        if (withoutCampaignIds.isNotEmpty()) {
            result.putAll(getAvailableGoalsForCampaignType(clientId, operator, withoutCampaignIds, metrikaData))
        }
        return result.toMap()
    }

    /**
     * Метод получения доступных целей стратегии, к которой не привязана кампания (например создание стратегии без привязки).
     * В качестве источника целей выступают счетчики метрики и тип кампаний - по дефолту ТГО.
     *
     * @property clientId - идентификатор клиента - владельца стратегии.
     * @property operator - оператор запроса.
     * @property requests - список запросов для получения списка целей.
     * @metrikaData - кеш запросов в метрику в рамках одного запроса.
     * */
    private fun getAvailableGoalsForCampaignType(
        clientId: ClientId,
        operator: Long,
        requests: List<StrategyAvailableGoalsRequest>,
        metrikaData: RequestBasedMetrikaClientAdapter?
    ): Map<StrategyAvailableGoalsRequest, Set<Goal>> {
        val campaignTypeWithCounterIds = requests.associateBy(Companion::toCampaignTypeWithCounterIds)
        val goalsMap = campaignGoalsService.getAvailableGoalsForCampaignType(
            operator,
            clientId,
            campaignTypeWithCounterIds.keys,
            metrikaData
        )

        return goalsMap.mapNotNull {
            campaignTypeWithCounterIds[it.key]?.let { request -> request to it.value }
        }.toMap()
    }

    /**
     * Метод получения доступных целей стратегии, к которой привязаны кампании.
     * В качестве источника целей выступают счетчики метрики, тип кампаний и цели кампаний провязанные через статистику БК.
     * @property clientId - идентификатор клиента - владельца стратегии.
     * @property operator - оператор запроса.
     * @property requests - список запросов для получения списка целей.
     * @metrikaData - кеш запросов в метрику в рамках одного запроса.
     * */
    private fun getAvailableGoalsByCampaignIds(
        clientId: ClientId,
        operator: Long,
        requests: List<StrategyAvailableGoalsRequest>,
        metrikaData: RequestBasedMetrikaClientAdapter?
    ): Map<StrategyAvailableGoalsRequest, Set<Goal>> {
        val campaignIdToCampaignTypeWithCounterIds =
            requests.flatMap(Companion::toCampaignTypeWithCounterIdsMap).toMap()
        val goalsByCampaignId = campaignGoalsService.getAvailableGoalsForCampaignId(
            operator,
            clientId,
            campaignIdToCampaignTypeWithCounterIds,
            metrikaData
        )

        return requests
            .associateBy { it }
            .mapValues {
                it.value.campaignIds.flatMap { campaignId ->
                    goalsByCampaignId[campaignId] ?: emptyList()
                }.toSet()
            }
    }

    companion object {
        private fun toCampaignTypeWithCounterIdsMap(request: StrategyAvailableGoalsRequest) =
            request.campaignIds
                .map {
                    it to toCampaignTypeWithCounterIds(request)
                }

        fun toCampaignTypeWithCounterIds(request: StrategyAvailableGoalsRequest): CampaignTypeWithCounterIds =
            CampaignTypeWithCounterIds()
                .withCounterIds(request.metrikaCounterIds)
                .withMobileGoalsFilter(request.mobileGoalsFilter())
                .withUnavailableAutoGoalsAllowed(request.isUnavailableAutoGoalsAllowed)

        /**
         * Запрос на получения списка целей стратегии.
         * @property campaignIds - список кампаний привязанных к стратегии.
         * @property metrikaCounterIds- список счетчиков стратегии.
         * @property mobileGoalsFilter - фильтр мобильных целей стратегии.
         * */
        data class StrategyAvailableGoalsRequest(
            val campaignIds: Set<Long>,
            val metrikaCounterIds: Set<Long>,
            private val campaignType: CampaignType? = null,
            val isUnavailableAutoGoalsAllowed: Boolean = false
        ) {
            fun mobileGoalsFilter(): MobileGoalsFilter = if (campaignType != null) {
                ForCampaignType(campaignType)
            } else {
                ForStrategy
            }

            val isCampaignIdsDefined: Boolean = campaignIds.isNotEmpty()
        }
    }
}
