package ru.yandex.direct.core.entity.metrika.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.AUDIENCE;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.CDP_SEGMENT;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.ECOMMERCE;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.GOAL;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.LAL_SEGMENT;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.MOBILE;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.SEGMENT;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Service
public class MetrikaGoalsValidationService {
    private static final Set<GoalType> TYPES_ALLOWED_TO_HAVE_LAL_SEGMENT = Set.of(GOAL, SEGMENT, AUDIENCE, ECOMMERCE,
            CDP_SEGMENT, MOBILE);

    private final RbacService rbacService;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final TurboLandingRepository turboLandingRepository;

    @Autowired
    public MetrikaGoalsValidationService(RbacService rbacService,
                                         CampMetrikaCountersService campMetrikaCountersService,
                                         TurboLandingRepository turboLandingRepository) {
        this.rbacService = rbacService;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.turboLandingRepository = turboLandingRepository;
    }

    public ValidationResult<Void, Defect> validateMetrikaGoalsByCounters(Long operatorUid,
                                                                         Set<Long> campaignIds) {
        ItemValidationBuilder<Void, Defect> vb = ItemValidationBuilder.of(null);

        if (campaignIds != null) {
            Set<Long> visibleCampaignIds = rbacService.getVisibleCampaigns(operatorUid, campaignIds);

            vb.list(List.copyOf(campaignIds), "campaignIds")
                    .checkEach(validId())
                    .checkEach(inSet(visibleCampaignIds), campaignNotFound(), When.isValid());
        }

        return vb.getResult();
    }

    /**
     * Получить все доступные счетчики клиента {@code clientId}.
     * если counterIds не пустой, то проверяет доступность только счетчиков из списка
     * если counterIds пустой, то возвращает все счетчики
     * Если у заданного клиента включена фича {@link FeatureName#GOALS_FROM_ALL_ORGS_ALLOWED},
     * то добавить к доступным счетчикам идентификаторы счетчиков организаций этого клиента,
     * выбранные из {@code counterIds}, даже если они недоступны клиенту
     */
    public Set<Long> getAvailableCounterIds(ClientId clientId, List<Long> counterIds) {
        return Sets.union(
                campMetrikaCountersService.getAvailableAndFilterInputCounterIdsInMetrikaForGoals(clientId, counterIds,
                        true),
                turboLandingRepository.getSystemTurboLandingMetrikaCounterIdsByClientId(clientId));
    }

    public ValidationResult<List<Long>, Defect> validateGoalsForLalSegmentCreation(List<Long> parentGoalIds,
                                                                                   List<Goal> clientGoals) {
        Map<Long, Goal> clientGoalsByIds = listToMap(clientGoals, GoalBase::getId);
        Set<Long> existingSegmentsParents = StreamEx.of(clientGoals)
                .filter(goal -> goal.getType() == LAL_SEGMENT)
                .map(GoalBase::getParentId)
                .toSet();

        ListValidationBuilder<Long, Defect> vb = ListValidationBuilder.of(parentGoalIds);
        vb.checkEach(notNull())
                .checkEach(inSet(clientGoalsByIds.keySet()), When.isValid())
                .checkEach(typeConstraint(clientGoalsByIds), When.isValid())
                .checkEach(notInSet(existingSegmentsParents)); // Parent already has LAL-segment

        return vb.getResult();
    }

    private static Constraint<Long, Defect> typeConstraint(Map<Long, Goal> clientGoalsByIds) {
        return fromPredicate(id -> TYPES_ALLOWED_TO_HAVE_LAL_SEGMENT.contains(clientGoalsByIds.get(id).getType()),
                invalidValue());
    }
}
