package ru.yandex.direct.core.entity.metrika.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.retargeting.model.MetrikaSegmentPreset;
import ru.yandex.direct.core.entity.retargeting.model.PresetsByCounter;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;

@Service
@ParametersAreNonnullByDefault
public class MetrikaSegmentValidationService {

    public ValidationResult<List<PresetsByCounter>, Defect> validateSegmentCreationParameters(
            List<PresetsByCounter> presetsByCounters, List<MetrikaSegmentPreset> availablePresets) {
        var vb = ListValidationBuilder.of(presetsByCounters, Defect.class);

        var editableCounterIds = StreamEx.of(availablePresets)
                .map(MetrikaSegmentPreset::getCounterId)
                .toSet();
        var availablePresetIdsByCounters = StreamEx.of(availablePresets)
                .mapToEntry(MetrikaSegmentPreset::getCounterId, MetrikaSegmentPreset::getPresetId)
                .sortedBy(Map.Entry::getKey)
                .collapseKeys(Collectors.toSet())
                .toMap();

        vb.checkEachBy(presetsByCounter ->
                validatePresetsByCounter(presetsByCounter, editableCounterIds, availablePresetIdsByCounters));

        return vb.getResult();
    }

    private ValidationResult<PresetsByCounter, Defect> validatePresetsByCounter(
            PresetsByCounter presetsByCounter, Set<Integer> editableCounterIds,
            Map<Integer, Set<Integer>> availablePresetsByCounters) {
        ModelItemValidationBuilder<PresetsByCounter> vb = ModelItemValidationBuilder.of(presetsByCounter);

        vb.item(PresetsByCounter.COUNTER_ID).check(inSet(editableCounterIds));
        vb.item(PresetsByCounter.PRESET_IDS).checkBy(presets -> validatePresetIds(
                presets,
                availablePresetsByCounters.get(presetsByCounter.getCounterId())),
                When.isTrue(!vb.getResult().hasAnyErrors()));
        return vb.getResult();
    }

    private ValidationResult<List<Integer>, Defect> validatePresetIds(List<Integer> ids, Set<Integer> availableIds) {
        ListValidationBuilder<Integer, Defect> vb = ListValidationBuilder.of(ids);
        vb.checkEach(inSet(availableIds));
        return vb.getResult();
    }
}
