package ru.yandex.direct.core.entity.metrika.utils;

import java.util.Comparator;
import java.util.List;
import java.util.Set;

import ru.yandex.direct.core.entity.retargeting.model.ConversionLevel;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;

/**
 * Компаратор для сортировки рекомендованных целей.
 * <p>
 * Алгоритм сортировки:
 * Первые в отсортированном списке рекомендованных целей идут зеленые цели (с достижениями больше
 * {@link #MIN_GOALS_CONVERSION_VISIT_COUNT_FOR_SUGGESTION} - в блоке стратегии имеют зеленую раскраску)
 * с типом из {@link #TYPES_OF_SUGGESTED_GOALS_BY_PRIORITY}, отсортированные по достижению в рамках одного типа.
 * Далее идут отсортированные по достижениям зеленые цели, типы которых не входят в
 * {@link #TYPES_OF_SUGGESTED_GOALS_BY_PRIORITY}, и не являются ACTION или URL.
 * Далее отсортированные по достижениям зеленые цели типов ACTION или URL
 * с оценкой конверсионности (conversion_level) isPerfect.
 * Далее остальные зеленые цели (типов ACTION или URL c оценкой конверсионности отличной от isPerfect),
 * отсортированные по достижениям.
 * В конце отсортированные по достижениям, не зеленые цели любых типов.
 */
public class GoalsForSuggestionComparator implements Comparator<Goal> {

    private static final List<MetrikaCounterGoalType> TYPES_OF_SUGGESTED_GOALS_BY_PRIORITY = List.of(
            MetrikaCounterGoalType.ECOMMERCE, MetrikaCounterGoalType.E_CART,
            MetrikaCounterGoalType.CALL, MetrikaCounterGoalType.FORM,
            MetrikaCounterGoalType.PHONE, MetrikaCounterGoalType.EMAIL);

    private static final Set<MetrikaCounterGoalType> TYPES_OF_PERFECT_GOALS = Set.of(
            MetrikaCounterGoalType.ACTION, MetrikaCounterGoalType.URL);

    private static final long MIN_GOALS_CONVERSION_VISIT_COUNT_FOR_SUGGESTION = 20;

    @Override
    public int compare(Goal goal1, Goal goal2) {
        GoalOrderDescription goal1OrderDescription = getGoalOrderDescription(goal1);
        GoalOrderDescription goal2OrderDescription = getGoalOrderDescription(goal2);

        int result = goal1OrderDescription.compareTo(goal2OrderDescription);
        if (result != 0) {
            return result;
        }

        if (goal1OrderDescription == GoalOrderDescription.GOAL_WITH_SPECIFIED_TYPE) {
            int priority1 = TYPES_OF_SUGGESTED_GOALS_BY_PRIORITY.indexOf(goal1.getMetrikaCounterGoalType());
            int priority2 = TYPES_OF_SUGGESTED_GOALS_BY_PRIORITY.indexOf(goal2.getMetrikaCounterGoalType());

            if (priority1 != priority2) {
                return priority1 - priority2;
            }

            return compareByConversionCount(goal1, goal2);
        }

        if (goal1OrderDescription == GoalOrderDescription.GOAL_WITH_NULL_CONVERSIONS) {
            return compareById(goal1, goal2);
        }

        return compareByConversionCount(goal1, goal2);
    }

    private static int compareByConversionCount(Goal goal1, Goal goal2) {
        int result = Long.compare(goal2.getConversionVisitsCount(), goal1.getConversionVisitsCount());
        if (result == 0) {
            return compareById(goal1, goal2);
        }
        return result;
    }

    private static int compareById(Goal goal1, Goal goal2) {
        return Long.compare(goal2.getId(), goal1.getId());
    }

    public enum GoalOrderDescription {
        GOAL_WITH_SPECIFIED_TYPE,
        GOAL_WITH_USUAL_TYPE,
        GOAL_WITH_PERFECT_CONVERSION_LEVEL,
        GOAL_WITH_NOT_PERFECT_CONVERSION_LEVEL,
        GOAL_WITH_NOT_ENOUGH_CONVERSIONS,
        GOAL_WITH_NULL_CONVERSIONS
    }

    public static GoalOrderDescription getGoalOrderDescription(Goal goal) {
        Long conversionCount = goal.getConversionVisitsCount();
        if (conversionCount == null) {
            return GoalOrderDescription.GOAL_WITH_NULL_CONVERSIONS;
        }
        if (conversionCount < MIN_GOALS_CONVERSION_VISIT_COUNT_FOR_SUGGESTION) {
            return GoalOrderDescription.GOAL_WITH_NOT_ENOUGH_CONVERSIONS;
        }

        MetrikaCounterGoalType type = goal.getMetrikaCounterGoalType();
        if (type == null) {
            type = MetrikaCounterGoalType.URL;
        }
        if (TYPES_OF_PERFECT_GOALS.contains(type)) {
            if (goal.getConversionLevel() == ConversionLevel.PERFECT) {
                return GoalOrderDescription.GOAL_WITH_PERFECT_CONVERSION_LEVEL;
            }
            return GoalOrderDescription.GOAL_WITH_NOT_PERFECT_CONVERSION_LEVEL;
        }
        if (TYPES_OF_SUGGESTED_GOALS_BY_PRIORITY.contains(type)) {
            return GoalOrderDescription.GOAL_WITH_SPECIFIED_TYPE;
        }

        return GoalOrderDescription.GOAL_WITH_USUAL_TYPE;
    }
}
