package ru.yandex.direct.core.entity.metrika.utils;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.retargeting.converter.GoalConverter;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalStatus;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;
import ru.yandex.direct.metrika.client.model.response.RetargetingCondition;

import static ru.yandex.direct.core.entity.retargeting.model.Goal.METRIKA_ECOMMERCE_BASE;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
public class MetrikaGoalsUtils {
    public static Map<Long, Set<Goal>> getGoalsByCampaignIds(Set<RetargetingCondition> metrikaGoals,
                                                      Map<Long, Set<Long>> countersByCampaignId) {

        return EntryStream.of(countersByCampaignId)
                .mapValues(HashSet::new)
                .mapValues(counterIds -> StreamEx.of(metrikaGoals)
                        .filter(goal -> counterIds.isEmpty() || countersHasGoal(counterIds, goal))
                        .map(GoalConverter::fromMetrikaRetargetingCondition)
                        .toSet())
                .toMap();
    }

    public static boolean countersHasGoal(Set<Long> counterIds, RetargetingCondition goal) {
        return counterIds.contains((long) goal.getCounterId()) ||
                (goal.getCounterIds() != null &&
                        counterIds.stream()
                                .map(Long::intValue)
                                .anyMatch(goal.getCounterIds()::contains));
    }

    public static Set<Long> counterIdsFromGoal(Goal goal) {
        if (goal.getCounterId() != null && isValidId((long) goal.getCounterId())) {
            return Set.of((long) goal.getCounterId());
        }
        return StreamEx.of(nvl(goal.getCounterIds(), Collections.emptySet()))
                .map(Integer::longValue)
                .toSet();
    }

    public static List<Long> getParentIds(List<Goal> goals) {
        return StreamEx.of(goals)
                .map(GoalBase::getParentId)
                .nonNull()
                .filter(goalId -> goalId != 0)
                .distinct()
                .toList();
    }

    public static Goal ecommerceGoalFromCounterId(Long counterId) {
        return (Goal) (new Goal()
                .withType(GoalType.ECOMMERCE)
                .withId(ecommerceGoalId(counterId))
                .withMetrikaCounterGoalType(MetrikaCounterGoalType.ECOMMERCE)
                .withStatus(GoalStatus.ACTIVE)
                .withCounterId(counterId.intValue())
                .withName(counterId.toString()));
    }

    public static Long ecommerceGoalId(Long counterId) {
        return METRIKA_ECOMMERCE_BASE + counterId;
    }
}
