package ru.yandex.direct.core.entity.minuskeywordspack.repository;

import java.math.BigInteger;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Configuration;
import org.jooq.InsertValuesStep2;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.jooq.types.ULong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.minuskeywordspack.MinusKeywordsPackUtils;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupsMinusWordsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.MinusWordsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static org.jooq.impl.DSL.count;
import static org.springframework.util.CollectionUtils.isEmpty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.util.RepositoryUtils.TRUE;
import static ru.yandex.direct.common.util.RepositoryUtils.booleanToLong;
import static ru.yandex.direct.core.entity.minuskeywordspack.MinusKeywordsPackUtils.calcHash;
import static ru.yandex.direct.dbschema.ppc.tables.AdgroupsMinusWords.ADGROUPS_MINUS_WORDS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignsMinusWords.CAMPAIGNS_MINUS_WORDS;
import static ru.yandex.direct.dbschema.ppc.tables.MinusWords.MINUS_WORDS;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class MinusKeywordsPackRepository {

    private static final JooqMapperWithSupplier<MinusKeywordsPack> MINUS_WORDS_MAPPER =
            JooqMapperWithSupplierBuilder.builder(MinusKeywordsPack::new)
                    .map(property(MinusKeywordsPack.ID, MINUS_WORDS.MW_ID))
                    .map(property(MinusKeywordsPack.NAME, MINUS_WORDS.MW_NAME))
                    .map(convertibleProperty(MinusKeywordsPack.MINUS_KEYWORDS, MINUS_WORDS.MW_TEXT,
                            MinusKeywordsPackUtils::minusKeywordsFromJson, MinusKeywordsPackUtils::minusKeywordsToJson))
                    .map(convertibleProperty(MinusKeywordsPack.HASH, MINUS_WORDS.MW_HASH, ULong::toBigInteger,
                            ULong::valueOf))
                    .map(booleanProperty(MinusKeywordsPack.IS_LIBRARY, MINUS_WORDS.IS_LIBRARY))
                    .build();


    private final DslContextProvider ppcDslContextProvider;
    private final ShardHelper shardHelper;

    @Autowired
    public MinusKeywordsPackRepository(DslContextProvider ppcDslContextProvider,
                                       ShardHelper shardHelper) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.shardHelper = shardHelper;
    }

    /**
     * Найти существующие идентификаторы библиотечных наборов минус фраз, принадлежащих клиенту по mwIds
     *
     * @param shard    шард
     * @param clientId ID клиента
     * @param mwIds    Набор ID минус фраз
     */
    public Set<Long> getClientExistingLibraryMinusKeywordsPackIds(int shard, ClientId clientId,
                                                                  Collection<Long> mwIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(MINUS_WORDS.MW_ID)
                .from(MINUS_WORDS)
                .where(MINUS_WORDS.MW_ID.in(mwIds))
                .and(MINUS_WORDS.CLIENT_ID.eq(clientId.asLong()))
                .and(MINUS_WORDS.IS_LIBRARY.eq(TRUE))
                .fetchSet(MINUS_WORDS.MW_ID);
    }

    /**
     * Получить количество библиотечных наборов минус фраз, принадлежащих клиенту
     *
     * @param shard    - шард
     * @param clientId - ID клиента
     * @return - количество библиотечных наборов
     */
    public int getLibraryPacksCount(int shard, ClientId clientId) {
        return ppcDslContextProvider.ppc(shard)
                .selectCount()
                .from(MINUS_WORDS)
                .where(MINUS_WORDS.CLIENT_ID.eq(clientId.asLong()))
                .and(MINUS_WORDS.IS_LIBRARY.eq(TRUE))
                .fetchOne(0, int.class);
    }

    /**
     * Получить список минус фраз
     *
     * @param shard    шард
     * @param clientId ID клиента
     * @param mwIds    Набор ID минус фраз
     */
    public List<MinusKeywordsPack> get(int shard, ClientId clientId, Collection<Long> mwIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(MINUS_WORDS_MAPPER.getFieldsToRead())
                .from(MINUS_WORDS)
                .where(MINUS_WORDS.MW_ID.in(mwIds))
                .and(MINUS_WORDS.CLIENT_ID.eq(clientId.asLong()))
                .fetch(MINUS_WORDS_MAPPER::fromDb);
    }

    /**
     * Получить библиотечные наборы клиента вместе с количеством привязанных групп.
     * Необязательный фильтр: список идентификаторов наборов
     * Результат сортируется по mw_name.
     *
     * @param clientId    - клиент
     * @param packIdIn    - идентификаторы наборов
     * @param limitOffset - ограничения на список результатов (нужно для внешнего апи)
     * @return - библиотечные наборы с количеством привязанных групп
     */
    public List<Pair<MinusKeywordsPack, Long>> getLibraryPacksWithLinksCount(int shard, ClientId clientId,
                                                                             @Nullable Set<Long> packIdIn,
                                                                             LimitOffset limitOffset) {
        final String linkedAdGroupCounterAlias = "PID_COUNTER";

        SelectConditionStep<Record> step = ppcDslContextProvider.ppc(shard)
                .select(MINUS_WORDS_MAPPER.getFieldsToRead())
                .select(count(ADGROUPS_MINUS_WORDS.PID).as(linkedAdGroupCounterAlias))
                .from(MINUS_WORDS)
                .leftJoin(ADGROUPS_MINUS_WORDS).on(MINUS_WORDS.MW_ID.eq(ADGROUPS_MINUS_WORDS.MW_ID))
                .where(MINUS_WORDS.CLIENT_ID.eq(clientId.asLong()))
                .and(MINUS_WORDS.IS_LIBRARY.eq(TRUE));

        if (packIdIn != null) {
            step.and(MINUS_WORDS.MW_ID.in(packIdIn));
        }

        return step.groupBy(MINUS_WORDS.MW_ID)
                .orderBy(MINUS_WORDS.MW_NAME)
                .offset(limitOffset.offset()).limit(limitOffset.limit())
                .fetch()
                .map(r -> Pair.of(MINUS_WORDS_MAPPER.fromDb(r), r.get(linkedAdGroupCounterAlias, Long.class)));
    }

    public Map<Long, MinusKeywordsPack> getMinusKeywordsPacks(int shard, ClientId clientId, Collection<Long> ids) {
        return ppcDslContextProvider.ppc(shard)
                .select(MINUS_WORDS_MAPPER.getFieldsToRead())
                .from(MINUS_WORDS)
                .where(MINUS_WORDS.CLIENT_ID.eq(clientId.asLong()))
                .and(MINUS_WORDS.MW_ID.in(ids))
                .fetchMap(MINUS_WORDS.MW_ID, MINUS_WORDS_MAPPER::fromDb);
    }

    public Map<ImmutableList<String>, MinusKeywordsPack> getLibraryMinusKeywordsPacksByKeywords(
            int shard,
            ClientId clientId,
            List<ImmutableList<String>> minusKeywords) {
        Map<BigInteger, ImmutableList<String>> keywordsByHash = StreamEx.of(minusKeywords).toMap(
                MinusKeywordsPackUtils::calcHash,
                Function.identity(),
                (mk1, mk2) -> mk1);

        Map<BigInteger, MinusKeywordsPack> existingPacks =
                getExistingMinusKeywordsByHashes(shard, clientId, keywordsByHash.keySet(), true);

        return EntryStream.of(existingPacks).mapKeys(keywordsByHash::get).toMap();
    }

    public Map<Long, MinusKeywordsPack> getMinusKeywordsByAdGroupIds(int shard, List<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(MINUS_WORDS_MAPPER.getFieldsToRead())
                .select(PHRASES.PID)
                .from(PHRASES)
                .join(MINUS_WORDS).on(MINUS_WORDS.MW_ID.eq(PHRASES.MW_ID))
                .where(PHRASES.PID.in(adGroupIds))
                .fetchMap(PHRASES.PID, MINUS_WORDS_MAPPER::fromDb);
    }

    /**
     * Получить группы (и их кампании), к которым привязаны библиотечные наборы минус фраз {@code mwIds}
     *
     * @param shard шард
     * @param mwIds Набор ID минус фраз
     * @return ключ - id группы, значение - id кампании
     */
    public Map<Long, Long> getLinkedAdGroupIdToCampaignIdMap(int shard, Collection<Long> mwIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID, PHRASES.CID)
                .from(ADGROUPS_MINUS_WORDS)
                .join(PHRASES).on(PHRASES.PID.eq(ADGROUPS_MINUS_WORDS.PID))
                .where(ADGROUPS_MINUS_WORDS.MW_ID.in(mwIds))
                .fetchMap(PHRASES.PID, PHRASES.CID);
    }

    /**
     * Получить группы, к которым привязаны библиотечные наборы минус фраз {@code mwIds}
     *
     * @param shard               шард
     * @param clientId            id клиента
     * @param mwIds               набор ID минус фраз
     * @param packNameContains    фильтр по имени: имя набора должно содержать подстроку (без учета регистра)
     * @param packNameNotContains фильтр по имени: имя набора не должно содержать подстроку (без учета регистра)
     * @return список идентификаторов групп
     */
    public Set<Long> getLinkedAdGroupIds(int shard, ClientId clientId,
                                         @Nullable Collection<Long> mwIds,
                                         @Nullable String packNameContains,
                                         @Nullable String packNameNotContains) {
        SelectConditionStep<Record1<Long>> conditionStep = ppcDslContextProvider.ppc(shard)
                .select(ADGROUPS_MINUS_WORDS.PID)
                .from(ADGROUPS_MINUS_WORDS)
                .join(MINUS_WORDS).on(MINUS_WORDS.MW_ID.eq(ADGROUPS_MINUS_WORDS.MW_ID))
                .where(MINUS_WORDS.CLIENT_ID.eq(clientId.asLong()))
                .and(MINUS_WORDS.IS_LIBRARY.isTrue());

        if (mwIds != null) {
            conditionStep = conditionStep.and(ADGROUPS_MINUS_WORDS.MW_ID.in(mwIds));
        }
        if (packNameContains != null) {
            conditionStep = conditionStep.and(MINUS_WORDS.MW_NAME.containsIgnoreCase(packNameContains));
        }
        if (packNameNotContains != null) {
            conditionStep = conditionStep.and(MINUS_WORDS.MW_NAME.notContainsIgnoreCase(packNameNotContains));
        }

        return conditionStep
                .fetchSet(ADGROUPS_MINUS_WORDS.PID);
    }

    /**
     * Получить кампании, к которым привязаны библиотечные наборы минус фраз {@code mwIds}
     *
     * @param shard шард
     * @param mwIds Набор ID минус фраз
     * @return ключ - id кампании, значение - тип кампании
     */
    public Map<Long, CampaignType> getLinkedCampaignIdToTypeMap(int shard, Collection<Long> mwIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS.TYPE)
                .from(CAMPAIGNS_MINUS_WORDS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(CAMPAIGNS_MINUS_WORDS.CID))
                .where(CAMPAIGNS_MINUS_WORDS.MW_ID.in(mwIds))
                .fetchMap(CAMPAIGNS.CID, r -> CampaignType.fromSource(r.get(CAMPAIGNS.TYPE)));
    }

    public void update(Configuration conf, Collection<AppliedChanges<MinusKeywordsPack>> appliedChanges) {
        appliedChanges.forEach(ac -> {
            List<String> words = ac.getNewValue(MinusKeywordsPack.MINUS_KEYWORDS);
            ac.modify(MinusKeywordsPack.HASH, !isEmpty(words) ? calcHash(words) : null);
        });
        JooqUpdateBuilder<MinusWordsRecord, MinusKeywordsPack> updateBuilder =
                new JooqUpdateBuilder<>(MINUS_WORDS.MW_ID, appliedChanges);

        updateBuilder.processProperty(MinusKeywordsPack.NAME, MINUS_WORDS.MW_NAME);
        updateBuilder.processProperty(MinusKeywordsPack.MINUS_KEYWORDS, MINUS_WORDS.MW_TEXT,
                MinusKeywordsPackUtils::minusKeywordsToJson);
        updateBuilder.processProperty(MinusKeywordsPack.HASH, MINUS_WORDS.MW_HASH, ULong::valueOf);

        DSL.using(conf)
                .update(MINUS_WORDS)
                .set(updateBuilder.getValues())
                .where(MINUS_WORDS.MW_ID.in(updateBuilder.getChangedIds()))
                .execute();
    }

    /**
     * Создать библиотечные наборы минус-фраз (без переиспользования).
     *
     * @param shard              шард
     * @param clientId           клиент
     * @param minusKeywordsPacks список минус-фраз
     * @return список id сохраненных наборов в порядке соответсвующем minusKeywordsPacks
     */
    public List<Long> createLibraryMinusKeywords(int shard, ClientId clientId,
                                                 List<MinusKeywordsPack> minusKeywordsPacks) {
        InsertHelper<MinusWordsRecord> insertHelper = new InsertHelper<>(ppcDslContextProvider.ppc(shard), MINUS_WORDS);

        List<Long> generatedIds = shardHelper.generateMinusWordsIds(minusKeywordsPacks.size());
        for (int i = 0; i < minusKeywordsPacks.size(); i++) {
            MinusKeywordsPack pack = minusKeywordsPacks.get(i);
            pack.setId(generatedIds.get(i));
            pack.setHash(calcHash(pack.getMinusKeywords()));

            insertHelper
                    .add(MINUS_WORDS_MAPPER, pack)
                    .set(MINUS_WORDS.CLIENT_ID, clientId.asLong())
                    .newRecord();
        }

        if (insertHelper.hasAddedRecords()) {
            insertHelper.execute();
        }

        return generatedIds;
    }

    /**
     * Создать "частные" наборы MinusKeywordsPack с переиспользованием существующих "частных" наборов
     */
    public List<Long> createPrivateMinusKeywords(int shard, ClientId clientId, List<MinusKeywordsPack> packs) {
        InsertHelper<MinusWordsRecord> insertHelper = new InsertHelper<>(ppcDslContextProvider.ppc(shard), MINUS_WORDS);

        Set<BigInteger> hashes = new HashSet<>(packs.size());
        packs.forEach(p -> {
            BigInteger hash = calcHash(p.getMinusKeywords());
            p.setHash(hash);
            hashes.add(hash);
        });

        Map<BigInteger, MinusKeywordsPack> existingPrivatePacks =
                getExistingMinusKeywordsByHashes(shard, clientId, hashes, false);
        int idCount = hashes.size() - existingPrivatePacks.size();
        Iterator<Long> ids = shardHelper.generateMinusWordsIds(idCount).iterator();

        for (MinusKeywordsPack pack : packs) {
            MinusKeywordsPack existing = existingPrivatePacks.get(pack.getHash());
            if (existing != null) {
                pack.setId(existing.getId());
                continue;
            }

            pack.setId(ids.next());
            existingPrivatePacks.put(pack.getHash(), pack);

            insertHelper
                    .add(MINUS_WORDS_MAPPER, pack)
                    .set(MINUS_WORDS.CLIENT_ID, clientId.asLong())
                    .newRecord();
        }
        if (insertHelper.hasAddedRecords()) {
            insertHelper.execute();
        }

        return mapList(packs, MinusKeywordsPack::getId);
    }

    /**
     * Возвращает существующие наборы минус-слов по их хешам.
     * Если есть несколько одинаковых наборов (по тексту фраз), в итоговой мапе вернется набор с наименьшим id.
     *
     * @param isLibrary отбирать библиотечные или частные наборы
     */
    private Map<BigInteger, MinusKeywordsPack> getExistingMinusKeywordsByHashes(
            int shard, ClientId clientId, Collection<BigInteger> hashes, boolean isLibrary) {
        Result<Record> result = ppcDslContextProvider.ppc(shard)
                .select(MINUS_WORDS_MAPPER.getFieldsToRead())
                .from(MINUS_WORDS)
                .where(MINUS_WORDS.CLIENT_ID.eq(clientId.asLong()))
                .and(MINUS_WORDS.MW_HASH.in(hashes))
                .and(MINUS_WORDS.IS_LIBRARY.eq(booleanToLong(isLibrary)))
                .fetch();
        return StreamEx.of(result).toMap(
                rec -> rec.getValue(MINUS_WORDS.MW_HASH).toBigInteger(),
                MINUS_WORDS_MAPPER::fromDb,
                (oldValue, newValue) -> oldValue.getId() < newValue.getId() ? oldValue : newValue);
    }


    /**
     * Удаляет библиотечные наборы минус фраз по id.
     * Если переданные наборы привязаны к какой-нибудь группе/кампании или наборы принадлежат другому клиенту
     * или какой-то набор не существует, то будет брошено IllegalStateException и ни один набор не будет удален.
     *
     * @param shard    - шард
     * @param clientId - клиент
     * @param packIds  - идентификаторы библиотечных наборов минус-фраз
     * @throws IllegalStateException - если набор привязан, не существует или принадлежит другому клиенту.
     */
    public void delete(int shard, ClientId clientId, List<Long> packIds) {
        ppcDslContextProvider.ppc(shard).transaction(conf -> {
            tryLockMinusKeywordsPacks(conf, clientId, packIds);

            Set<Long> linkedIds = getLibraryPackIdsWithLinks(conf, packIds);
            checkState(linkedIds.isEmpty(),
                    "Minus keyword packs with ids [%s] linked to adGroup or campaign", linkedIds);

            DSL.using(conf)
                    .deleteFrom(MINUS_WORDS)
                    .where(MINUS_WORDS.MW_ID.in(packIds))
                    .execute();
        });
    }

    /**
     * Возвращает id, которые имеют связь с группами и/или кампаниями (без Configuration)
     */
    public Set<Long> getLibraryPackIdsWithLinks(int shard, Collection<Long> minusKeywordsPackIds) {
        Configuration configuration = ppcDslContextProvider.ppc(shard).configuration();
        return getLibraryPackIdsWithLinks(configuration, minusKeywordsPackIds);
    }

    /**
     * Возвращает id, которые имеют связь с группами и/или кампаниями
     */
    private Set<Long> getLibraryPackIdsWithLinks(Configuration conf, Collection<Long> minusKeywordsPackIds) {
        Set<Long> mwIdsWithLinks = DSL.using(conf)
                .select(ADGROUPS_MINUS_WORDS.MW_ID)
                .from(ADGROUPS_MINUS_WORDS)
                .where(ADGROUPS_MINUS_WORDS.MW_ID.in(minusKeywordsPackIds)).fetchSet(ADGROUPS_MINUS_WORDS.MW_ID);
        Set<Long> mwIdsLinkedToCampaigns = DSL.using(conf)
                .select(CAMPAIGNS_MINUS_WORDS.MW_ID)
                .from(CAMPAIGNS_MINUS_WORDS)
                .where(CAMPAIGNS_MINUS_WORDS.MW_ID.in(minusKeywordsPackIds)).fetchSet(CAMPAIGNS_MINUS_WORDS.MW_ID);
        mwIdsWithLinks.addAll(mwIdsLinkedToCampaigns);
        return mwIdsWithLinks;
    }

    /**
     * Берет лок на переданные библиотечные наборы заданного клиента.
     * Бросает exception, если взять лок не удалось по причине отсутствия элементов в бд.
     *
     * @param conf                 jooq кофиг
     * @param clientId             id клиента
     * @param minusKeywordsPackIds список id наборов минус-фраз
     */
    public void tryLockMinusKeywordsPacks(Configuration conf, ClientId clientId,
                                          Collection<Long> minusKeywordsPackIds) {
        Set<Long> allPackIds = new HashSet<>(minusKeywordsPackIds);
        Set<Long> lockedIds = lockMinusKeywordsPacks(conf, clientId, allPackIds);
        checkState(allPackIds.equals(lockedIds),
                "Some minus keyword packs don't exist at the time of taking the lock (locked only %s of %s)",
                lockedIds, allPackIds);
    }

    /**
     * Берет лок на переданные библиотечные наборы заданного клиента. Возвращает id залоченных наборов
     */
    private Set<Long> lockMinusKeywordsPacks(Configuration conf, ClientId clientId,
                                             Collection<Long> minusKeywordsPackIds) {
        return DSL.using(conf)
                .select(MINUS_WORDS.MW_ID)
                .from(MINUS_WORDS)
                .where(MINUS_WORDS.MW_ID.in(minusKeywordsPackIds)
                        .and(MINUS_WORDS.IS_LIBRARY.eq(TRUE))
                        .and(MINUS_WORDS.CLIENT_ID.eq(clientId.asLong())))
                .forUpdate()
                .fetchSet(MINUS_WORDS.MW_ID);
    }

    /**
     * Возвращает все привязанные библиотечные наборы минус фраз для групп
     *
     * @param adGroupIds id групп, для которых надо получить привязанные наборы минус фраз
     * @return маппинг id группы на список id наборов минус фраз
     */
    public Map<Long, List<Long>> getAdGroupsLibraryMinusKeywordsPacks(int shard, Collection<Long> adGroupIds) {
        Map<Long, List<Long>> adGroupsWithPacks = ppcDslContextProvider.ppc(shard)
                .select(ADGROUPS_MINUS_WORDS.PID, ADGROUPS_MINUS_WORDS.MW_ID)
                .from(ADGROUPS_MINUS_WORDS)
                .where(ADGROUPS_MINUS_WORDS.PID.in(adGroupIds))
                .orderBy(ADGROUPS_MINUS_WORDS.PID, ADGROUPS_MINUS_WORDS.MW_ID)
                .fetchGroups(ADGROUPS_MINUS_WORDS.PID, ADGROUPS_MINUS_WORDS.MW_ID);
        adGroupIds.forEach(adGroupId -> adGroupsWithPacks.putIfAbsent(adGroupId, emptyList()));
        return adGroupsWithPacks;
    }

    /**
     * Возвращает все привязанные библиотечные наборы минус фраз для кампаний
     *
     * @param campaignIds id кампаний, для которых надо получить привязанные наборы минус фраз
     * @return маппинг id кампании на список id наборов минус фраз
     */
    public Map<Long, List<Long>> getCampaignsLibraryMinusKeywordsPacks(int shard, Collection<Long> campaignIds) {
        Map<Long, List<Long>> campaignsWithPacks = ppcDslContextProvider.ppc(shard)
                .select(CAMPAIGNS_MINUS_WORDS.CID, CAMPAIGNS_MINUS_WORDS.MW_ID)
                .from(CAMPAIGNS_MINUS_WORDS)
                .where(CAMPAIGNS_MINUS_WORDS.CID.in(campaignIds))
                .orderBy(CAMPAIGNS_MINUS_WORDS.CID, CAMPAIGNS_MINUS_WORDS.MW_ID)
                .fetchGroups(CAMPAIGNS_MINUS_WORDS.CID, CAMPAIGNS_MINUS_WORDS.MW_ID);
        campaignIds.forEach(campaignId -> campaignsWithPacks.putIfAbsent(campaignId, emptyList()));
        return campaignsWithPacks;
    }

    /**
     * Привязывает библиотечные наборы минус фраз к группе
     *
     * @param adGroupMinusKeywordsPacksIds маппинг id группы на список id привязываемых наборов минус фраз
     */
    public void addAdGroupToPackLinks(Configuration config,
                                      Map<Long, ? extends Collection<Long>> adGroupMinusKeywordsPacksIds) {
        InsertValuesStep2<AdgroupsMinusWordsRecord, Long, Long> insertStep = config.dsl()
                .insertInto(ADGROUPS_MINUS_WORDS)
                .columns(ADGROUPS_MINUS_WORDS.PID, ADGROUPS_MINUS_WORDS.MW_ID);
        adGroupMinusKeywordsPacksIds.forEach((adGroupId, mkPackIds) ->
                mkPackIds.forEach(mkPackId -> insertStep.values(adGroupId, mkPackId)));
        insertStep.execute();
    }

    /**
     * Удалить все привязки наборов минус-фраз к переданным группам
     *
     * @param adGroupIds - идентификаторы групп, для которых будут удалены связи с наборами минус-фраз.
     */
    public void deleteAdGroupToPackLinks(Configuration conf, List<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }
        DSL.using(conf)
                .deleteFrom(ADGROUPS_MINUS_WORDS)
                .where(ADGROUPS_MINUS_WORDS.PID.in(adGroupIds))
                .execute();
    }

    /**
     * Удалить все привязки наборов минус-фраз к переданным группам
     *
     * @param adGroupIds - идентификаторы групп, для которых будут удалены связи с наборами минус-фраз.
     */
    public void deleteAdGroupToPackLinks(int shard, List<Long> adGroupIds) {
        Configuration configuration = ppcDslContextProvider.ppc(shard).configuration();
        deleteAdGroupToPackLinks(configuration, adGroupIds);
    }

}
