package ru.yandex.direct.core.entity.minuskeywordspack.service;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.minuskeywordspack.container.AddedMinusKeywordsPackInfo;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.tree.SubOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class AddMinusKeywordsPackSubOperation implements SubOperation<List<String>> {

    private MinusKeywordsPacksAddOperation packsAddOperation;

    /**
     * Наборы с какими индексами надо добавить в базу. Не все надо добавлять, когда, например,
     * вышележащая сущность невалидна, то есть не будет добавлена в базу, соответственно, набор не будет к ней привязан.
     */
    private Set<Integer> indexesToApply;
    /**
     * Id добавленных наборов для привязки к вышележащей сущности
     */
    private List<Long> minusKeywordsPackIds;

    public AddMinusKeywordsPackSubOperation(Applicability applicability, List<List<String>> minusKeywords,
                                            MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
                                            MinusKeywordsPacksAddOperationFactory minusKeywordsPackSubOperationFactory,
                                            ClientId clientId, int shard) {
        List<MinusKeywordsPack> minusKeywordsPacks = mapList(minusKeywords, this::modelFromMinusWords);
        this.packsAddOperation = minusKeywordsPackSubOperationFactory
                .newInstance(applicability, minusKeywordsPacks, minusPhraseValidationMode, clientId, shard, true);
    }

    /**
     * Создает модель набора минус фраз для операции
     */
    private MinusKeywordsPack modelFromMinusWords(List<String> minusKeywords) {
        return new MinusKeywordsPack().withMinusKeywords(minusKeywords);
    }

    /**
     * Валидируем наборы при помощи операции и переносим ошибки валидации с внутреннего поля модели на элементы
     * результирующего списка.
     */
    @Override
    public ValidationResult<List<List<String>>, Defect> prepare() {
        packsAddOperation.prepare();
        ValidationResult<List<MinusKeywordsPack>, Defect> operationPrepareResult =
                packsAddOperation.getValidationResult();
        return operationPrepareResult.transformUnchecked(new MinusKeywordsPackValidationTransformer());
    }

    public void setIndexesToApply(Set<Integer> indexesToApply) {
        this.indexesToApply = indexesToApply;
    }

    @Override
    public void apply() {
        checkNotNull(indexesToApply, "Indexes to apply should be set");
        if (packsAddOperation.getResult().isPresent()) {
            minusKeywordsPackIds = emptyList();
            return;
        }
        MassResult<AddedMinusKeywordsPackInfo> result = packsAddOperation.apply(indexesToApply);
        minusKeywordsPackIds =
                mapList(result.getResult(), res -> ifNotNull(res.getResult(), AddedMinusKeywordsPackInfo::getId));
    }

    public List<Long> getMinusKeywordPackIds() {
        return minusKeywordsPackIds;
    }

    @ParametersAreNonnullByDefault
    private static class MinusKeywordsPackValidationTransformer implements ValidationResult.ValidationResultTransformer<Defect> {
        @SuppressWarnings("unchecked")
        @Override
        public <OV> List<List<String>> transformValue(Path path, @Nullable OV oldValue) {
            List<MinusKeywordsPack> minusKeywords = (List<MinusKeywordsPack>) oldValue;
            return mapList(minusKeywords, MinusKeywordsPack::getMinusKeywords);
        }

        @Override
        public Map<PathNode, ValidationResult<?, Defect>> transformSubResults(Path path,
                                                                              Map<PathNode, ValidationResult<?, Defect>> subResults) {
            PathNode minusKeywordsPathNode = new PathNode.Field(MinusKeywordsPack.MINUS_KEYWORDS.name());
            boolean noErrorsBesidesMinusKeywords = StreamEx.of(subResults.values())
                    .flatMap(subResult -> EntryStream.of(subResult.getSubResults()))
                    .noneMatch(entry -> !entry.getKey().equals(minusKeywordsPathNode) && entry.getValue().hasErrors());
            checkState(noErrorsBesidesMinusKeywords, "Any errors besides minus keywords list are unexpected");
            return EntryStream.of(subResults)
                    .<ValidationResult<?, Defect>>mapValues(
                            subResult -> subResult.getSubResults().get(minusKeywordsPathNode))
                    .toMap();
        }
    }
}
