package ru.yandex.direct.core.entity.minuskeywordspack.service;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Configuration;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.StatusShowsForecast;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.MinusKeywordPreparingTool;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.minuskeywordspack.container.UpdatedMinusKeywordsPackInfo;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.minuskeywordspack.service.validation.UpdateMinusKeywordsPackValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ModelChangesValidatedStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.model.AppliedChanges.isChanged;
import static ru.yandex.direct.model.ModelChanges.isPropertyChanged;
import static ru.yandex.direct.model.ModelChanges.propertyModifier;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция обновления библиотечного элемента минус фраз (частные наборы обновлению не подлежат)
 */
public class MinusKeywordsPacksUpdateOperation extends SimpleAbstractUpdateOperation<MinusKeywordsPack,
        UpdatedMinusKeywordsPackInfo> {

    private final DslContextProvider dslContextProvider;
    private final MinusKeywordsPackRepository minusKeywordsPackRepository;
    private final MinusKeywordPreparingTool minusKeywordPreparingTool;
    private final UpdateMinusKeywordsPackValidationService validationService;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final BannerCommonRepository bannerCommonRepository;

    private final MinusPhraseValidator.ValidationMode minusPhraseValidationMode;

    private final ClientId clientId;
    private final int shard;

    public MinusKeywordsPacksUpdateOperation(Applicability applicability,
                                             List<ModelChanges<MinusKeywordsPack>> modelChanges,
                                             DslContextProvider dslContextProvider,
                                             MinusKeywordsPackRepository minusKeywordsPackRepository,
                                             MinusKeywordPreparingTool minusKeywordPreparingTool,
                                             UpdateMinusKeywordsPackValidationService validationService,
                                             AdGroupRepository adGroupRepository,
                                             CampaignRepository campaignRepository,
                                             BannerCommonRepository bannerCommonRepository,
                                             MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
                                             ClientId clientId,
                                             int shard) {
        super(applicability, modelChanges, id -> new MinusKeywordsPack().withId(id));

        this.dslContextProvider = dslContextProvider;
        this.minusKeywordsPackRepository = minusKeywordsPackRepository;
        this.minusKeywordPreparingTool = minusKeywordPreparingTool;
        this.validationService = validationService;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.bannerCommonRepository = bannerCommonRepository;

        this.minusPhraseValidationMode = minusPhraseValidationMode;
        this.clientId = clientId;
        this.shard = shard;
    }


    @Override
    protected ValidationResult<List<ModelChanges<MinusKeywordsPack>>, Defect> validateModelChanges(
            List<ModelChanges<MinusKeywordsPack>> modelChanges) {
        return validationService.preValidate(modelChanges, minusPhraseValidationMode, clientId, shard);
    }

    @Override
    protected void onModelChangesValidated(ModelChangesValidatedStep<MinusKeywordsPack> modelChangesValidatedStep) {
        prepareMinusKeywordsForSaving(modelChangesValidatedStep);
    }

    /**
     * Подготовка минус-фраз в объектах ModelChanges к сохранению.
     * Проводится после первого этапа валидации для объектов ModelChanges без ошибок.
     * Изменяет валидные ModelChanges.
     */
    private void prepareMinusKeywordsForSaving(ModelChangesValidatedStep<MinusKeywordsPack> modelChangesValidatedStep) {
        StreamEx.of(modelChangesValidatedStep.getValidModelChanges())
                .filter(isPropertyChanged(MinusKeywordsPack.MINUS_KEYWORDS))
                .forEach(propertyModifier(MinusKeywordsPack.MINUS_KEYWORDS,
                        minusKeywordPreparingTool::fullPrepareForSaving));
    }

    @Override
    protected Collection<MinusKeywordsPack> getModels(Collection<Long> ids) {
        return minusKeywordsPackRepository.get(shard, clientId, ids);
    }

    @Override
    protected ValidationResult<List<MinusKeywordsPack>, Defect> validateAppliedChanges(
            ValidationResult<List<MinusKeywordsPack>, Defect> validationResult) {
        return validationService.validate(validationResult);
    }

    @Override
    protected List<UpdatedMinusKeywordsPackInfo> execute(
            List<AppliedChanges<MinusKeywordsPack>> applicableAppliedChanges) {
        dslContextProvider.ppcTransaction(shard, conf -> {
            minusKeywordsPackRepository.update(conf, applicableAppliedChanges);
            updateAdditionalData(conf, applicableAppliedChanges);
        });

        return createResultInfo(applicableAppliedChanges);
    }

    /**
     * Изменяет данные у связанных объектов
     * Если к минус фразам привязана группа:
     * 1. сбрасываем у группы статус синхронизации с бк, statusShowsForecast и lastChange
     * 2. у кампании устанавливаем autobudgetForecastDate в null
     * <p>
     * Если к минус фразам привязана кампания:
     * 1. у кампании сбрасывает статус синхронизации с бк
     * 2. у кампании устанавливаем autobudgetForecastDate в null
     * 3. у группы сбрасываем statusShowsForecast
     * 4. у динамических и перформанс баннеров сбрасываем статус синхронизации с бк
     */
    private void updateAdditionalData(Configuration conf,
                                      Collection<AppliedChanges<MinusKeywordsPack>> appliedChanges) {
        Set<Long> minusKeywordsIdsToSync = StreamEx.of(appliedChanges)
                .filter(isChanged(MinusKeywordsPack.MINUS_KEYWORDS))
                .map(ac -> ac.getModel().getId())
                .toSet();

        if (minusKeywordsIdsToSync.isEmpty()) {
            return;
        }

        Map<Long, Long> adGroupIdToCampaignId =
                minusKeywordsPackRepository.getLinkedAdGroupIdToCampaignIdMap(shard, minusKeywordsIdsToSync);
        Map<Long, CampaignType> linkedCampaignIdToType =
                minusKeywordsPackRepository.getLinkedCampaignIdToTypeMap(shard, minusKeywordsIdsToSync);
        Set<Long> campaignIdsToResetForecast = new HashSet<>(adGroupIdToCampaignId.values());
        campaignIdsToResetForecast.addAll(linkedCampaignIdToType.keySet());

        adGroupRepository.setSignificantlyChangedCoverageState(conf, adGroupIdToCampaignId.keySet());
        campaignRepository.setAutobudgetForecastDate(conf, campaignIdsToResetForecast, null);

        updateAdditionalDataLinkedCampaigns(conf, linkedCampaignIdToType);
    }

    private void updateAdditionalDataLinkedCampaigns(Configuration conf,
                                                     Map<Long, CampaignType> linkedCampaignIdToType) {
        Set<Long> campaignsToDropBannersStatusBsSynced = EntryStream.of(linkedCampaignIdToType)
                .filterValues(type -> type.equals(CampaignType.DYNAMIC)
                        || type.equals(CampaignType.PERFORMANCE))
                .keys()
                .toSet();
        campaignRepository.updateStatusBsSyncedWithLastChange(conf, linkedCampaignIdToType.keySet(), StatusBsSynced.NO);
        bannerCommonRepository.updateStatusBsSyncedByCampaignIds(conf, campaignsToDropBannersStatusBsSynced,
                false, StatusBsSynced.NO);
        adGroupRepository.updateStatusShowsForecastByCampaignIds(conf, linkedCampaignIdToType.keySet(),
                StatusShowsForecast.NEW);
    }

    private List<UpdatedMinusKeywordsPackInfo> createResultInfo(
            List<AppliedChanges<MinusKeywordsPack>> appliedChanges) {
        return mapList(appliedChanges, changes -> {
            MinusKeywordsPack minusKeywordsPack = changes.getModel();
            return new UpdatedMinusKeywordsPackInfo()
                    .withId(minusKeywordsPack.getId())
                    .withName(minusKeywordsPack.getName())
                    .withMinusKeywords(minusKeywordsPack.getMinusKeywords());
        });
    }
}
