package ru.yandex.direct.core.entity.minuskeywordspack.service.validation;

import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseBeforeNormalizationValidator.minusKeywordsAreValidBeforeNormalization;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.GROUP_MINUS_KEYWORDS_MAX_LENGTH_BEFORE_NORMALIZATION;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.maxLibraryPacksCount;

@Service
public class AddMinusKeywordsPackValidationService {

    private final MinusKeywordsPackRepository packRepository;
    private final MinusKeywordsPackValidationService packValidationService;

    @Autowired
    public AddMinusKeywordsPackValidationService(
            MinusKeywordsPackRepository packRepository,
            MinusKeywordsPackValidationService packValidationService) {
        this.packRepository = packRepository;
        this.packValidationService = packValidationService;
    }

    public ValidationResult<List<MinusKeywordsPack>, Defect> preValidate(
            List<MinusKeywordsPack> minusKeywordsPacks,
            MinusPhraseValidator.ValidationMode validationMode,
            boolean isPrivatePacks, int shard, ClientId clientId) {
        return ListValidationBuilder.of(minusKeywordsPacks, Defect.class)
                .checkEachBy(pack -> preValidateMinusKeywordsPack(pack, validationMode))
                .checkBy(packs -> checkLibraryPacksCount(shard, clientId, packs), When.isTrue(!isPrivatePacks))
                .getResult();
    }

    private ValidationResult<MinusKeywordsPack, Defect> preValidateMinusKeywordsPack(
            MinusKeywordsPack pack,
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode) {
        ModelItemValidationBuilder<MinusKeywordsPack> vb = ModelItemValidationBuilder.of(pack);

        vb.item(MinusKeywordsPack.MINUS_KEYWORDS)
                .checkBy(minusKeywordsAreValidBeforeNormalization(
                        GROUP_MINUS_KEYWORDS_MAX_LENGTH_BEFORE_NORMALIZATION,
                        minusPhraseValidationMode), When.isValid());

        return vb.getResult();
    }

    /**
     * Валидация количества библиотечных наборов у пользователя (количество существующих наборов в бд + колиество новых)
     */
    private ValidationResult<List<MinusKeywordsPack>, Defect> checkLibraryPacksCount(int shard, ClientId clientId,
                                                                                     List<MinusKeywordsPack> newPacks) {
        int existingCount = packRepository.getLibraryPacksCount(shard, clientId);

        return ListValidationBuilder.of(newPacks, Defect.class)
                .checkEach(maxLibraryPacksCount(newPacks.size() + existingCount))
                .getResult();
    }

    public void validate(ValidationResult<List<MinusKeywordsPack>, Defect> preValidationResult, boolean isPrivatePacks) {
        new ListValidationBuilder<>(preValidationResult)
                .checkEachBy(pack -> packValidationService.validatePack(pack, isPrivatePacks), When.isValid());
    }
}
