package ru.yandex.direct.core.entity.minuskeywordspack.service.validation;

import java.util.List;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseBeforeNormalizationValidator.minusKeywordsAreValidBeforeNormalization;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.GROUP_MINUS_KEYWORDS_MAX_LENGTH_BEFORE_NORMALIZATION;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseDefects.minusWordsPackNotFound;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
public class UpdateMinusKeywordsPackValidationService {

    private final MinusKeywordsPackRepository minusKeywordsPackRepository;
    private final MinusKeywordsPackValidationService validationService;
    private final ModelChangesValidationTool preValidationTool;

    @Autowired
    public UpdateMinusKeywordsPackValidationService(
            MinusKeywordsPackRepository minusKeywordsPackRepository,
            MinusKeywordsPackValidationService validationService) {
        this.minusKeywordsPackRepository = minusKeywordsPackRepository;
        this.validationService = validationService;
        this.preValidationTool = ModelChangesValidationTool.builder()
                .objectNotFoundDefect(minusWordsPackNotFound())
                .minSize(1)
                .build();
    }

    public ValidationResult<List<ModelChanges<MinusKeywordsPack>>, Defect> preValidate(
            List<ModelChanges<MinusKeywordsPack>> modelChangesList,
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
            ClientId clientId,
            int shard) {
        Set<Long> mwIds = listToSet(modelChangesList, ModelChanges::getId);
        Set<Long> existingIds =
                minusKeywordsPackRepository.getClientExistingLibraryMinusKeywordsPackIds(shard, clientId, mwIds);

        return ListValidationBuilder.of(modelChangesList, Defect.class)
                .checkBy(mcList -> preValidationTool.validateModelChangesList(mcList, existingIds))
                .checkEachBy(minusKeywordsBeforeNormalizationValidator(minusPhraseValidationMode), When.isValid())
                .getResult();
    }

    public ValidationResult<List<MinusKeywordsPack>, Defect> validate(
            ValidationResult<List<MinusKeywordsPack>, Defect> preValidationResult) {
        return new ListValidationBuilder<>(preValidationResult)
                .checkEachBy(pack -> validationService.validatePack(pack, false), When.isValid())
                .getResult();
    }

    private Validator<ModelChanges<MinusKeywordsPack>, Defect> minusKeywordsBeforeNormalizationValidator(
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode) {
        return modelChanges -> {
            ItemValidationBuilder<ModelChanges<MinusKeywordsPack>, Defect> vb =
                    ItemValidationBuilder.of(modelChanges, Defect.class);

            if (modelChanges.isPropChanged(MinusKeywordsPack.MINUS_KEYWORDS)) {
                vb.item(modelChanges.getChangedProp(MinusKeywordsPack.MINUS_KEYWORDS),
                        MinusKeywordsPack.MINUS_KEYWORDS.name())
                        .checkBy(minusKeywordsAreValidBeforeNormalization(
                                GROUP_MINUS_KEYWORDS_MAX_LENGTH_BEFORE_NORMALIZATION,
                                minusPhraseValidationMode));
            }

            return vb.getResult();
        };
    }
}
