package ru.yandex.direct.core.entity.mobileapp.repository

import org.jooq.DSLContext
import org.jooq.Record
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.mobileapp.model.SkAdNetworkSlot
import ru.yandex.direct.dbschema.ppcdict.tables.IosSkadnetworkSlots.IOS_SKADNETWORK_SLOTS
import ru.yandex.direct.dbschema.ppcdict.tables.ShardClientId.SHARD_CLIENT_ID
import ru.yandex.direct.dbschema.ppcdict.tables.ShardIncCid.SHARD_INC_CID
import ru.yandex.direct.dbutil.wrapper.DslContextProvider

@Repository
class IosSkAdNetworkSlotRepository(private val dslContextProvider: DslContextProvider) {

    internal fun getAllocatedSlotsByAppBundleId(dslContext: DSLContext, appBundleId: String): Set<Int> {
        return dslContext.select(IOS_SKADNETWORK_SLOTS.SLOT)
            .from(IOS_SKADNETWORK_SLOTS)
            .where(IOS_SKADNETWORK_SLOTS.BUNDLE_ID.eq(appBundleId))
            .fetch(IOS_SKADNETWORK_SLOTS.SLOT)
            .asSequence()
            .map { it.toInt() }
            .toSet()
    }

    internal fun getDeletedCampaignIdsWithSlots(dslContext: DSLContext, offset: Long, limit: Long): List<Long> {
        return dslContext.select(IOS_SKADNETWORK_SLOTS.CID)
            .from(IOS_SKADNETWORK_SLOTS)
            .leftJoin(SHARD_INC_CID).on(SHARD_INC_CID.CID.eq(IOS_SKADNETWORK_SLOTS.CID))
            .leftJoin(SHARD_CLIENT_ID).on(SHARD_CLIENT_ID.CLIENT_ID.eq(SHARD_INC_CID.CLIENT_ID))
            .where(SHARD_CLIENT_ID.SHARD.isNull())
            .orderBy(IOS_SKADNETWORK_SLOTS.CID)
            .offset(offset)
            .limit(limit)
            .fetch(IOS_SKADNETWORK_SLOTS.CID)
    }

    internal fun getCampaignIdsWithSlots(dslContext: DSLContext, shard: Int, offset: Long, limit: Long): List<Long> {
        return dslContext.select(IOS_SKADNETWORK_SLOTS.CID)
            .from(IOS_SKADNETWORK_SLOTS)
            .join(SHARD_INC_CID).on(SHARD_INC_CID.CID.eq(IOS_SKADNETWORK_SLOTS.CID))
            .join(SHARD_CLIENT_ID).on(SHARD_CLIENT_ID.CLIENT_ID.eq(SHARD_INC_CID.CLIENT_ID))
            .where(SHARD_CLIENT_ID.SHARD.eq(shard.toLong()))
            .orderBy(IOS_SKADNETWORK_SLOTS.CID)
            .offset(offset)
            .limit(limit)
            .fetch(IOS_SKADNETWORK_SLOTS.CID)
    }

    fun getSlotsByAppBundleIds(appBundleIds: Collection<String>) =
        getSlotsByAppBundleIds(dslContextProvider.ppcdict(), appBundleIds)

    private fun getSlotsByAppBundleIds(dslContext: DSLContext, appBundleIds: Collection<String>): List<SkAdNetworkSlot> {
        return dslContext.select(SkAdNetworkSlotDbMapper.fieldsToRead)
            .from(IOS_SKADNETWORK_SLOTS)
            .where(IOS_SKADNETWORK_SLOTS.BUNDLE_ID.`in`(appBundleIds))
            .fetch { SkAdNetworkSlotDbMapper.fromRecord(it) }
    }

    internal fun getSlotsByCampaignIds(dslContext: DSLContext, campaignIds: Collection<Long>): List<SkAdNetworkSlot> {
        return dslContext.select(SkAdNetworkSlotDbMapper.fieldsToRead)
            .from(IOS_SKADNETWORK_SLOTS)
            .where(IOS_SKADNETWORK_SLOTS.CID.`in`(campaignIds))
            .fetch { SkAdNetworkSlotDbMapper.fromRecord(it) }
    }

    fun addSlot(dslContext: DSLContext, slot: SkAdNetworkSlot) {
        dslContext.insertInto(IOS_SKADNETWORK_SLOTS)
            .set(IOS_SKADNETWORK_SLOTS.BUNDLE_ID, slot.appBundleId)
            .set(IOS_SKADNETWORK_SLOTS.CID, slot.campaignId)
            .set(IOS_SKADNETWORK_SLOTS.SLOT, slot.slot.toLong())
            .execute()
    }

    internal fun deleteSlotsByCampaignIds(dslContext: DSLContext, campaignIds: Collection<Long>) {
        dslContext.deleteFrom(IOS_SKADNETWORK_SLOTS)
            .where(IOS_SKADNETWORK_SLOTS.CID.`in`(campaignIds))
            .execute()
    }
}

object SkAdNetworkSlotDbMapper {
    val fieldsToRead = setOf(
        IOS_SKADNETWORK_SLOTS.BUNDLE_ID,
        IOS_SKADNETWORK_SLOTS.CID,
        IOS_SKADNETWORK_SLOTS.SLOT
    )

    fun fromRecord(record: Record): SkAdNetworkSlot {
        return SkAdNetworkSlot(
            appBundleId = record.get(IOS_SKADNETWORK_SLOTS.BUNDLE_ID)!!,
            campaignId = record.get(IOS_SKADNETWORK_SLOTS.CID)!!,
            slot = record.get(IOS_SKADNETWORK_SLOTS.SLOT)!!.toInt())
    }
}
