package ru.yandex.direct.core.entity.mobileapp.repository

import org.jooq.impl.DSL
import org.jooq.types.ULong
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.mobileapp.model.MobileGoalConversions
import ru.yandex.direct.grid.schema.yt.Tables.DIRECTMOBILEAPPSTAT_BS
import ru.yandex.direct.ytcomponents.service.MobileAppStatDynContextProvider
import ru.yandex.direct.ytwrapper.YtTableUtils
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL
import ru.yandex.yt.ytclient.wire.UnversionedRowset
import java.time.Instant
import java.time.temporal.ChronoUnit
import ru.yandex.direct.core.entity.mobileapp.model.MobileConversions


// см. https://st.yandex-team.ru/BSDEV-81533
@Repository
class MobileAppConversionStatisticRepository(
    private val dynContextProvider: MobileAppStatDynContextProvider
) {
    companion object {
        private val MOBILE_APP_STAT = DIRECTMOBILEAPPSTAT_BS.`as`("MobileAppStat")
        private val ATTRIBUTED_SUM =
            DSL.sum(MOBILE_APP_STAT.ATTRIBUTED).`as`("AttributedSum")
        private val NOT_ATTRIBUTED_SUM =
            DSL.sum(MOBILE_APP_STAT.NOT_ATTRIBUTED).`as`("NotAttributedSum")
        private val HAS_REVENUE_SUM =
            DSL.sum(MOBILE_APP_STAT.HAS_REVENUE).`as`("HasRevenue")
        private val GOAL_ID = YtTableUtils.aliased(MOBILE_APP_STAT.GOAL_ID)
        private val YANDEX_APP_ID = YtTableUtils.aliased(MOBILE_APP_STAT.YANDEX_APP_ID)
    }

    fun getConversionStats(appId: String, platform: String, goalIds: List<Long>, daysNum: Int): List<MobileGoalConversions> {
        val timestampFrom: Long = Instant.now().minus(daysNum.toLong(), ChronoUnit.DAYS).epochSecond

        val bsGoalIds = goalIds.map { replaceToBs(it) }.toSet()

        val query = YtDSL.ytContext()
            .select(GOAL_ID, ATTRIBUTED_SUM, NOT_ATTRIBUTED_SUM, HAS_REVENUE_SUM)
            .from(MOBILE_APP_STAT)
            .where(MOBILE_APP_STAT.APP_ID.eq(appId))
            .and(MOBILE_APP_STAT.PLATFORM.eq(platform))
            .and(MOBILE_APP_STAT.GOAL_ID.`in`(bsGoalIds))
            .and(MOBILE_APP_STAT.UPDATE_TIME.ge(timestampFrom))
            .groupBy(GOAL_ID)

        val rowset = dynContextProvider.getContext().executeSelect(query)

        val goalIdColIdx = YtTableUtils.findColumnOrThrow(rowset.schema, GOAL_ID)
        val attributedColIdx = YtTableUtils.findColumnOrThrow(rowset.schema, ATTRIBUTED_SUM)
        val notAttributedColIdx = YtTableUtils.findColumnOrThrow(rowset.schema, NOT_ATTRIBUTED_SUM)
        val hasRevenueColIdx = YtTableUtils.findColumnOrThrow(rowset.schema, HAS_REVENUE_SUM)

        return rowset.rows.map {
            MobileGoalConversions(
                goalId = replaceToDirect(it.values[goalIdColIdx].longValue()),
                attributedConversions = it.values[attributedColIdx].longValue(),
                notAttributedConversions = it.values[notAttributedColIdx].longValue(),
                hasRevenueConversions = it.values[hasRevenueColIdx].longValue()
            )
        }
    }

    fun getCommonConversionStats(
        appId: String,
        platform: String,
        appTrackerIds: List<Long>,
        daysNum: Int
    ): MobileConversions? {
        val timestampFrom: Long = Instant.now().minus(daysNum.toLong(), ChronoUnit.DAYS).epochSecond

        val query = YtDSL.ytContext()
            .select(ATTRIBUTED_SUM, NOT_ATTRIBUTED_SUM)
            .from(MOBILE_APP_STAT)
            .where(MOBILE_APP_STAT.APP_ID.eq(appId)
                .and(MOBILE_APP_STAT.PLATFORM.eq(platform))
                .and(MOBILE_APP_STAT.APP_TRACKER_ID.`in`(appTrackerIds))
                .and(MOBILE_APP_STAT.UPDATE_TIME.ge(timestampFrom))
            )
            .groupBy(MOBILE_APP_STAT.APP_ID)

        val rowset = dynContextProvider.getContext().executeSelect(query)

        val attributedColIdx = YtTableUtils.findColumnOrThrow(rowset.schema, ATTRIBUTED_SUM)
        val notAttributedColIdx = YtTableUtils.findColumnOrThrow(rowset.schema, NOT_ATTRIBUTED_SUM)

        return rowset.rows.map {
            MobileConversions(
                attributedConversions = it.values[attributedColIdx].longValue(),
                notAttributedConversions = it.values[notAttributedColIdx].longValue(),
            )
        }.firstOrNull()
    }

    fun getVerifiedAppIds(bundleIds: Collection<String>, mobileAppIds: Collection<Long>): List<Long> {
        val query = YtDSL.ytContext()
            .select(MOBILE_APP_STAT.YANDEX_APP_ID)
            .from(MOBILE_APP_STAT)
            .where(MOBILE_APP_STAT.APP_ID.`in`(bundleIds))
            .and(MOBILE_APP_STAT.PLATFORM.`in`("android", "ios"))
            .and(MOBILE_APP_STAT.YANDEX_APP_ID.`in`(mobileAppIds))

        val rowset: UnversionedRowset = dynContextProvider.getContext().executeSelect(query)
        return rowset.rows.map { e -> e.values[0].longValue() }
    }

    fun getConversionStatsByAppId(
        daysNum: Int
    ): Map<Long, List<MobileGoalConversions>> {
        val timestampFrom: Long = Instant.now().minus(daysNum.toLong(), ChronoUnit.DAYS).epochSecond

        val query = YtDSL.ytContext()
            .select(YANDEX_APP_ID, GOAL_ID, ATTRIBUTED_SUM, NOT_ATTRIBUTED_SUM)
            .from(MOBILE_APP_STAT)
            .where(MOBILE_APP_STAT.UPDATE_TIME.ge(timestampFrom))
            .and(YANDEX_APP_ID.ge(ULong.valueOf(0L)))
            .groupBy(YANDEX_APP_ID, GOAL_ID)

        val rowset = dynContextProvider.getContext().executeSelect(query)

        val appIdColIdx = YtTableUtils.findColumnOrThrow(rowset.schema, YANDEX_APP_ID)
        val goalIdColIdx = YtTableUtils.findColumnOrThrow(rowset.schema, GOAL_ID)
        val attributedColIdx = YtTableUtils.findColumnOrThrow(rowset.schema, ATTRIBUTED_SUM)
        val notAttributedColIdx = YtTableUtils.findColumnOrThrow(rowset.schema, NOT_ATTRIBUTED_SUM)

        return rowset.rows
            .groupBy { it.values[appIdColIdx].longValue() }
            .mapValues {
                it.value.map {
                    MobileGoalConversions(
                        goalId = replaceToDirect(it.values[goalIdColIdx].longValue()),
                        attributedConversions = it.values[attributedColIdx].longValue(),
                        notAttributedConversions = it.values[notAttributedColIdx].longValue(),
                    )
                }
            }
    }
}

// https://st.yandex-team.ru/BSDEV-81533#5fe21babf7b42c0323de4a64
fun replaceToBs(goalId: Long): Long {
    return if (goalId == 4L) {
        5L
    } else {
        goalId
    }
}

fun replaceToDirect(bsGoalId: Long): Long {
    return if (bsGoalId == 5L) {
        4L
    } else {
        bsGoalId
    }
}
