package ru.yandex.direct.core.entity.mobileapp.repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.jooq.types.ULong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.bsexport.resources.model.MobileAppForBsExport;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.domain.repository.DomainRepository;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppDeviceTypeTargeting;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppDisplayedAttribute;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppNetworkTargeting;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppPrimaryAction;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppStoreType;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTracker;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerTrackingSystem;
import ru.yandex.direct.core.entity.mobileapp.model.MobileExternalTrackerEvent;
import ru.yandex.direct.core.entity.mobileapp.model.StoreAppIdWithClientId;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository;
import ru.yandex.direct.core.entity.mobilegoals.MobileAppGoalsAppmetrikaRepository;
import ru.yandex.direct.core.entity.mobilegoals.MobileAppGoalsExternalTrackerRepository;
import ru.yandex.direct.dbschema.ppc.enums.MobileContentOsType;
import ru.yandex.direct.dbschema.ppc.tables.records.MobileAppsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.jooq.impl.DSL.max;
import static org.jooq.impl.DSL.min;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.clientIdProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.convertibleEnumSet;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_APPS;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_APP_TRACKERS;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_CONTENT;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Repository
public class MobileAppRepository {

    private static final JooqMapperWithSupplier<MobileApp> APP_MAPPER = createMobileAppMapper();

    private static final JooqMapperWithSupplier<MobileAppTracker> TRACKER_MAPPER =
            createMobileAppTrackerMapper();

    private static final JooqReaderWithSupplier<MobileAppForBsExport> MOBILE_APP_FOR_BS_EXPORT_READER =
            createMobileAppForBsExportReader();

    private static final int MAX_FETCHED_TRACKERS = 10000;

    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final MobileContentRepository mobileContentRepository;
    private final DomainRepository domainRepository;
    private final MobileAppGoalsExternalTrackerRepository mobileAppGoalsExternalTrackerRepository;
    private final MobileAppGoalsAppmetrikaRepository mobileAppGoalsAppmetrikaRepository;

    @Autowired
    public MobileAppRepository(ShardHelper shardHelper,
                               DslContextProvider dslContextProvider,
                               MobileContentRepository mobileContentRepository,
                               DomainRepository domainRepository,
                               MobileAppGoalsExternalTrackerRepository mobileAppGoalsExternalTrackerRepository,
                               MobileAppGoalsAppmetrikaRepository mobileAppGoalsAppmetrikaRepository) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.mobileContentRepository = mobileContentRepository;
        this.domainRepository = domainRepository;
        this.mobileAppGoalsExternalTrackerRepository = mobileAppGoalsExternalTrackerRepository;
        this.mobileAppGoalsAppmetrikaRepository = mobileAppGoalsAppmetrikaRepository;
    }

    public Long addMobileApp(int shard, MobileApp mobileApp) {
        Long id = shardHelper.generateMobileAppIds(1).get(0);

        mobileApp.setId(id);
        new InsertHelper<>(dslContextProvider.ppc(shard), MOBILE_APPS)
                .add(APP_MAPPER, mobileApp)
                .execute();
        return id;
    }

    public void addMobileApps(DSLContext dslContext, ClientId clientId, List<MobileApp> mobileApps) {
        List<Long> ids = shardHelper.generateMobileAppIds(mobileApps.size());
        StreamEx.of(mobileApps).zipWith(ids.stream())
                .forKeyValue(MobileApp::setId);

        mobileApps.forEach(app -> app.setClientId(clientId));

        new InsertHelper<>(dslContext, MOBILE_APPS)
                .addAll(APP_MAPPER, mobileApps)
                .executeIfRecordsAdded();
    }

    public void addMobileAppTrackers(int shard, Collection<MobileAppTracker> mobileAppTrackers) {
        addMobileAppTrackers(dslContextProvider.ppc(shard), mobileAppTrackers);
    }

    public void addMobileAppTrackers(DSLContext dslContext, Collection<MobileAppTracker> mobileAppTrackers) {
        List<Long> ids = shardHelper.generateMobileAppTrackerIds(mobileAppTrackers.size());
        StreamEx.of(mobileAppTrackers).zipWith(ids.stream())
                .forKeyValue(MobileAppTracker::setId);

        new InsertHelper<>(dslContext, MOBILE_APP_TRACKERS)
                .addAll(TRACKER_MAPPER, mobileAppTrackers)
                .execute();
    }

    /**
     * Удалить все трекеры, связанные с мобильными приложениями, с указанными id
     *
     * @param mobileAppIds id мобильных приложений, трекеры которых следует удалить
     */
    public void deleteAllTrackersOfMobileApp(DSLContext dslContext, Collection<Long> mobileAppIds) {
        dslContext.deleteFrom(MOBILE_APP_TRACKERS)
                .where(MOBILE_APP_TRACKERS.MOBILE_APP_ID.in(mobileAppIds))
                .execute();
    }

    public List<MobileApp> getMobileApps(int shard, @Nullable ClientId clientId,
                                         @Nullable Collection<Long> mobileAppIds) {
        DSLContext dslContext = dslContextProvider.ppc(shard);

        List<MobileApp> result = dslContext
                .select(APP_MAPPER.getFieldsToRead())
                .from(MOBILE_APPS)
                .where(clientId == null ? DSL.trueCondition() : MOBILE_APPS.CLIENT_ID.eq(clientId.asLong()))
                .and(mobileAppIds == null ? DSL.trueCondition() : MOBILE_APPS.MOBILE_APP_ID.in(mobileAppIds))
                .limit(1000)
                .fetch(APP_MAPPER::fromDb);

        List<MobileAppTracker> mobileAppTrackerRows = dslContext
                .select(TRACKER_MAPPER.getFieldsToRead())
                .from(MOBILE_APP_TRACKERS)
                .where(clientId == null ? DSL.trueCondition() : MOBILE_APP_TRACKERS.CLIENT_ID.eq(clientId.asLong()))
                .and(MOBILE_APP_TRACKERS.MOBILE_APP_ID.in(mapList(result, MobileApp::getId)))
                .limit(MAX_FETCHED_TRACKERS + 1)
                .fetch(TRACKER_MAPPER::fromDb);

        Map<Long, List<MobileExternalTrackerEvent>> externalTrackerEventsByAppId =
                mobileAppGoalsExternalTrackerRepository
                        .getEventsByAppIds(dslContext, mapList(result, MobileApp::getId), true)
                        .stream()
                        .collect(Collectors.groupingBy(MobileExternalTrackerEvent::getMobileAppId));

        checkState(mobileAppTrackerRows.size() <= MAX_FETCHED_TRACKERS,
                "too many tracker records in the database, clientId = %s", clientId);

        Map<Long, List<MobileAppTracker>> mobileAppTrackersByAppId = mobileAppTrackerRows.stream()
                .collect(Collectors.groupingBy(MobileAppTracker::getMobileAppId, toList()));

        Map<Long, MobileContent> mobileContentById = mobileContentRepository
                .getMobileContent(shard, mapList(result, MobileApp::getMobileContentId))
                .stream()
                .collect(toMap(MobileContent::getId, identity()));

        List<Long> domainIds = mapList(result, MobileApp::getDomainId);
        List<Domain> domains = domainRepository.getDomainsByIdsFromDict(domainIds);
        Map<Long, Domain> domainsByIds = listToMap(domains, Domain::getId);

        for (MobileApp mobileApp : result) {
            mobileApp.setTrackers(mobileAppTrackersByAppId.getOrDefault(mobileApp.getId(), emptyList()));
            mobileApp.setMobileContent(mobileContentById.get(mobileApp.getMobileContentId()));
            mobileApp.setDomain(ifNotNull(domainsByIds.get(mobileApp.getDomainId()), Domain::getDomain));
            mobileApp.setMobileExternalTrackerEvents(externalTrackerEventsByAppId.getOrDefault(mobileApp.getId(),
                    emptyList()));
            mobileApp.setMobileAppMetrikaEvents(mobileAppGoalsAppmetrikaRepository
                    .getEventsByMobileApps(singleton(mobileApp), true));
        }

        return result;
    }

    public Map<Long, List<MobileAppTracker>> getMobileTrackers(int shard, ClientId clientId,
                                                               Collection<Long> mobileAppIds) {
        DSLContext dslContext = dslContextProvider.ppc(shard);

        List<MobileAppTracker> mobileAppTrackerRows = dslContext
                .select(TRACKER_MAPPER.getFieldsToRead())
                .from(MOBILE_APP_TRACKERS)
                .where(MOBILE_APP_TRACKERS.CLIENT_ID.eq(clientId.asLong()))
                .and(MOBILE_APP_TRACKERS.MOBILE_APP_ID.in(mobileAppIds))
                .limit(MAX_FETCHED_TRACKERS + 1)
                .fetch(TRACKER_MAPPER::fromDb);

        Map<Long, List<MobileAppTracker>> mobileAppTrackersByAppId = mobileAppTrackerRows.stream()
                .collect(Collectors.groupingBy(MobileAppTracker::getMobileAppId, toList()));
        return mobileAppTrackersByAppId;
    }

    public List<MobileApp> getMobileAppsWithoutRelations(int shard, Collection<Long> mobileAppIds) {
        return dslContextProvider.ppc(shard)
                .select(APP_MAPPER.getFieldsToRead())
                .from(MOBILE_APPS)
                .where(MOBILE_APPS.MOBILE_APP_ID.in(mobileAppIds))
                .fetch(APP_MAPPER::fromDb);
    }

    public Set<Long> getMobileAppIds(int shard, ClientId clientId, Collection<Long> mobileAppIds) {
        return dslContextProvider.ppc(shard)
                .select(MOBILE_APPS.MOBILE_APP_ID)
                .from(MOBILE_APPS)
                .where(MOBILE_APPS.CLIENT_ID.eq(clientId.asLong()))
                .and(MOBILE_APPS.MOBILE_APP_ID.in(mobileAppIds))
                .fetchSet(MOBILE_APPS.MOBILE_APP_ID);
    }

    public List<MobileApp> getMobileAppsChunk(int shard, Long firstId, Long limit) {
        return dslContextProvider.ppc(shard)
                .select(APP_MAPPER.getFieldsToRead())
                .from(MOBILE_APPS)
                .where(MOBILE_APPS.MOBILE_APP_ID.gt(firstId))
                .orderBy(MOBILE_APPS.MOBILE_APP_ID.asc())
                .limit(limit)
                .fetch(APP_MAPPER::fromDb);
    }

    public void updateVerification(int shard, List<Long> mobileAppIds, boolean verificationFlag) {
        if (mobileAppIds.isEmpty()) {
            return;
        }

        dslContextProvider.ppc(shard)
                .update(MOBILE_APPS)
                .set(MOBILE_APPS.HAS_IDENTIFIED_POSTBACKS, verificationFlag ? 1L : 0L)
                .where(MOBILE_APPS.MOBILE_APP_ID.in(mobileAppIds))
                .execute();
    }

    public void updateVerificationByStoreAppId(
            int shard,
            List<StoreAppIdWithClientId> storeAppIdsForUpdate,
            boolean verificationFlag,
            MobileContentOsType osType) {
        if (storeAppIdsForUpdate.isEmpty()) {
            return;
        }

        var storeAppIdTableField = osType.equals(MobileContentOsType.iOS)
                ? MOBILE_CONTENT.BUNDLE_ID
                : MOBILE_CONTENT.STORE_CONTENT_ID;

        var rows = storeAppIdsForUpdate.stream()
                .map(b -> row(b.getStoreAppId(), b.getClientId()))
                .collect(toList());

        dslContextProvider.ppc(shard)
                .update(MOBILE_APPS
                        .join(MOBILE_CONTENT)
                        .on(MOBILE_APPS.MOBILE_CONTENT_ID
                                .eq(MOBILE_CONTENT.MOBILE_CONTENT_ID
                                        .cast(MOBILE_APPS.MOBILE_CONTENT_ID.getDataType()))))
                .set(MOBILE_APPS.HAS_IDENTIFIED_POSTBACKS, verificationFlag ? 1L : 0L)
                .where(row(storeAppIdTableField, MOBILE_CONTENT.CLIENT_ID).in(rows))
                .execute();
    }

    public List<StoreAppIdWithClientId> getStoreAppIdsToClientsWithVerification(
            int shard,
            List<Long> mobileAppIds,
            MobileContentOsType osType) {
        var requiredField = osType.equals(MobileContentOsType.iOS)
                ? MOBILE_CONTENT.BUNDLE_ID
                : MOBILE_CONTENT.STORE_CONTENT_ID;

        var query = dslContextProvider.ppc(shard)
                .select(requiredField, MOBILE_APPS.CLIENT_ID)
                .from(MOBILE_CONTENT)
                .join(MOBILE_APPS).on(
                        MOBILE_APPS.MOBILE_CONTENT_ID
                                .eq(MOBILE_CONTENT.MOBILE_CONTENT_ID
                                        .cast(MOBILE_APPS.MOBILE_CONTENT_ID.getDataType())))
                .where(MOBILE_CONTENT.OS_TYPE.eq(osType))
                .and(MOBILE_APPS.MOBILE_APP_ID.in(mobileAppIds))
                .and(requiredField.notEqual(""));

        if (osType.equals(MobileContentOsType.iOS)) {
            query.and(MOBILE_CONTENT.BUNDLE_ID.isNotNull());
        }

        return query.groupBy(requiredField, MOBILE_APPS.CLIENT_ID)
                .having(max(MOBILE_APPS.HAS_IDENTIFIED_POSTBACKS).eq(1L))
                .and(min(MOBILE_APPS.HAS_IDENTIFIED_POSTBACKS).eq(0L))
                .fetch(res -> new StoreAppIdWithClientId(
                        res.get(requiredField), res.get(MOBILE_APPS.CLIENT_ID)));
    }

    /**
     * По списку или другой коллекции mobileContentId получить самый большой id mobileApp,
     * у которого такой mobileContentId.
     *
     * @return Map с mobileContentId в качестве ключа и mobileAppId в качестве значения;
     * если с таким mobileContentId нет никаких приложений, таких данных в структуре не будет
     * (containsKey(mobileContentId) вернёт false)
     */
    public Map<Long, Long> getLatestMobileAppIdsByMobileContentIds(int shard, ClientId clientId,
                                                                   Collection<Long> mobileContentIds) {
        if (mobileContentIds.isEmpty()) {
            return emptyMap();
        }

        Field<Long> maxMobileAppId = DSL.max(MOBILE_APPS.MOBILE_APP_ID);
        return dslContextProvider.ppc(shard)
                .select(MOBILE_APPS.MOBILE_CONTENT_ID, maxMobileAppId)
                .from(MOBILE_APPS)
                .where(MOBILE_APPS.CLIENT_ID.eq(clientId.asLong()))
                .and(MOBILE_APPS.MOBILE_CONTENT_ID.in(mapList(mobileContentIds, ULong::valueOf)))
                .groupBy(MOBILE_APPS.MOBILE_CONTENT_ID)
                .fetchMap(record -> record.get(MOBILE_APPS.MOBILE_CONTENT_ID).longValue(),
                        record -> record.get(maxMobileAppId));
    }

    /**
     * По списку или другой коллекции mobileContentId получить список mobileAppId,
     * у которого такой mobileContentId.
     *
     * @return List с mobileAppId
     */
    public List<Long> getMobileAppIdsByMobileContentIds(int shard, Collection<Long> mobileContentIds) {
        if (mobileContentIds.isEmpty()) {
            return emptyList();
        }

        return dslContextProvider.ppc(shard)
                .select(MOBILE_APPS.MOBILE_APP_ID)
                .from(MOBILE_APPS)
                .where(MOBILE_APPS.MOBILE_CONTENT_ID.in(mapList(mobileContentIds, ULong::valueOf)))
                .fetch(MOBILE_APPS.MOBILE_APP_ID);
    }

    /**
     * Обновить таблицу mobile_apps в БД по данным из переданных AppliedChanges для модели MobileApp
     *
     * @param changes список изменений модели MobileApp, которые нужно записать в БД
     */
    public void updateMobileApps(DSLContext dslContext, List<AppliedChanges<MobileApp>> changes) {
        JooqUpdateBuilder<MobileAppsRecord, MobileApp> ub =
                new JooqUpdateBuilder<>(MOBILE_APPS.MOBILE_APP_ID, changes);

        ub.processProperty(MobileApp.NAME, MOBILE_APPS.NAME);
        ub.processProperty(MobileApp.DISPLAYED_ATTRIBUTES, MOBILE_APPS.DISPLAYED_ATTRIBUTES,
                s -> RepositoryUtils.setToDb(s, MobileAppDisplayedAttribute::getTypedValue));
        ub.processProperty(MobileApp.MINIMAL_OPERATING_SYSTEM_VERSION, MOBILE_APPS.MIN_OS_VERSION);
        ub.processProperty(MobileApp.DEVICE_TYPE_TARGETING, MOBILE_APPS.DEVICE_TYPE_TARGETING,
                s -> RepositoryUtils.setToDb(s, MobileAppDeviceTypeTargeting::getTypedValue));
        ub.processProperty(MobileApp.NETWORK_TARGETING, MOBILE_APPS.NETWORK_TARGETING,
                s -> RepositoryUtils.setToDb(s, MobileAppNetworkTargeting::getTypedValue));
        ub.processProperty(MobileApp.DOMAIN_ID, MOBILE_APPS.DOMAIN_ID);
        ub.processProperty(MobileApp.PRIMARY_ACTION, MOBILE_APPS.PRIMARY_ACTION,
                MobileAppPrimaryAction::toSource);
        ub.processProperty(MobileApp.APP_METRIKA_APPLICATION_ID, MOBILE_APPS.APP_METRIKA_APPLICATION_ID);

        dslContext.update(MOBILE_APPS)
                .set(ub.getValues())
                .where(MOBILE_APPS.MOBILE_APP_ID.in(ub.getChangedIds()))
                .execute();
    }

    /**
     * Получить мобильние приложения для кампаний для экспорта в БК
     */
    public Map<Long, MobileAppForBsExport> getCampaignsMobileAppsForBsExport(int shard,
                                                                             Collection<Long> campaignIds) {
        var fieldsToRead = new ArrayList<>(MOBILE_APP_FOR_BS_EXPORT_READER.getFieldsToRead());
        fieldsToRead.add(CAMPAIGNS_MOBILE_CONTENT.CID);
        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(CAMPAIGNS_MOBILE_CONTENT)
                .join(MOBILE_APPS)
                .on(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID.eq(MOBILE_APPS.MOBILE_APP_ID))
                .where(CAMPAIGNS_MOBILE_CONTENT.CID.in(campaignIds))
                .fetchMap(CAMPAIGNS_MOBILE_CONTENT.CID, MOBILE_APP_FOR_BS_EXPORT_READER::fromDb);
    }

    private static JooqMapperWithSupplier<MobileApp> createMobileAppMapper() {
        return JooqMapperWithSupplierBuilder.builder(MobileApp::new)
                .map(property(MobileApp.ID, MOBILE_APPS.MOBILE_APP_ID))
                .map(clientIdProperty(MobileApp.CLIENT_ID, MOBILE_APPS.CLIENT_ID))
                .map(convertibleProperty(MobileApp.STORE_TYPE, MOBILE_APPS.STORE_TYPE,
                        MobileAppStoreType::fromSource,
                        MobileAppStoreType::toSource))
                .map(property(MobileApp.STORE_HREF, MOBILE_APPS.STORE_HREF))
                .map(property(MobileApp.NAME, MOBILE_APPS.NAME))
                .map(convertibleProperty(MobileApp.HAS_VERIFICATION, MOBILE_APPS.HAS_IDENTIFIED_POSTBACKS,
                        e -> e != null && e > 0,
                        e -> e != null && e ? 1L : 0L
                ))
                .map(convertibleProperty(MobileApp.MOBILE_CONTENT_ID, MOBILE_APPS.MOBILE_CONTENT_ID,
                        ULong::longValue,
                        ULong::valueOf))
                .map(property(MobileApp.DOMAIN_ID, MOBILE_APPS.DOMAIN_ID))
                .map(convertibleEnumSet(MobileApp.DEVICE_TYPE_TARGETING, MOBILE_APPS.DEVICE_TYPE_TARGETING,
                        MobileAppDeviceTypeTargeting::fromTypedValue,
                        MobileAppDeviceTypeTargeting::getTypedValue))
                .map(convertibleEnumSet(MobileApp.NETWORK_TARGETING, MOBILE_APPS.NETWORK_TARGETING,
                        MobileAppNetworkTargeting::fromTypedValue, MobileAppNetworkTargeting::getTypedValue))
                .map(property(MobileApp.MINIMAL_OPERATING_SYSTEM_VERSION, MOBILE_APPS.MIN_OS_VERSION))
                .map(convertibleEnumSet(MobileApp.DISPLAYED_ATTRIBUTES, MOBILE_APPS.DISPLAYED_ATTRIBUTES,
                        MobileAppDisplayedAttribute::fromTypedValue, MobileAppDisplayedAttribute::getTypedValue))
                .map(convertibleProperty(MobileApp.PRIMARY_ACTION, MOBILE_APPS.PRIMARY_ACTION,
                        MobileAppPrimaryAction::fromSource,
                        MobileAppPrimaryAction::toSource))
                .map(property(MobileApp.APP_METRIKA_APPLICATION_ID, MOBILE_APPS.APP_METRIKA_APPLICATION_ID))
                .build();
    }

    private static JooqMapperWithSupplier<MobileAppTracker> createMobileAppTrackerMapper() {
        return JooqMapperWithSupplierBuilder.builder(MobileAppTracker::new)
                .map(property(MobileAppTracker.ID, MOBILE_APP_TRACKERS.MOBILE_APP_TRACKER_ID))
                .map(clientIdProperty(MobileAppTracker.CLIENT_ID, MOBILE_APP_TRACKERS.CLIENT_ID))
                .map(property(MobileAppTracker.MOBILE_APP_ID, MOBILE_APP_TRACKERS.MOBILE_APP_ID))
                .map(convertibleProperty(MobileAppTracker.TRACKING_SYSTEM, MOBILE_APP_TRACKERS.TRACKING_SYSTEM,
                        MobileAppTrackerTrackingSystem::fromSource,
                        MobileAppTrackerTrackingSystem::toSource))
                .map(property(MobileAppTracker.TRACKER_ID, MOBILE_APP_TRACKERS.TRACKER_ID))
                .map(property(MobileAppTracker.URL, MOBILE_APP_TRACKERS.URL))
                .map(property(MobileAppTracker.IMPRESSION_URL, MOBILE_APP_TRACKERS.IMPRESSION_URL))
                .map(convertibleProperty(MobileAppTracker.USER_PARAMS, MOBILE_APP_TRACKERS.USER_PARAMS,
                        MobileAppMappings::userParamsFromJson,
                        MobileAppMappings::userParamsToJson))
                .build();
    }

    private static JooqReaderWithSupplier<MobileAppForBsExport> createMobileAppForBsExportReader() {
        return JooqReaderWithSupplierBuilder.builder(MobileAppForBsExport::new)
                .readProperty(MobileAppForBsExport.MOBILE_APP_ID, fromField(MOBILE_APPS.MOBILE_APP_ID))
                .readProperty(MobileAppForBsExport.MOBILE_CONTENT_ID,
                        fromField(MOBILE_APPS.MOBILE_CONTENT_ID).by(ULong::longValue))
                .readProperty(MobileAppForBsExport.DOMAIN_ID, fromField(MOBILE_APPS.DOMAIN_ID))
                .readProperty(MobileAppForBsExport.STORE_HREF, fromField(MOBILE_APPS.STORE_HREF))
                .build();
    }
}
