package ru.yandex.direct.core.entity.mobileapp.service

import org.jooq.DSLContext
import org.jooq.exception.DataAccessException
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.mobileapp.model.SkAdNetworkSlot
import ru.yandex.direct.core.entity.mobileapp.repository.IosSkAdNetworkSlotRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import java.time.Duration
import kotlin.random.Random

@Service
class IosSkAdNetworkSlotManager(
    private val dslContextProvider: DslContextProvider,
    private val mobileAppService: MobileAppService,
    private val skAdNetworkSlotsConfigProvider: SkAdNetworkSlotsConfigProvider,
    private val iosSkAdNetworkSlotRepository: IosSkAdNetworkSlotRepository,
    private val grutSkadNetworkSlotService: GrutSkadNetworkSlotService,

    ) {
    companion object {
        private val logger = LoggerFactory.getLogger(IosSkAdNetworkSlotManager::class.java)
    }

    /**
     * Проверка наличия свободного слота для приложения
     */
    fun applicationHasVacantSlot(clientId: ClientId, mobileAppId: Long): Boolean {
        val mobileApp = mobileAppService.getMobileApp(clientId, mobileAppId).orElseGet { null }
            ?: return false
        val appBundleId = mobileApp.mobileContent?.bundleId
            ?: return false
        return applicationHasVacantSlot(appBundleId)
    }

    fun applicationHasVacantSlot(appBundleId: String): Boolean {
        val config = skAdNetworkSlotsConfigProvider.getConfig()
        val dslContext = dslContextProvider.ppcdict()
        val allocatedSlots = iosSkAdNetworkSlotRepository.getAllocatedSlotsByAppBundleId(dslContext, appBundleId)
        return findFreeSlot(allocatedSlots, config.slotsNumber) != null
    }

    /**
     * Получить слоты выделенные для кампаний
     */
    fun getAllocatedSlotsByCampaignIds(campaignIds: Collection<Long>): List<SkAdNetworkSlot> {
        val dslContext = dslContextProvider.ppcdict()
        return iosSkAdNetworkSlotRepository.getSlotsByCampaignIds(dslContext, campaignIds)
    }

    /**
     * Получить слоты выделенные в рамках указанных приложений
     */
    fun getAllocatedSlotsByBundleIds(appBundleIds: Collection<String>): List<SkAdNetworkSlot> {
        return iosSkAdNetworkSlotRepository.getSlotsByAppBundleIds(appBundleIds)
    }

    /**
     * Получить ID кампаний, которые занимают слоты
     */
    fun getCampaignsWithAllocatedSlots(shard: Int, offset: Long, limit: Long): List<Long> {
        val dslContext = dslContextProvider.ppcdict()
        return iosSkAdNetworkSlotRepository.getCampaignIdsWithSlots(dslContext, shard, offset, limit)
    }

    /**
     * Получить ID кампаний, которые в базе отсутствуют
     */
    fun getDeletedCampaignsWithAllocatedSlots(offset: Long, limit: Long): List<Long> {
        val dslContext = dslContextProvider.ppcdict()
        return iosSkAdNetworkSlotRepository.getDeletedCampaignIdsWithSlots(dslContext, offset, limit)
    }

    /**
     * Выделяет кампании слот, для приложения связанного с кампанией.
     *
     * Если кампании уже выделен слот возвращает его
     * Если кампании выделен слот под другое приложение (не может такого быть!) возвращает null
     * Если свободных слотов нет -- возвращаем null
     */
    fun allocateCampaignSlot(appBundleId: String, campaignId: Long, updateInGrut: Boolean = false): Int? {
        val config = skAdNetworkSlotsConfigProvider.getConfig()
        val dslContext = dslContextProvider.ppcdict()
        return allocateCampaignSlot(config, dslContext, appBundleId, campaignId, updateInGrut)
    }

    private fun allocateCampaignSlot(
        config: SkAdNetworkSlotsConfig,
        dslContext: DSLContext,
        appBundleId: String,
        campaignId: Long,
        updateInGrut: Boolean,
    ): Int? {
        for (i in 1..config.triesNumber) {
            var slot: Int? = null
            try {
                dslContext.transaction { txConfig ->
                    slot = allocateCampaignSlotInternal(
                        txConfig.dsl(),
                        appBundleId,
                        campaignId,
                        config.slotsNumber,
                        updateInGrut
                    )
                }
            } catch (e: DataAccessException) {
                logger.info("Couldn't allocate slot for bundleId=${appBundleId}, campaignId=${campaignId}", e)
                continue
            }
            return slot
        }
        return null
    }

    private fun allocateCampaignSlotInternal(
        dslContext: DSLContext, appBundleId: String,
        campaignId: Long, slotsNumber: Int, updateInGrut: Boolean,
    ): Int? {
        val allocatedSlots = iosSkAdNetworkSlotRepository.getAllocatedSlotsByAppBundleId(dslContext, appBundleId)
        val slot = findFreeSlot(allocatedSlots, slotsNumber)
            ?: return null
        val slotDesc = SkAdNetworkSlot(appBundleId, campaignId, slot)
        if (updateInGrut) {
            logger.info("Set slot $slotsNumber for campaign $campaignId in GrUT")
            grutSkadNetworkSlotService.setSlotForCampaign(campaignId, appBundleId, slotsNumber)
        }
        iosSkAdNetworkSlotRepository.addSlot(dslContext, slotDesc)
        return slotDesc.slot
    }

    /**
     * Освободить слоты выделенные для кампаний
     */
    fun freeSlotsByCampaignIds(campaignIds: Collection<Long>, updateInGrut: Boolean) {
        if (updateInGrut) {
            logger.info("Free slot for s $campaignIds in GrUT")
            grutSkadNetworkSlotService.freeSlotForCampaigns(campaignIds)
        }
        iosSkAdNetworkSlotRepository.deleteSlotsByCampaignIds(dslContextProvider.ppcdict(), campaignIds)
    }
}

data class SkAdNetworkSlotsConfig(
    val slotsNumber: Int, // кол-во слотов
    val triesNumber: Int // кол-во попыток занятия слота
)

private fun findFreeSlot(slots: Set<Int>, upperBound: Int): Int? {
    val initial = Random.nextInt(1, upperBound + 1)
    val candidateSequence = IntRange(initial, upperBound).asSequence() + IntRange(1, initial - 1)
    return candidateSequence.find { it !in slots }
}

fun interface SkAdNetworkSlotsConfigProvider {
    fun getConfig(): SkAdNetworkSlotsConfig
}

@Component
class SkAdNetworkSlotsConfigProviderPpcProperties(
    ppcPropertiesSupport: PpcPropertiesSupport
) : SkAdNetworkSlotsConfigProvider {

    private val slotsNumberProperty: PpcProperty<Int> = ppcPropertiesSupport.get(
        PpcPropertyNames.SKADNETWORK_APP_SLOTS_NUMBER, Duration.ofMinutes(1)
    )

    override fun getConfig(): SkAdNetworkSlotsConfig {
        return SkAdNetworkSlotsConfig(slotsNumber = slotsNumberProperty.getOrDefault(10), triesNumber = 3)
    }
}
