package ru.yandex.direct.core.entity.mobileapp.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.TransactionalRunnable;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.mobileapp.MobileAppConverter;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTracker;
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppRepository;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.StatusIconModerate;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository;
import ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService;
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser;
import ru.yandex.direct.core.entity.mobilegoals.MobileAppGoalsService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.time.LocalDateTime.now;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

public class MobileAppAddOperation extends SimpleAbstractAddOperation<MobileApp, Long> {
    private final DslContextProvider dslContextProvider;
    private final MobileAppRepository mobileAppRepository;
    private final MobileContentService mobileContentService;
    private final MobileContentRepository mobileContentRepository;
    private final DomainService domainService;
    private final MobileAppAddValidationService mobileAppAddValidationService;

    @Nullable
    private final User operator;
    private final int shard;
    private final ClientId clientId;
    private List<MobileApp> mobileAppsToAddMobileContent;
    private Map<MobileApp, Long> existingMobileAppIds;
    private final MobileAppConverter mobileAppConverter;
    private final MobileAppGoalsService mobileAppGoalsService;

    private final boolean mergeExistingMobileApps;

    MobileAppAddOperation(Applicability applicability, DslContextProvider dslContextProvider,
                          MobileAppRepository mobileAppRepository,
                          MobileContentService mobileContentService,
                          MobileContentRepository mobileContentRepository,
                          DomainService domainService,
                          MobileAppAddValidationService mobileAppAddValidationService,
                          MobileAppConverter mobileAppConverter,
                          @Nullable User operator, int shard, ClientId clientId,
                          List<MobileApp> models,
                          MobileAppGoalsService mobileAppGoalsService,
                          boolean mergeExistingMobileApps) {
        super(applicability, models);
        this.dslContextProvider = dslContextProvider;
        this.mobileAppRepository = mobileAppRepository;
        this.mobileContentService = mobileContentService;
        this.mobileContentRepository = mobileContentRepository;
        this.domainService = domainService;
        this.mobileAppAddValidationService = mobileAppAddValidationService;
        this.mobileAppConverter = mobileAppConverter;
        this.operator = operator;
        this.shard = shard;
        this.clientId = clientId;
        this.mobileAppGoalsService = mobileAppGoalsService;
        this.mergeExistingMobileApps = mergeExistingMobileApps;
        mobileAppsToAddMobileContent = emptyList();
        existingMobileAppIds = emptyMap();
    }

    @Override
    protected void validate(ValidationResult<List<MobileApp>, Defect> preValidationResult) {
        new ListValidationBuilder<>(preValidationResult)
                .checkEach(notNull())
                .checkEachBy(app -> mobileAppAddValidationService.validateMobileApp(operator, app, clientId));
    }

    @Override
    protected void onModelsValidated(ModelsValidatedStep<MobileApp> modelsValidatedStep) {
        Collection<MobileApp> mobileApps = modelsValidatedStep.getValidModelsMap().values();

        Set<String> allStoreUrls = listToSet(mobileApps, MobileApp::getStoreHref);

        Map<String, MobileContent> existingMobileContents =
                mobileContentRepository.getByMobileAppStoreUrl(shard, clientId, allStoreUrls);

        if (mergeExistingMobileApps) {
            // Если нужно найти уже существующие мобильные приложения, то берем самый большой идентификатор
            // существующего мобильного приложения, ссылающегося на тот же самый мобильный контент.
            Map<Long, Long> existingMobileAppIdByMobileContentId =
                    mobileAppRepository.getLatestMobileAppIdsByMobileContentIds(
                            shard, clientId, mapList(existingMobileContents.values(), MobileContent::getId));

            existingMobileAppIds = StreamEx.of(mobileApps)
                    .mapToEntry(MobileApp::getStoreHref)
                    .mapValues(existingMobileContents::get)
                    .nonNullValues()
                    .mapValues(mc -> existingMobileAppIdByMobileContentId.get(mc.getId()))
                    .nonNullValues()
                    .toCustomMap(IdentityHashMap::new);
        }

        // заполнить для существующих mobileContentId и storeType
        StreamEx.of(mobileApps)
                .mapToEntry(MobileApp::getStoreHref)
                .mapValues(existingMobileContents::get)
                .nonNullValues()
                .forKeyValue((mobileApp, existentMobileContent) -> {
                    mobileApp.setMobileContentId(existentMobileContent.getId());
                    mobileApp.setMobileContent(existentMobileContent);
                    mobileApp.setStoreType(mobileAppConverter.osTypeToStoreType(existentMobileContent.getOsType()));
                    if (mobileApp.getDomain() == null) {
                        mobileApp.setDomainId(existentMobileContent.getPublisherDomainId());
                    }
                });

        List<String> urlsForNonexistentMobileContent =
                ImmutableList.copyOf(Sets.difference(allStoreUrls, existingMobileContents.keySet()));
        Map<String, MobileAppStoreUrl> urlToParsedUrl = StreamEx.of(urlsForNonexistentMobileContent)
                .mapToEntry(MobileAppStoreUrlParser::parseStrict)
                .toMap();
        Map<MobileAppStoreUrl, Optional<MobileContent>> mobileContentsFromYt =
                mobileContentService.getMobileContentFromYt(shard, urlToParsedUrl.values());
        Map<String, MobileContent> urlToMobileContentsFromYt = EntryStream.of(urlToParsedUrl)
                .mapValues(url -> mobileContentsFromYt.get(url).orElse(null))
                .filterValues(Objects::nonNull)
                .toMap();

        mobileAppsToAddMobileContent = new ArrayList<>();
        for (MobileApp mobileApp: mobileApps) {
            if (mobileApp.getMobileContentId() != null || existingMobileAppIds.containsKey(mobileApp)) {
                continue;
            }
            MobileContent mobileContentFromYt = urlToMobileContentsFromYt.get(mobileApp.getStoreHref());
            MobileContent mobileContent;
            // Мобильный контент мог быть получен из YT, а мог быть уже заполнен у приложения, например,
            // при копировании. Если нет ни того ни другого, создаем из ссылки.
            if (mobileContentFromYt != null) {
                mobileContent = mobileContentFromYt;
            } else {
                mobileContent = mobileApp.getMobileContent();
                if (mobileContent == null) {
                    mobileContent = urlToParsedUrl.get(mobileApp.getStoreHref()).toMobileContent();
                }
            }
            mobileContent.setClientId(clientId.asLong());
            mobileContent.setId(null);
            if (mobileApp.getDomain() == null) {
                mobileApp.setDomainId(mobileContent.getPublisherDomainId());
            }
            mobileApp.setStoreType(mobileAppConverter.osTypeToStoreType(mobileContent.getOsType()));
            mobileApp.setMobileContent(mobileContent);
            mobileAppsToAddMobileContent.add(mobileApp);
        }
    }

    @Override
    protected List<Long> execute(List<MobileApp> validModelsToApply) {
        if (validModelsToApply.isEmpty()) {
            return emptyList();
        }
        List<MobileApp> nonExistentApps =
                filterList(validModelsToApply, Predicate.not(existingMobileAppIds::containsKey));
        if (!nonExistentApps.isEmpty()) {
            TransactionalRunnable addTransaction = conf -> {
                DSLContext dslContext = conf.dsl();
                addNonexistentMobileContents(dslContext, clientId, mobileAppsToAddMobileContent);
                addDomains(dslContext, nonExistentApps);
                mobileAppRepository.addMobileApps(dslContext, clientId, nonExistentApps);
                addToMobileAppTrackers(dslContext, clientId, nonExistentApps);
                mobileAppGoalsService.updateMobileAppGoalsForExternalTracker(dslContext, clientId, nonExistentApps);
            };
            dslContextProvider.ppcTransaction(shard, addTransaction);
            // на ppcdict, поэтому вне транзакции
            mobileAppGoalsService.updateMobileAppGoalsForAppmetrika(clientId, nonExistentApps);
        }
        existingMobileAppIds.forEach(MobileApp::setId);
        return mapList(validModelsToApply, MobileApp::getId);
    }

    private void addNonexistentMobileContents(DSLContext dslContext,
                                              ClientId clientId, List<MobileApp> mobileAppsToAddMobileContent) {
        List<MobileContent> toAdd = mapList(mobileAppsToAddMobileContent, MobileApp::getMobileContent);
        prepareDefaultValues(toAdd);
        List<Long> mobileContentIds = mobileContentRepository.getOrCreateMobileContentList(
                dslContext, clientId, toAdd);
        EntryStream.zip(this.mobileAppsToAddMobileContent, mobileContentIds)
                .forKeyValue(MobileApp::setMobileContentId);
    }

    private void addToMobileAppTrackers(DSLContext dslContext, ClientId clientId, List<MobileApp> mobileApps) {
        List<MobileAppTracker> trackers = StreamEx.of(mobileApps)
                .mapToEntry(MobileApp::getId, MobileApp::getTrackers)
                .flatMapValues(Collection::stream)
                .peekKeyValue((mobileAppId, tracker) -> tracker.setMobileAppId(mobileAppId))
                .values()
                .peek(tracker -> tracker.setClientId(clientId))
                .collect(Collectors.toList());
        if (!trackers.isEmpty()) {
            mobileAppRepository.addMobileAppTrackers(dslContext, trackers);
        }
    }

    private void addDomains(DSLContext dslContext, List<MobileApp> validModelsToApply) {
        List<String> domains = StreamEx.of(validModelsToApply)
                .map(MobileApp::getDomain)
                .nonNull()
                .distinct()
                .toList();
        List<Long> domainIds = domainService.getOrCreate(dslContext, domains);
        Map<String, Long> domainToId = EntryStream.zip(domains, domainIds).toMap();
        validModelsToApply.forEach(mobileApp -> {
            String domain = mobileApp.getDomain();
            if (domain != null) {
                mobileApp.setDomainId(
                        checkNotNull(domainToId.get(domain), String.format("There is no domainId for '%s'", domain)));
            }
        });
    }

    private void prepareDefaultValues(List<MobileContent> mobileContents) {
        mobileContents.forEach(mobileContent -> mobileContent
                .withCreateTime(nvl(mobileContent.getCreateTime(), now()))
                .withStatusBsSynced(nvl(mobileContent.getStatusBsSynced(), StatusBsSynced.NO))
                .withStatusIconModerate(nvl(mobileContent.getStatusIconModerate(), StatusIconModerate.READY))
                .withTriesCount(nvl(mobileContent.getTriesCount(), 0))
                .withIsAvailable(nvl(mobileContent.getIsAvailable(), false)));
    }

    /**
     * Запрещаем частичное исполнение операции.
     */
    @Override
    public MassResult<Long> apply(Set<Integer> elementIndexesToApply) {
        throw new UnsupportedOperationException("Partial apply is unsupported by the operation");
    }
}
