package ru.yandex.direct.core.entity.mobileapp.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.mobileapp.MobileAppConverter;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppAutomaticCreationRequestWithSuppliedMobileContent;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppAutomaticCreationRequestWithoutSuppliedMobileContent;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppDeviceTypeTargeting;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppDisplayedAttribute;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppNetworkTargeting;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppPrimaryAction;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTracker;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerTrackingSystem;
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppRepository;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис создаёт по переданной модели MobileContent и некоторым другим данным
 * записи в mobile_apps и mobile_app_trackers. Здесь есть бизнес-логика:
 * некоторые данные в mobile_apps получаются из данных в mobile_content преобразованием
 * по каким-то правилам; например, флаг "показывать рейтинг приложения" ставится,
 * только есть в MobileContent рейтинг есть и он больше или равен 3.
 */
@ParametersAreNonnullByDefault
@Service
public class MobileAppAutomaticCreationService {
    private static final BigDecimal DISPLAY_RATING_THRESHOLD = BigDecimal.valueOf(3L);
    private static final Logger LOGGER = LoggerFactory.getLogger(MobileAppAutomaticCreationService.class);


    private final MobileAppRepository mobileAppRepository;
    private final MobileAppConverter mobileAppConverter;
    private final MobileContentRepository mobileContentRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public MobileAppAutomaticCreationService(
            MobileAppRepository mobileAppRepository,
            MobileAppConverter mobileAppConverter,
            MobileContentRepository mobileContentRepository,
            ShardHelper shardHelper) {
        this.mobileAppRepository = mobileAppRepository;
        this.mobileAppConverter = mobileAppConverter;
        this.mobileContentRepository = mobileContentRepository;
        this.shardHelper = shardHelper;
    }

    /**
     * Создать MobileApp из объектов MobileContent, которые есть в базе
     *
     * @return соответствие переданных для создания данных и ID новых (или найденных) MobileApp
     */
    public Map<MobileAppAutomaticCreationRequestWithSuppliedMobileContent, Long> getOrCreateMobileAppsFromMobileContentModels(
            ClientId clientId, Collection<MobileAppAutomaticCreationRequestWithSuppliedMobileContent> requests) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Map<Long, Long> existingMobileAppIdsByMobileContentIds =
                mobileAppRepository.getLatestMobileAppIdsByMobileContentIds(shard, clientId,
                        mapList(requests, request -> request.getMobileContent().getId()));

        log(clientId, "existingMobileAppIdsByMobileContentIds", existingMobileAppIdsByMobileContentIds);

        Map<MobileAppAutomaticCreationRequestWithSuppliedMobileContent, Long> result = new HashMap<>();
        for (MobileAppAutomaticCreationRequestWithSuppliedMobileContent request : requests) {
            MobileContent mobileContent = request.getMobileContent();
            Long mobileContentId = mobileContent.getId();

            if (existingMobileAppIdsByMobileContentIds.containsKey(mobileContentId)) {
                log(clientId, "mobileApp exists for mobileContentId", mobileContentId);
                result.put(request, existingMobileAppIdsByMobileContentIds.get(mobileContentId));
                continue;
            }

            MobileApp mobileApp = createMobileAppAutomatically(
                    mobileContent,
                    request.getStoreHref(),
                    request.getMinimalOperatingSystemVersion(),
                    request.getTrackerUrl(),
                    request.getImpressionUrl());

            log(clientId, "created mobileApp", mobileApp);
            result.put(request, mobileApp.getId());
        }

        log(clientId, "result", result);
        return result;
    }

    /**
     * Создать MobileApp из URL приложений в сторе
     *
     * @return соответствие переданных для создания данных и ID новых (или найденных) MobileApp
     */
    public Map<MobileAppAutomaticCreationRequestWithoutSuppliedMobileContent, Long> getOrCreateMobileAppsFromStoreHrefs(
            ClientId clientId, Collection<MobileAppAutomaticCreationRequestWithoutSuppliedMobileContent> requests) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        List<Long> gotOrCreatedMobileContentIds = mobileContentRepository.getOrCreate(
                shard, clientId,
                mapList(requests, MobileAppAutomaticCreationRequestWithoutSuppliedMobileContent::getParsedStoreHref));
        log(clientId, "gotOrCreatedMobileContentIds", gotOrCreatedMobileContentIds);

        List<MobileContent> gotOrCreatedMobileContentModels = mobileContentRepository
                .getMobileContent(shard, gotOrCreatedMobileContentIds);
        log(clientId, "gotOrCreatedMobileContentModels", gotOrCreatedMobileContentModels);

        Map<MobileAppStoreUrl, MobileContent> mobileContentByStoreHref = gotOrCreatedMobileContentModels.stream()
                .collect(toMap(MobileAppStoreUrl::fromMobileContent, identity()));
        log(clientId, "mobileContentByStoreHref", mobileContentByStoreHref);

        Map<MobileAppAutomaticCreationRequestWithoutSuppliedMobileContent, MobileAppAutomaticCreationRequestWithSuppliedMobileContent>
                requestsToRequestsWithSuppliedMobileContents = requests.stream()
                .collect(toMap(identity(),
                        request -> new MobileAppAutomaticCreationRequestWithSuppliedMobileContent(
                                mobileContentByStoreHref.get(request.getParsedStoreHref()),
                                request.getStoreHref(),
                                request.getMinimalOperatingSystemVersion(),
                                request.getTrackerUrl(),
                                request.getImpressionUrl())));
        log(clientId, "requestsToRequestsWithSuppliedMobileContents", requestsToRequestsWithSuppliedMobileContents);

        Map<MobileAppAutomaticCreationRequestWithSuppliedMobileContent, Long> createdApps =
                getOrCreateMobileAppsFromMobileContentModels(clientId,
                        requestsToRequestsWithSuppliedMobileContents.values());
        log(clientId, "createdApps", createdApps);

        return requests.stream()
                .collect(toMap(identity(),
                        request -> createdApps.get(requestsToRequestsWithSuppliedMobileContents.get(request))));
    }

    /**
     * Создаёт приложение по переданным данным.
     *
     * @param mobileContent                 какие данные удалось получить из поискового индекса; здесь может быть
     *                                      "пустышка": clientId, osType, contentType, storeCountry, storeContentId,
     *                                      а всё остальное не заполнено, например, когда у потребителя не было возможности
     *                                      сходить
     * @param storeHref                     URL страницы магазина с этим приложением
     * @param minimalOperatingSystemVersion минимальная версия ОС, на которую ставится приложение, может отсутствовать
     * @param trackerUrl                    URL трекинговой ссылки, через которую пользователи попадают в магазин, может отсутствовать
     * @param impressionUrl                 URL трекинговой ссылки, в которую отправляется факт показа, может отсутствовать
     * @return объект модели, соответствующий новой записи в БД в таблице mobile_apps
     */
    MobileApp createMobileAppAutomatically(MobileContent mobileContent,
                                           String storeHref,
                                           @Nullable String minimalOperatingSystemVersion,
                                           @Nullable String trackerUrl,
                                           @Nullable String impressionUrl) {
        Long clientId = mobileContent.getClientId();
        ClientId clientIdObject = ClientId.fromLong(clientId);

        Set<MobileAppDisplayedAttribute> displayedAttributes = constructMobileAppDisplayedAttributes(mobileContent);

        MobileApp mobileApp = new MobileApp()
                .withClientId(clientIdObject)
                .withDeviceTypeTargeting(EnumSet.allOf(MobileAppDeviceTypeTargeting.class))
                .withDisplayedAttributes(displayedAttributes)
                .withDomainId(mobileContent.getPublisherDomainId())
                .withMinimalOperatingSystemVersion(minimalOperatingSystemVersion)
                .withMobileContent(mobileContent)
                .withMobileContentId(mobileContent.getId())
                .withName(nvl(mobileContent.getName(), ""))
                .withNetworkTargeting(EnumSet.allOf(MobileAppNetworkTargeting.class))
                .withPrimaryAction(MobileAppPrimaryAction.INSTALL)
                .withStoreHref(storeHref)
                .withStoreType(mobileAppConverter.osTypeToStoreType(mobileContent.getOsType()));

        int shard = shardHelper.getShardByClientIdStrictly(clientIdObject);

        mobileAppRepository.addMobileApp(shard, mobileApp);

        List<MobileAppTracker> trackers = constructTrackers(trackerUrl, impressionUrl, clientIdObject, mobileApp.getId());
        mobileApp.setTrackers(trackers);
        if (!trackers.isEmpty()) {
            mobileAppRepository.addMobileAppTrackers(shard, trackers);
        }

        return mobileApp;
    }

    private static List<MobileAppTracker> constructTrackers(@Nullable String trackerUrl,
                                                            @Nullable String impressionUrl,
                                                            ClientId clientId, Long mobileAppId) {
        List<MobileAppTracker> trackers;

        if (trackerUrl != null || impressionUrl != null) {
            trackers = singletonList(new MobileAppTracker()
                    .withClientId(clientId)
                    .withMobileAppId(mobileAppId)
                    .withTrackingSystem(MobileAppTrackerTrackingSystem.OTHER)
                    .withUrl(trackerUrl)
                    .withImpressionUrl(impressionUrl));
        } else {
            trackers = emptyList();
        }
        return trackers;
    }

    private static Set<MobileAppDisplayedAttribute> constructMobileAppDisplayedAttributes(MobileContent mobileContent) {
        Set<MobileAppDisplayedAttribute> displayedAttributes = EnumSet.noneOf(MobileAppDisplayedAttribute.class);

        if (mobileContent.getRating() != null && mobileContent.getRating().compareTo(DISPLAY_RATING_THRESHOLD) >= 0) {
            displayedAttributes.add(MobileAppDisplayedAttribute.RATING);
        }

        if (mobileContent.getRatingVotes() != null) {
            displayedAttributes.add(MobileAppDisplayedAttribute.RATING_VOTES);
        }

        if (mobileContent.getIconHash() != null && !mobileContent.getIconHash().isEmpty()) {
            displayedAttributes.add(MobileAppDisplayedAttribute.ICON);
        }

        return displayedAttributes;
    }

    private static void log(ClientId clientId, String dataName, Object data) {
        String dataDump;
        try {
            dataDump = JsonUtils.toJsonNullable(data);
        } catch (IllegalArgumentException e) {
            log(clientId, "error trying to serialize " + dataName + ": " + ExceptionUtils.getStackTrace(e));
            dataDump = data.toString();
        }

        log(clientId, dataName + " = " + dataDump);
    }

    @SuppressWarnings("squid:S2629")
    private static void log(ClientId clientId, String message) {
        LOGGER.info("[clientId=" + clientId.asLong() + "] " + message);
    }

}
