package ru.yandex.direct.core.entity.mobileapp.service;

import java.util.Collection;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.mobileapp.MobileAppConverter;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppRepository;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository;
import ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService;
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser;
import ru.yandex.direct.core.entity.mobilegoals.MobileAppGoalsService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;

import static java.util.Collections.singletonList;

@ParametersAreNonnullByDefault
@Service
public class MobileAppService implements EntityService<MobileApp, Long> {
    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final MobileAppRepository mobileAppRepository;
    private final MobileContentRepository mobileContentRepository;
    private final MobileContentService mobileContentService;
    private final DomainService domainService;
    private final UserService userService;
    private final MobileAppAddValidationService mobileAppAddValidationService;
    private final MobileAppConverter mobileAppConverter;
    private final MobileAppGoalsService mobileAppGoalsService;

    public MobileAppService(DslContextProvider dslContextProvider,
                            ShardHelper shardHelper,
                            MobileAppRepository mobileAppRepository,
                            MobileContentRepository mobileContentRepository,
                            MobileContentService mobileContentService,
                            DomainService domainService,
                            UserService userService,
                            MobileAppAddValidationService mobileAppAddValidationService,
                            MobileAppConverter mobileAppConverter, MobileAppGoalsService mobileAppGoalsService) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.mobileAppRepository = mobileAppRepository;
        this.mobileContentRepository = mobileContentRepository;
        this.mobileContentService = mobileContentService;
        this.domainService = domainService;
        this.userService = userService;
        this.mobileAppAddValidationService = mobileAppAddValidationService;
        this.mobileAppConverter = mobileAppConverter;
        this.mobileAppGoalsService = mobileAppGoalsService;
    }

    /**
     * Записывает объекты MobileApp в базу вместе со связанными (trackers; storeHref превращается в MobileContent
     * в базе)
     * <p>
     * Меняет переданный объект mobileApp: заполняет mobileContentId (но не mobileContent)
     */
    public Long addMobileApp(ClientId clientId, MobileApp mobileApp) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        // TODO здесь несколько запросов, которые по идее стоит делать в транзакции

        MobileAppStoreUrl parsedUrl =
                MobileAppStoreUrlParser.parse(mobileApp.getStoreHref()).orElseThrow(IllegalArgumentException::new);
        Long mobileContentId = mobileContentRepository.getOrCreate(shard, clientId, singletonList(parsedUrl)).get(0);

        mobileApp.setMobileContentId(mobileContentId);

        mobileAppRepository.addMobileAppTrackers(shard, mobileApp.getTrackers());
        return mobileAppRepository.addMobileApp(shard, mobileApp);
    }

    public List<MobileApp> getMobileApps(ClientId clientId) {
        return mobileAppRepository.getMobileApps(
                shardHelper.getShardByClientIdStrictly(clientId), clientId, null);
    }

    public List<MobileApp> getMobileApps(ClientId clientId, Collection<Long> mobileAppIds) {
        return mobileAppRepository.getMobileApps(
                shardHelper.getShardByClientIdStrictly(clientId), clientId, mobileAppIds);
    }

    public Optional<MobileApp> getMobileApp(ClientId clientId, Long mobileAppId) {
        List<MobileApp> mobileApps = mobileAppRepository.getMobileApps(
                shardHelper.getShardByClientIdStrictly(clientId), clientId,
                singletonList(mobileAppId));
        return mobileApps.isEmpty() ? Optional.empty() : Optional.of(mobileApps.get(0));
    }

    public MobileAppAddOperation createAddPartialOperation(@Nullable User operator, ClientId clientId,
                                                           List<MobileApp> mobileApps) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return createAddOperation(operator, shard, clientId, mobileApps, Applicability.PARTIAL, false);
    }

    public MobileAppAddOperation createAddFullOperation(@Nullable User operator, ClientId clientId,
                                                           List<MobileApp> mobileApps) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return createAddOperation(operator, shard, clientId, mobileApps, Applicability.FULL, false);
    }

    private MobileAppAddOperation createAddOperation(
            @Nullable User operator,
            int shard,
            ClientId clientId,
            List<MobileApp> mobileApps,
            Applicability applicability,
            boolean mergeExistingMobileApps) {
        return new MobileAppAddOperation(applicability, dslContextProvider, mobileAppRepository,
                mobileContentService, mobileContentRepository, domainService, mobileAppAddValidationService,
                mobileAppConverter,
                operator, shard, clientId, mobileApps,
                mobileAppGoalsService, mergeExistingMobileApps);
    }

    @Override
    public List<MobileApp> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return mobileAppRepository.getMobileApps(shard, clientId, ids);
    }

    @Override
    public MassResult add(ClientId clientId, Long operatorUid, List<MobileApp> entities, Applicability applicability) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        MobileAppAddOperation addMobileAppOperation =
                createAddOperation(userService.getUser(operatorUid), shard, clientId, entities, applicability, false);
        return addMobileAppOperation.prepareAndApply();
    }

    @Override
    public MassResult copy(CopyOperationContainer copyContainer, List<MobileApp> entities,
                           Applicability applicability) {
        MobileAppAddOperation addMobileAppOperation =
                createAddOperation(
                        userService.getUser(copyContainer.getOperatorUid()),
                        copyContainer.getShardTo(),
                        copyContainer.getClientIdTo(),
                        entities,
                        applicability,
                        true);
        return addMobileAppOperation.prepareAndApply();
    }
}
