package ru.yandex.direct.core.entity.mobileapp.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;

import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTracker;
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppRepository;
import ru.yandex.direct.core.entity.mobilegoals.MobileAppGoalsService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция обновления данных о мобильном приложении
 */
public class MobileAppUpdateOperation extends SimpleAbstractUpdateOperation<MobileApp, Long> {
    // Нижеперечисленные проперти не могут быть измененены этой операцией
    private static final Set<ModelProperty> UNSUPPORTED_CHANGES = ImmutableSet.of(
            MobileApp.ID, MobileApp.CLIENT_ID, MobileApp.STORE_TYPE, MobileApp.STORE_HREF,
            MobileApp.MOBILE_CONTENT, MobileApp.MOBILE_CONTENT_ID, MobileApp.DOMAIN_ID
    );

    private final DslContextProvider dslContextProvider;
    private final int shard;
    private final ClientId clientId;
    private final MobileAppRepository mobileAppRepository;
    private final DomainService domainService;
    private final ModelChangesValidationTool preValidationTool;
    private final MobileAppUpdateValidationService mobileAppUpdateValidationService;
    private final MobileAppGoalsService mobileAppGoalsService;
    @Nullable
    private final User operator;

    MobileAppUpdateOperation(Applicability applicability,
                             DslContextProvider dslContextProvider, DomainService domainService,
                             MobileAppRepository mobileAppRepository,
                             MobileAppUpdateValidationService mobileAppUpdateValidationService,
                             @Nullable User operator,
                             int shard, ClientId clientId, List<ModelChanges<MobileApp>> modelChanges,
                             MobileAppGoalsService mobileAppGoalsService) {
        super(applicability, modelChanges, id -> new MobileApp().withId(id));

        // Нужно проверить, что операция инициализирована поддерживаемыми изменениями
        modelChanges.forEach(mc -> checkArgument(
                Sets.intersection(UNSUPPORTED_CHANGES, mc.getChangedPropsNames()).isEmpty(),
                "ModelChanges with unsupported changes is passed to MobileAppUpdateOperation"));

        this.dslContextProvider = dslContextProvider;
        this.operator = operator;
        this.shard = shard;
        this.clientId = clientId;
        this.mobileAppRepository = mobileAppRepository;
        this.domainService = domainService;
        this.mobileAppUpdateValidationService = mobileAppUpdateValidationService;
        this.preValidationTool = new ModelChangesValidationTool();
        this.mobileAppGoalsService = mobileAppGoalsService;
    }

    @Override
    protected ValidationResult<List<ModelChanges<MobileApp>>, Defect> validateModelChanges(
            List<ModelChanges<MobileApp>> modelChanges) {
        List<Long> ids = mapList(modelChanges, ModelChanges::getId);
        Set<Long> existingIds = listToSet(getModels(ids), MobileApp::getId);
        ListValidationBuilder<ModelChanges<MobileApp>, Defect> vb = new ListValidationBuilder<>(
                preValidationTool.validateModelChangesList(modelChanges, existingIds));
        vb.checkEachBy(mc -> mobileAppUpdateValidationService.validateModelChanges(operator, mc));
        return vb.getResult();
    }

    @Override
    protected ValidationResult<List<MobileApp>, Defect> validateAppliedChanges(
            ValidationResult<List<MobileApp>, Defect> validationResult) {
        return new ListValidationBuilder<>(validationResult)
                .checkEachBy(app -> mobileAppUpdateValidationService.validateMobileApp(app, clientId))
                .getResult();
    }

    @Override
    protected Collection<MobileApp> getModels(Collection<Long> ids) {
        return mobileAppRepository.getMobileApps(shard, clientId, ids);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<MobileApp>> applicableAppliedChanges) {
        if (applicableAppliedChanges.isEmpty()) {
            return emptyList();
        }
        // это обновление, поэтому все идентификаторы приложений тут должны присутствовать (проверяется валидацией)
        List<Long> mobileAppIds = applicableAppliedChanges.stream()
                .map(AppliedChanges::getModel)
                .map(ModelWithId::getId)
                .collect(toList());

        List<MobileApp> mobileApps = mapList(applicableAppliedChanges, AppliedChanges::getModel);
        dslContextProvider.ppcTransaction(shard, conf -> {
            DSLContext dslContext = conf.dsl();
            updateDomains(dslContext, applicableAppliedChanges);
            mobileAppRepository.updateMobileApps(dslContext, applicableAppliedChanges);
            rewriteTrackers(dslContext, applicableAppliedChanges);

            mobileAppGoalsService.updateMobileAppGoalsForExternalTracker(dslContext, clientId, mobileApps);
        });

        // на ppcdict, поэтому вне транзакции
        mobileAppGoalsService.updateMobileAppGoalsForAppmetrika(clientId, mobileApps);
        return mobileAppIds;
    }

    private void updateDomains(DSLContext dslContext, List<AppliedChanges<MobileApp>> appliedChanges) {
        Map<Long, Optional<String>> mobileAppIdToNewDomain = StreamEx.of(appliedChanges)
                .filter(ac -> ac.changed(MobileApp.DOMAIN))
                .map(AppliedChanges::getModel)
                .mapToEntry(MobileApp::getId, MobileApp::getDomain)
                .mapValues(Optional::ofNullable)
                .toMap();

        List<String> newUniqueDomains = EntryStream.of(mobileAppIdToNewDomain)
                .values()
                .filter(Optional::isPresent)
                .map(Optional::get)
                .distinct()
                .collect(toList());
        List<Long> domainIds = domainService.getOrCreate(dslContext, newUniqueDomains);
        Map<String, Long> domainToId = EntryStream.zip(newUniqueDomains, domainIds).toMap();

        for (AppliedChanges<MobileApp> ac : appliedChanges) {
            Long mobileAppId = ac.getModel().getId();
            if (!mobileAppIdToNewDomain.containsKey(mobileAppId)) {
                continue;
            }
            Optional<String> newDomain = mobileAppIdToNewDomain.get(mobileAppId);
            if (newDomain.isPresent()) {
                Long domainId = checkNotNull(domainToId.get(newDomain.get()),
                        String.format("There is no domainId for '%s'", newDomain.get()));
                ac.modify(MobileApp.DOMAIN_ID, domainId);
            } else {
                ac.modify(MobileApp.DOMAIN_ID, null);
            }
        }
    }

    private void rewriteTrackers(DSLContext dslContext, List<AppliedChanges<MobileApp>> appliedChanges) {
        List<MobileAppTracker> trackers = StreamEx.of(appliedChanges)
                .map(AppliedChanges::getModel)
                .mapToEntry(MobileApp::getId, MobileApp::getTrackers)
                .flatMapValues(Collection::stream)
                .peekKeyValue((appId, tracker) -> tracker.setMobileAppId(appId))
                .values()
                .peek(tracker -> tracker.setClientId(clientId))
                .toList();
        Set<Long> mobileAppIds = StreamEx.of(appliedChanges)
                .map(AppliedChanges::getModel)
                .map(MobileApp::getId)
                .toSet();
        mobileAppRepository.deleteAllTrackersOfMobileApp(dslContext, mobileAppIds);
        if (!trackers.isEmpty()) {
            mobileAppRepository.addMobileAppTrackers(dslContext, trackers);
        }
    }
}
