package ru.yandex.direct.core.entity.mobileapp.service;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects;
import ru.yandex.direct.core.entity.banner.type.href.BannerWithHrefConstraints;
import ru.yandex.direct.core.entity.mobileapp.Constants;
import ru.yandex.direct.core.entity.mobileapp.model.AppmetrikaUniqueKey;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppStoreType;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTracker;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerTrackingSystem;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerUrlTemplate;
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppRepository;
import ru.yandex.direct.core.entity.mobileapp.util.MobileAppUtil;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService;
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser;
import ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsService;
import ru.yandex.direct.core.entity.uac.service.trackingurl.TrackingUrlParseService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.mobileapp.MobileAppDefects.appMetrikaApplicationAlreadyUsed;
import static ru.yandex.direct.core.entity.mobileapp.MobileAppDefects.invalidAppStoreUrl;
import static ru.yandex.direct.core.entity.mobileapp.MobileAppDefects.invalidDomain;
import static ru.yandex.direct.core.entity.mobileapp.MobileAppDefects.invalidUrl;
import static ru.yandex.direct.core.entity.mobileapp.MobileAppDefects.trackerUrlDomainDoesNotMatchSelectedTrackerSystem;
import static ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerTrackingSystem.ADJUST;
import static ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerTrackingSystem.APPMETRICA;
import static ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerTrackingSystem.OTHER;
import static ru.yandex.direct.core.entity.mobileapp.util.MobileAppUtil.getAppmetrikaUniqueKey;
import static ru.yandex.direct.core.validation.constraints.Constraints.validDomain;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;

@Component
@ParametersAreNonnullByDefault
public class MobileAppValidationService {
    private static final int MAX_TRACKERS = 10;
    private static final int MAX_DOMAIN_URL_SIZE = 100;
    static final int MAX_STORE_HREF_LENGTH = 1024;
    private static final int MAX_NAME_LENGTH = 1024;
    static final int MAX_TRACKER_URL_LENGTH = 1024;
    static final int MAX_TRACKER_ID_LENGTH = 255;
    static final int MAX_USER_PARAM_LENGTH = 40;
    static final int MAX_USER_PARAM_SIZE = 20;

    private static final String ANDROID_TRACKING_URL_MACROS = "google_aid";
    private static final String IOS_TRACKING_URL_MACROS = "ios_ifa";
    private static final String LOGID_MACROS = "logid";
    private static final String DOMAIN_FOR_ADJUST = "app.adjust.com";
    private static final String DOMAIN_FOR_ADJUST_IMPRESSION = "view.adjust.com";

    private final TrustedRedirectsService trustedRedirectsService;
    private final TrackerDomainConstraintFactory trackerDomainConstraintFactory;
    private final ShardHelper shardHelper;
    private final MobileAppRepository mobileAppRepository;
    private final MobileContentService mobileContentService;
    private final TrackingUrlParseService trackingUrlParseService;

    public MobileAppValidationService(TrustedRedirectsService trustedRedirectsService, ShardHelper shardHelper,
                                      MobileAppRepository mobileAppRepository,
                                      MobileContentService mobileContentService,
                                      TrackingUrlParseService trackingUrlParseService) {
        this.trustedRedirectsService = trustedRedirectsService;
        trackerDomainConstraintFactory = new TrackerDomainConstraintFactory();
        this.shardHelper = shardHelper;
        this.mobileAppRepository = mobileAppRepository;
        this.mobileContentService = mobileContentService;
        this.trackingUrlParseService = trackingUrlParseService;
    }

    ValidationResult<MobileApp, Defect> validateMobileApp(MobileApp mobileApp, ClientId clientId) {
        ModelItemValidationBuilder<MobileApp> vb = ModelItemValidationBuilder.of(mobileApp);

        MobileAppStoreUrl parsedUrl = MobileAppStoreUrlParser.parse(mobileApp.getStoreHref()).orElse(null);

        vb.item(MobileApp.STORE_HREF)
                .check(notNull())
                .check(notBlank())
                .check(maxStringLength(MAX_STORE_HREF_LENGTH))
                .check(validHref(), invalidUrl(), When.isValid())
                .check(validAppStoreUrl(parsedUrl), When.isValid());

        vb.item(MobileApp.NAME)
                .check(notNull())
                .check(notBlank())
                .check(maxStringLength(MAX_NAME_LENGTH));

        vb.item(MobileApp.DISPLAYED_ATTRIBUTES)
                .check(notNull());

        if (parsedUrl != null) {
            // т.к. в валидации трекера используется информация об ОС
            vb.list(MobileApp.TRACKERS)
                    .check(notNull())
                    .check(maxListSize(MAX_TRACKERS))
                    .checkEachBy(v -> validateMobileAppTracker(v, parsedUrl.getOsType()), When.isValid());

            // т.к. parsedUrl используется для получения mobileContent
            Set<AppmetrikaUniqueKey> usedAppmetrikaUniqueKeys = getUsedAppmetrikaUniqueKeys(clientId, mobileApp.getId());
            AppmetrikaUniqueKey mobileAppKey = getAppmetrikaUniqueKey(mobileApp);

            if (mobileApp.getMobileContent() == null) {
                MobileContent mobileContent = mobileContentService.getMobileContent(clientId, mobileApp.getStoreHref(),
                        parsedUrl, true, false).orElse(null);

                if (mobileContent != null) {
                    mobileAppKey.setBundleId(mobileContent.getOsType() == OsType.IOS ? mobileContent.getBundleId() : mobileContent.getStoreContentId());
                    mobileAppKey.setStoreType(mobileContent.getOsType() == OsType.IOS ? MobileAppStoreType.APPLEAPPSTORE : MobileAppStoreType.GOOGLEPLAYSTORE);
                }
            }

            if (mobileAppKey.getBundleId() != null) {
                vb.check(validMobileAppKey(mobileAppKey, usedAppmetrikaUniqueKeys));
            }
        }

        vb.item(MobileApp.DOMAIN)
                .check(maxStringLength(MAX_DOMAIN_URL_SIZE))
                .check(notBlank(), When.isValid())
                .check(validDomain(), invalidDomain(), When.isValid());

        return vb.getResult();
    }

    private Set<AppmetrikaUniqueKey> getUsedAppmetrikaUniqueKeys(ClientId clientId, Long appId) {
        return mobileAppRepository
                .getMobileApps(shardHelper.getShardByClientIdStrictly(clientId), clientId, null)
                .stream().filter(m -> !m.getId().equals(appId))
                .filter(m -> m.getAppMetrikaApplicationId() != null)
                .map(MobileAppUtil::getAppmetrikaUniqueKey)
                .collect(Collectors.toSet());
    }

    private static Constraint<MobileApp, Defect> validMobileAppKey(AppmetrikaUniqueKey mobileAppKey,
                                                                   Set<AppmetrikaUniqueKey> usedMobileAppKeys) {
        return Constraint.fromPredicate(mobileApp -> !usedMobileAppKeys.contains(mobileAppKey),
                appMetrikaApplicationAlreadyUsed());
    }

    private static Constraint<String, Defect> validAppStoreUrl(@Nullable MobileAppStoreUrl parsedUrl) {
        return Constraint.fromPredicate(url -> parsedUrl != null, invalidAppStoreUrl());
    }

    ValidationResult<MobileAppTracker, Defect> validateMobileAppTracker(MobileAppTracker mobileAppTracker,
                                                                        OsType osType) {
        ModelItemValidationBuilder<MobileAppTracker> vb = ModelItemValidationBuilder.of(mobileAppTracker);
        if (mobileAppTracker.getTrackingSystem() == APPMETRICA && "".equals(mobileAppTracker.getUrl())) {
            return vb.getResult();
        }

        String domainFromUrl = extractDomain(mobileAppTracker.getUrl());
        boolean isAdjust = mobileAppTracker.getTrackingSystem() == ADJUST || domainFromUrl.equals(DOMAIN_FOR_ADJUST);
        String impressionDomainFromUrl = extractDomain(mobileAppTracker.getImpressionUrl());
        boolean isAdjustImpression = mobileAppTracker.getTrackingSystem() == ADJUST || impressionDomainFromUrl.equals(DOMAIN_FOR_ADJUST_IMPRESSION);

        boolean other = mobileAppTracker.getTrackingSystem() == OTHER;
        vb.item(MobileAppTracker.URL)
                .check(notNull())
                .check(notBlank())
                .check(maxStringLength(MAX_TRACKER_URL_LENGTH))
                .check(validHref(), invalidUrl(), When.isValid())
                .check(trackingUrlContainMacros(ANDROID_TRACKING_URL_MACROS),
                        When.isValidAnd(When.isTrue(osType == OsType.ANDROID && !other)))
                .check(trackingUrlContainMacros(IOS_TRACKING_URL_MACROS),
                        When.isValidAnd(When.isTrue(osType == OsType.IOS && !other)))
                .check(trackingUrlContainMacros(LOGID_MACROS),
                        When.isValidAnd(When.isTrue(!isAdjust && !other)))
                .check(BannerWithHrefConstraints.validTrackingHref(trustedRedirectsService), When.isValid())
                .check(trackerDomainConstraintFactory.createForClick(mobileAppTracker.getTrackingSystem()), When.isValid());
        vb.item(MobileAppTracker.IMPRESSION_URL)
                .check(notNull())
                .check(maxStringLength(MAX_TRACKER_URL_LENGTH))
                .check(validHref(), invalidUrl(),
                        When.isValidAnd(When.valueIs(StringUtils::isNotBlank)))
                .check(trackingUrlContainMacros(ANDROID_TRACKING_URL_MACROS),
                        When.isValidAnd(When.valueIs(
                                val -> StringUtils.isNotBlank(val) && osType == OsType.ANDROID && !other)))
                .check(trackingUrlContainMacros(IOS_TRACKING_URL_MACROS),
                        When.isValidAnd(When.valueIs(
                                val -> StringUtils.isNotBlank(val) && osType == OsType.IOS && !other)))
                .check(trackingUrlContainMacros(LOGID_MACROS),
                        When.isValidAnd(When.valueIs(
                                val -> StringUtils.isNotBlank(val) && !isAdjustImpression && !other)))
                .check(BannerWithHrefConstraints.validImpressionUrl(trustedRedirectsService, trackingUrlParseService, mobileAppTracker.getUrl()),
                        When.isValidAnd(When.valueIs(StringUtils::isNotBlank)))
                .check(trackerDomainConstraintFactory.createForImpression(mobileAppTracker.getTrackingSystem()),
                        When.isValidAnd(When.valueIs(StringUtils::isNotBlank)));
        vb.item(MobileAppTracker.TRACKING_SYSTEM)
                .check(notNull());
        vb.item(MobileAppTracker.TRACKER_ID)
                .check(notBlank(), When.isFalse(other))
                .check(maxStringLength(MAX_TRACKER_ID_LENGTH), When.isFalse(other));
        vb.list(MobileAppTracker.USER_PARAMS)
                .check(notNull())
                .check(eachNotNull())
                .check(maxListSize(MAX_USER_PARAM_SIZE), When.isValid())
                .checkEach(maxStringLength(MAX_USER_PARAM_LENGTH), When.isValid())
                .checkEach(notBlank(), When.isValid())
                .checkEach(matchPattern(Pattern.compile("[0-9a-zA-Z_%$~-]+")), When.isValid());
        return vb.getResult();
    }

    private static String extractDomain(String url) {
        try {
            return new URL(url).getHost();
        } catch (MalformedURLException ignore) {
            return "";
        }
    }

    private Constraint<String, Defect> trackingUrlContainMacros(String macros) {
        return Constraint.fromPredicate(url -> StringUtils.containsIgnoreCase(url, "{" + macros + "}"),
                BannerDefects.trackingUrlDoesntContainMacros());
    }

    /**
     * Эта штука валидирует соответствие домена в урле, домену выбранной трекинговой системы.
     * Соответствие должно быть строгим, т.к. урл должен создаваться по шаблону для трекинговой системы
     */
    private static class TrackerDomainConstraintFactory {
        private final Map<MobileAppTrackerTrackingSystem, String> trackingSystemToClickDomain;
        private final Map<MobileAppTrackerTrackingSystem, String> trackingSystemToImpressionDomain;

        TrackerDomainConstraintFactory() {
            this.trackingSystemToClickDomain = EntryStream.of(Constants.TRACKING_SYSTEMS_CLICK_DESC)
                    .mapValues(MobileAppTrackerUrlTemplate::getBaseUrlTemplate)
                    .mapValues(MobileAppValidationService::extractDomain)
                    .toMap();
            this.trackingSystemToImpressionDomain = EntryStream.of(Constants.TRACKING_SYSTEMS_IMPRESSION_DESC)
                    .mapValues(MobileAppTrackerUrlTemplate::getBaseUrlTemplate)
                    .mapValues(MobileAppValidationService::extractDomain)
                    .toMap();
        }

        private Constraint<String, Defect> createFor(@Nullable MobileAppTrackerTrackingSystem trackingSystem,
                                                     Map<MobileAppTrackerTrackingSystem, String> domain) {
            if (trackingSystem == OTHER) {
                // нет никакого соотвествия
                return url -> null;
            } else {
                return Constraint.fromPredicate(
                        url -> !domain.containsKey(trackingSystem) ||
                                Objects.equals(extractDomain(url), domain.get(trackingSystem)),
                        trackerUrlDomainDoesNotMatchSelectedTrackerSystem()
                );
            }
        }

        Constraint<String, Defect> createForClick(@Nullable MobileAppTrackerTrackingSystem trackingSystem) {
            return createFor(trackingSystem, trackingSystemToClickDomain);
        }

        Constraint<String, Defect> createForImpression(@Nullable MobileAppTrackerTrackingSystem trackingSystem) {
            return createFor(trackingSystem, trackingSystemToImpressionDomain);
        }
    }
}
