package ru.yandex.direct.core.entity.mobileapp.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.context.annotation.Bean;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.mobileapp.model.MobileAppStoreType;
import ru.yandex.direct.utils.UrlUtils;
import ru.yandex.direct.utils.model.UrlParts;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Arrays.asList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.core.entity.mobileapp.model.MobileAppStoreType.APPLEAPPSTORE;
import static ru.yandex.direct.core.entity.mobileapp.model.MobileAppStoreType.GOOGLEPLAYSTORE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Service
public class TrackingUrlNormalizerService {
    private static final TrackingUrlNormalizerService INSTANCE = new TrackingUrlNormalizerService();

    private static final Pattern MACRO_PATTERN = Pattern.compile("\\{(.*?)}");
    private static final String CAMPAIGN_NAME_LAT_MACRO = "{campaign_name_lat}";
    private static final String IOS_IFA_MACRO = "{ios_ifa}";
    private static final String GOOGLE_AID_MACRO = "{google_aid}";
    private static final String LOGID_MACRO = "{logid}";

    private static final Map<String, String> MACRO_REPLACEMENTS = ImmutableMap.<String, String>builder()
            .put("{ANDROIDID}", "")
            .put("{GOOGLEAID}", GOOGLE_AID_MACRO)
            .put("{IOSIFA}", IOS_IFA_MACRO)
            .build();

    private static final List<String> MACRO_PREFIXES_TO_DELETE = asList("GOOGLE_AID_LC_", "ANDROID_ID_LC_", "IDFA_LC_");
    private static final Pattern MACRO_HAS_PREFIX_TO_DELETE = Pattern.compile("^(?:"
            + String.join("|", mapList(MACRO_PREFIXES_TO_DELETE, macro -> "\\{" + macro))
            + ")");

    private static final Map<String, String> QUERY_PARAMETERS_WITH_DEFAULT_VALUES =
            ImmutableMap.of("utm_campaign", CAMPAIGN_NAME_LAT_MACRO);
    private static final Set<String> QUERY_PARAMETERS_TO_DELETE_IF_WITHOUT_MACRO = singleton("utm_content");
    private static final Set<String> QUERY_PARAMETERS_TO_DELETE = singleton("yclid");
    private static final Map<TrackerType, TrackerSpecificHandler> TRACKER_HANDLERS;

    static {
        List<String> tuneDomainPatterns = asList(
                "(^|\\.)hastrk[0-9]{1,2}\\.com$",
                "(^|\\.)api-[0-9]{1,2}\\.com$",
                "(^|\\.)measurementapi\\.com$",
                "(^|\\.)tlnk\\.io$");

        List<String> flurryDomainPatterns = asList("(^|\\.)ad\\.apps\\.fm$", "(^|\\.)flurry\\.com$");

        String appmetricaDomainPattern = "^redirect\\.appmetrica\\.yandex\\.(?:com|ru)$";

        // TODO можно как-то совместить с ru.yandex.direct.core.entity.mobileapp.Constants.TRACKING_SYSTEMS_CLICK/IMPRESSION?
        TRACKER_HANDLERS = ImmutableMap.<TrackerType, TrackerSpecificHandler>builder()

                .put(TrackerType.TUNE, new TrackerSpecificHandlerBuilder(tuneDomainPatterns)
                        .fixedParameterForStore(APPLEAPPSTORE, "ios_ifa", IOS_IFA_MACRO)
                        .fixedParameterForStore(GOOGLEPLAYSTORE, "google_aid", GOOGLE_AID_MACRO)
                        .logIdParameter("publisher_ref_id")

                        .defaultParameter("sub_campaign", "{campaign_id}_{campaign_name}")
                        .defaultParameter("sub_adgroup", "{gbid}")
                        .defaultParameter("sub_ad", "{ad_id}")
                        .defaultParameter("sub_keyword", "{phrase_id}{retargeting_id}_{keyword}{adtarget_name}")

                        .build())

                .put(TrackerType.APPSFLYER, new TrackerSpecificHandlerBuilder("^app\\.appsflyer\\.com$")
                        .fixedParameterForStore(APPLEAPPSTORE, "idfa", IOS_IFA_MACRO)
                        .fixedParameterForStore(GOOGLEPLAYSTORE, "advertising_id", GOOGLE_AID_MACRO)
                        .logIdParameter("clickid")

                        .defaultParameter("c", CAMPAIGN_NAME_LAT_MACRO)
                        .defaultParameter("af_c_id", "{campaign_id}")
                        .defaultParameter("af_adset_id", "{gbid}")
                        .defaultParameter("af_ad_id", "{ad_id}")
                        .defaultParameter("af_keywords", "{phrase_id}{retargeting_id}_{keyword}{adtarget_name}")
                        .defaultParameter("af_siteid", "{source_type}_{source}")

                        .build())

                .put(TrackerType.ADJUST, new TrackerSpecificHandlerBuilder("(^|\\.)adjust\\.(com|io)")

                        .fixedParameterForStore(APPLEAPPSTORE, "idfa", IOS_IFA_MACRO)
                        .fixedParameterForStore(GOOGLEPLAYSTORE, "gps_adid", GOOGLE_AID_MACRO)

                        .defaultParameter("campaign", "{campaign_id}_{campaign_name}")
                        .defaultParameter("adgroup", "{gbid}")
                        .defaultParameter("creative", "{ad_id}_{phrase_id}{retargeting_id}_{keyword}{adtarget_name}")

                        .build())

                .put(TrackerType.APPMETRICA, new TrackerSpecificHandlerBuilder(appmetricaDomainPattern)

                        .fixedParameterForStore(APPLEAPPSTORE, "ios_ifa", IOS_IFA_MACRO)
                        .fixedParameterForStore(GOOGLEPLAYSTORE, "google_aid", GOOGLE_AID_MACRO)
                        .logIdParameter("click_id")

                        .defaultParameter("c", "{campaign_id}_{campaign_name_lat}")
                        .defaultParameter("adgroup_id", "{gbid}")
                        .defaultParameter("creative_id", "{ad_id}")
                        .defaultParameter("criteria", "{phrase_id}{retargeting_id}_{keyword}{adtarget_name}")
                        .defaultParameter("site_id", "{source_type}_{source}")

                        .build())

                .put(TrackerType.KOCHAVA, new TrackerSpecificHandlerBuilder("(^|\\.)kochava.com$")

                        .fixedParameterForStore(APPLEAPPSTORE, "network_id", "1516")
                        .fixedParameterForStore(APPLEAPPSTORE, "device_id_type", "idfa")
                        .fixedParameterForStore(APPLEAPPSTORE, "idfa", IOS_IFA_MACRO)
                        .fixedParameterForStore(APPLEAPPSTORE, "device_id", IOS_IFA_MACRO)

                        .fixedParameterForStore(GOOGLEPLAYSTORE, "network_id", "1517")
                        .fixedParameterForStore(GOOGLEPLAYSTORE, "device_id_type", "adid")
                        .fixedParameterForStore(GOOGLEPLAYSTORE, "adid", GOOGLE_AID_MACRO)
                        .fixedParameterForStore(GOOGLEPLAYSTORE, "device_id", GOOGLE_AID_MACRO)

                        .logIdParameter("click_id")

                        .defaultParameter("campaign_id", CAMPAIGN_NAME_LAT_MACRO)
                        .defaultParameter("site_id", "{source_type}_{source}")
                        .defaultParameter("creative_id", "{ad_id}_{phrase_id}{retargeting_id}_{keyword}{adtarget_name}")

                        .build())

                .put(TrackerType.FLURRY, new TrackerSpecificHandlerBuilder(flurryDomainPatterns)

                        .fixedParameterForStore(APPLEAPPSTORE, "ios_ifa", IOS_IFA_MACRO)
                        .fixedParameterForStore(GOOGLEPLAYSTORE, "google_aid", GOOGLE_AID_MACRO)
                        .logIdParameter("reqid")

                        .build())

                .build();

        checkState(Arrays.stream(TrackerType.values()).allMatch(TRACKER_HANDLERS::containsKey));
    }

    private TrackingUrlNormalizerService() {
        // поскольку никакого состояния у бина нет, никакой логики в конструкторе не требуется
    }

    @Bean
    public static TrackingUrlNormalizerService instance() {
        return INSTANCE;
    }

    public String normalizeTrackingUrl(String trackingUrl, MobileAppStoreType storeType) {
        String urlWithCanonizedMacros = normalizeMacros(trackingUrl);

        UrlParts urlParts = UrlUtils.laxParseUrl(urlWithCanonizedMacros);
        if (!urlParts.getProtocol().equals("http") && !urlParts.getProtocol().equals("https")) {
            throw new IllegalArgumentException("invalid protocol " + urlParts.getProtocol()
                    + " in tracking URL " + trackingUrl);
        }

        String domain = urlParts.getDomain();
        final List<Pair<String, String>> updatedParameters;
        if (urlParts.getParameters() != null) {
            updatedParameters = updateParameters(domain, storeType, urlParts.getParameters());
        } else {
            updatedParameters = null;
        }

        UrlParts resultingUrl = urlParts.toBuilder()
                .withProtocol("https")
                .withParameters(updatedParameters)
                .build();

        return resultingUrl.toUrl();
    }

    String normalizeMacros(String trackingUrl) {
        Matcher matcher = MACRO_PATTERN.matcher(trackingUrl);

        StringBuilder result = new StringBuilder();
        int lastEnd = 0;

        while (matcher.find()) {
            String macro = matcher.group();

            final String replacement;
            if (MACRO_REPLACEMENTS.containsKey(macro)) {
                replacement = MACRO_REPLACEMENTS.get(macro);
            } else if (MACRO_HAS_PREFIX_TO_DELETE.matcher(macro).find()) {
                replacement = "";
            } else {
                replacement = macro.toLowerCase();
            }

            result.append(trackingUrl, lastEnd, matcher.start());
            result.append(replacement);

            lastEnd = matcher.end();
        }

        result.append(trackingUrl, lastEnd, trackingUrl.length());
        return result.toString();
    }

    List<Pair<String, String>> updateParameters(String domain, MobileAppStoreType storeType,
                                                Collection<Pair<String, String>> queryParameters) {
        List<Pair<String, String>> parametersWithCommonFiltersApplied = queryParameters.stream()
                .filter(param -> param.getValue() != null
                        && !param.getValue().isEmpty()
                        && !param.getValue().equals("_"))
                .filter(param -> !QUERY_PARAMETERS_TO_DELETE.contains(param.getKey()))
                .filter(param -> !QUERY_PARAMETERS_TO_DELETE_IF_WITHOUT_MACRO.contains(param.getKey())
                        || containsMacro(param.getValue()))
                .map(param -> {
                    if (QUERY_PARAMETERS_WITH_DEFAULT_VALUES.containsKey(param.getKey()) && !containsMacro(
                            param.getValue())) {
                        return Pair.of(param.getKey(), QUERY_PARAMETERS_WITH_DEFAULT_VALUES.get(param.getKey()));
                    }
                    return param;
                })
                .collect(toList());

        Optional<TrackerSpecificHandler> trackerHandlerOptional = Arrays.stream(TrackerType.values())
                .map(TRACKER_HANDLERS::get)
                .filter(handler -> handler.handlesDomain(domain))
                .findFirst();

        if (!trackerHandlerOptional.isPresent()) {
            return parametersWithCommonFiltersApplied;
        }

        return filterTrackerSpecificParameters(trackerHandlerOptional.get(), parametersWithCommonFiltersApplied, storeType);
    }

    private List<Pair<String, String>> filterTrackerSpecificParameters(TrackerSpecificHandler handler,
                                                                       List<Pair<String, String>> parameters, MobileAppStoreType storeType) {
        Map<String, Pair<String, String>> fixedParametersForStoreByName =
                handler.getFixedParametersForStoreByName(storeType);
        Map<String, Pair<String, String>> defaultParametersByName = handler.getDefaultParametersByName();
        Set<String> seenParameterNames = new HashSet<>();

        List<Pair<String, String>> result = StreamEx.of(parameters)
                .map(parameter -> {
                    String name = parameter.getKey();

                    if (fixedParametersForStoreByName.containsKey(name)) {
                        return fixedParametersForStoreByName.get(name);
                    }

                    String value = parameter.getValue();
                    if (defaultParametersByName.containsKey(name) && (value.isEmpty() || !containsMacro(value))) {
                        return defaultParametersByName.get(name);
                    }

                    return parameter;
                })
                .peek(parameter -> seenParameterNames.add(parameter.getKey()))
                .toCollection(ArrayList::new);

        List<Pair<String, String>> fixedParametersForStore = handler.getFixedParametersForStore(storeType);
        List<Pair<String, String>> defaultParameters = handler.getDefaultParameters();
        result.addAll(StreamEx.of(fixedParametersForStore).append(defaultParameters)
                .filter(definedParameter -> !seenParameterNames.contains(definedParameter.getKey()))
                .toList());

        return result;
    }

    private static boolean containsMacro(String value) {
        return MACRO_PATTERN.matcher(value).find();
    }

    // TODO если в ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerTrackingSystem появится kochava, стоит совместить
    enum TrackerType {
        TUNE,
        APPSFLYER,
        ADJUST,
        APPMETRICA,
        KOCHAVA,
        FLURRY
    }

    static class TrackerSpecificHandler {
        private final Pattern domainMatchingPattern;
        private final Map<MobileAppStoreType, Map<String, Pair<String, String>>> fixedParametersByStoreAndName;
        private final Map<String, Pair<String, String>> defaultParametersByName;
        private final Map<MobileAppStoreType, List<Pair<String, String>>> fixedParametersByStore;
        private final List<Pair<String, String>> defaultParameters;

        TrackerSpecificHandler(Pattern domainMatchingPattern,
                               Map<MobileAppStoreType, Map<String, Pair<String, String>>> fixedParametersByStoreAndName,
                               Map<String, Pair<String, String>> defaultParametersByName,
                               Map<MobileAppStoreType, List<Pair<String, String>>> fixedParametersByStore,
                               List<Pair<String, String>> defaultParameters) {
            this.domainMatchingPattern = domainMatchingPattern;

            this.fixedParametersByStoreAndName =
                    immutableCopyOfByStoreMap(fixedParametersByStoreAndName, ImmutableMap::copyOf);
            this.defaultParametersByName = ImmutableMap.copyOf(defaultParametersByName);
            this.fixedParametersByStore = immutableCopyOfByStoreMap(fixedParametersByStore, ImmutableList::copyOf);
            this.defaultParameters = ImmutableList.copyOf(defaultParameters);
        }

        private <T> Map<MobileAppStoreType, T> immutableCopyOfByStoreMap(Map<MobileAppStoreType, T> source,
                                                                         Function<T, T> valueCopier) {
            return EntryStream.of(source)
                    .mapValues(valueCopier)
                    .toMapAndThen(ImmutableMap::copyOf);
        }

        boolean handlesDomain(String domain) {
            return domainMatchingPattern.matcher(domain).find();
        }

        Map<String, Pair<String, String>> getFixedParametersForStoreByName(MobileAppStoreType storeType) {
            return fixedParametersByStoreAndName.get(storeType);
        }

        Map<String, Pair<String, String>> getDefaultParametersByName() {
            return defaultParametersByName;
        }

        List<Pair<String, String>> getFixedParametersForStore(MobileAppStoreType storeType) {
            return fixedParametersByStore.get(storeType);
        }

        List<Pair<String, String>> getDefaultParameters() {
            return defaultParameters;
        }
    }

    static class TrackerSpecificHandlerBuilder {
        private final Pattern domainMatchingPattern;
        private final Map<MobileAppStoreType, Map<String, Pair<String, String>>> fixedParametersByStoreAndName;
        private final Map<String, Pair<String, String>> defaultParametersByName;
        private final Map<MobileAppStoreType, List<Pair<String, String>>> fixedParametersByStore;
        private final List<Pair<String, String>> defaultParameters;

        TrackerSpecificHandlerBuilder(List<String> domainMatchingPatterns) {
            domainMatchingPattern = Pattern.compile("(?:"
                    + String.join("|", domainMatchingPatterns)
                    + ")");

            fixedParametersByStoreAndName = newByStoreMap(HashMap::new);
            defaultParametersByName = new HashMap<>();
            fixedParametersByStore = newByStoreMap(ArrayList::new);
            defaultParameters = new ArrayList<>();
        }

        TrackerSpecificHandlerBuilder(String domainMatchingPattern) {
            this(singletonList(domainMatchingPattern));
        }

        private static <T> Map<MobileAppStoreType, T> newByStoreMap(Supplier<T> valueSupplier) {
            return Stream.of(APPLEAPPSTORE, GOOGLEPLAYSTORE)
                    .collect(toMap(identity(),
                            storeType -> valueSupplier.get(),
                            (u, v) -> {
                                throw new IllegalStateException(String.format("Duplicate key %s", u));
                            },
                            HashMap::new));
        }

        TrackerSpecificHandlerBuilder fixedParameterForStore(MobileAppStoreType storeType,
                                                             String name, String value) {
            Pair<String, String> parameter = Pair.of(name, value);
            return fixedParameterForStore(storeType, parameter);
        }

        private TrackerSpecificHandlerBuilder fixedParameterForStore(MobileAppStoreType storeType,
                                                                     Pair<String, String> parameter) {
            fixedParametersByStoreAndName.get(storeType).put(parameter.getKey(), parameter);
            fixedParametersByStore.get(storeType).add(parameter);

            return this;
        }

        TrackerSpecificHandlerBuilder logIdParameter(String name) {
            Pair<String, String> parameter = Pair.of(name, LOGID_MACRO);
            fixedParameterForStore(APPLEAPPSTORE, parameter);
            fixedParameterForStore(GOOGLEPLAYSTORE, parameter);
            return this;
        }

        TrackerSpecificHandlerBuilder defaultParameter(String name, String value) {
            Pair<String, String> parameter = Pair.of(name, value);
            defaultParametersByName.put(name, parameter);
            defaultParameters.add(parameter);
            return this;
        }

        TrackerSpecificHandler build() {
            return new TrackerSpecificHandler(domainMatchingPattern,
                    fixedParametersByStoreAndName, defaultParametersByName,
                    fixedParametersByStore, defaultParameters);
        }
    }
}
