package ru.yandex.direct.core.entity.mobilecontent.container;

import java.util.Collections;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.mobilecontent.model.ContentType;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.uac.model.Store;


/**
 * Разобранное представление URL cтора (Google Play или Apple App Store)
 */
@ParametersAreNonnullByDefault
public class MobileAppStoreUrl {
    public static final String DEFAULT_COUNTRY = "RU";
    public static final String DEFAULT_LANGUAGE = "RU";

    private final OsType osType;
    private final Store store;
    private final ContentType contentType;
    private final String storeCountry;
    private final String storeLanguage;
    private final String storeContentId;
    private final boolean isDefaultCountry;

    public MobileAppStoreUrl(OsType osType, Store store, ContentType contentType, String storeCountry,
                             String storeLanguage, String storeContentId,
                             boolean isDefaultCountry) {
        this.osType = osType;
        this.store = store;
        this.contentType = contentType;
        this.storeCountry = storeCountry;
        this.storeLanguage = storeLanguage;
        this.storeContentId = storeContentId;
        this.isDefaultCountry = isDefaultCountry;
    }

    public OsType getOsType() {
        return osType;
    }

    public static MobileAppStoreUrl fromMobileContent(MobileContent mobileContent) {
        return new MobileAppStoreUrl(mobileContent.getOsType(),
                mobileContent.getOsType() == OsType.IOS ? Store.ITUNES : Store.GOOGLE_PLAY,
                mobileContent.getContentType(), mobileContent.getStoreCountry(), DEFAULT_LANGUAGE,
                mobileContent.getStoreContentId(),
                false);
    }

    public ContentType getContentType() {
        return contentType;
    }

    public String getStoreCountry() {
        return storeCountry;
    }

    public Store getStore() {
        return store;
    }

    public String getStoreContentId() {
        return storeContentId;
    }

    public boolean isDefaultCountry() {
        return isDefaultCountry;
    }

    public MobileContent toMobileContent() {
        return new MobileContent()
                .withOsType(osType)
                .withContentType(contentType)
                .withStoreContentId(storeContentId)
                .withStoreCountry(storeCountry)
                .withPrices(Collections.emptyMap());
    }

    public String getStoreLanguage() {
        return storeLanguage;
    }

    @Override
    public String toString() {
        return "MobileAppStoreUrl{" +
                "osType=" + osType +
                ", contentType=" + contentType +
                ", storeCountry='" + storeCountry + '\'' +
                ", storeContentId='" + storeContentId + '\'' +
                ", isDefaultCountry=" + isDefaultCountry +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        MobileAppStoreUrl that = (MobileAppStoreUrl) o;
        return osType == that.osType &&
                contentType == that.contentType &&
                Objects.equals(storeCountry, that.storeCountry) &&
                Objects.equals(storeContentId, that.storeContentId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(osType, contentType, storeCountry, storeContentId);
    }
}
