package ru.yandex.direct.core.entity.mobilecontent.converter;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.CharMatcher;
import com.google.common.base.Utf8;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.EnumUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.service.validation.BannerLettersConstants;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.mobilecontent.model.AgeLabel;
import ru.yandex.direct.core.entity.mobilecontent.model.ApiMobileContentYT;
import ru.yandex.direct.core.entity.mobilecontent.model.AvailableAction;
import ru.yandex.direct.core.entity.mobilecontent.model.ContentType;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentExternalWorldMoney;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.mobilecontent.model.StoreActionForPrices;
import ru.yandex.direct.core.entity.mobilecontent.model.StoreCountry;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.inside.yt.kosher.impl.ytree.serialization.YTreeTextSerializer;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonMap;
import static java.util.stream.Collectors.joining;
import static ru.yandex.direct.core.entity.mobilecontent.service.MobileContentServiceConstants.FIXED_DOMAINS;
import static ru.yandex.direct.core.entity.mobilecontent.service.MobileContentServiceConstants.YT_AGE_LABELS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Конвертер для {@link MobileContent} и данных по мобильным приложениям из таблицы в YT
 */
@Service
@ParametersAreNonnullByDefault
public class MobileContentYtConverter {
    private static final Logger logger = LoggerFactory.getLogger(MobileContentYtConverter.class);
    private static final Pattern ICON_URL_TO_HASH = Pattern.compile("(\\d+\\/(?:\\S+__)?[0-9,a-f]{32})");
    private static final String OS_VERSION_PARTS_REGEX = "(?:\\.\\s*|\\s+)";
    private static final int MAX_VERSION_PARTS_NUM = 2;
    private static final ObjectMapper MAPPER = new ObjectMapper()
            .configure(JsonParser.Feature.ALLOW_MISSING_VALUES, true);
    private static final CharMatcher ALLOWED_NAME_CHARS =
            CharMatcher.anyOf(BannerLettersConstants.ALLOW_BANNER_LETTERS_STR);
    public static final String UNDEFINED_NAME = "undefined";

    private final DomainService domainService;
    private final DslContextProvider dslContextProvider;

    public MobileContentYtConverter(DomainService domainService, DslContextProvider dslContextProvider) {
        this.domainService = domainService;
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Достает из MDS URL хеш картинки
     *
     * @param url URL из MDS типа http://avatars.mds.yandex.net/get-$(shop)/$(hash/hash)/$(alias)
     */
    @Nullable
    static String iconUrlToHash(@Nonnull String url) {
        Matcher m = ICON_URL_TO_HASH.matcher(url);
        return m.find() ? m.group(1) : null;
    }

    /**
     * Маппер содержимого таблицы YT в {@link MobileContent}.
     * <p>
     * Side-эффект: Домены не создает и не назначает сам, а складывает их в мап websites. Иначе при массовом
     * апдейте идет слишком много запросов к базе
     */
    public MobileContent ytToMobileContent(YTreeMapNode row, OsType osType, Map<String, Optional<String>> websites) {
        Map<StoreActionForPrices, MobileContentExternalWorldMoney> prices = StreamEx
                .of(StoreActionForPrices.values())
                .mapToEntry(storeActionForPrices -> new MobileContentExternalWorldMoney()
                        .withSum(BigDecimal.valueOf(row.getDoubleO("price").orElse(0.0)))
                        .withCurrency(row.getStringO("currency").orElse("USD")))
                .toMap();

        String appId = row.getString("app_id");
        websites.put(appId, row.get("website")
                .filter(YTreeNode::isStringNode)
                .map(YTreeNode::stringValue));

        @Nullable StoreCountry country = EnumUtils.getEnum(StoreCountry.class, row.getString("lang").toUpperCase());
        String minOsVersion = getMinOsVersion(row.getStringO("os_version").orElse(""));
        String name = ALLOWED_NAME_CHARS.retainFrom(row.getString("name"));
        if (name.isBlank()) {
            name = UNDEFINED_NAME;
        }
        return new MobileContent()
                .withId(0L)
                .withName(name)
                .withContentType(ContentType.APP)
                .withOsType(osType)
                .withStoreCountry(row.getString("lang").toUpperCase())
                .withStoreContentId(appId)
                .withIconHash(row.getStringO("icon").map(MobileContentYtConverter::iconUrlToHash).orElse(null))
                .withRating(convertRating(row.getDoubleO("rating")))
                .withRatingVotes(row.getLongO("rating_count").orElse(0L))
                .withAgeLabel(Optional.ofNullable(YT_AGE_LABELS.get(osType))
                        .flatMap(os -> Optional.ofNullable(os.get(country)))
                        .flatMap(c -> row.getStringO("adult").map(c::get))
                        .orElse(AgeLabel._18_2B))
                .withGenre(getGenres(row))
                .withBundleId(row.getStringO("bundle").orElse(null))
                .withMinOsVersion(minOsVersion.length() <= 10 ? minOsVersion : minOsVersion.substring(0, 10))
                .withPrices(singletonMap(row.getString("lang").toUpperCase(), prices))
                .withAvailableActions(singleton(AvailableAction.download))
                .withDownloads(row.getStringO("downloads").map(MobileContentYtConverter::convertDownloads).orElse(null))
                .withScreens(MobileContentYtConverter.getScreensEx(row))
                .withIsAvailable(true);
    }

    private String retainOnlyUtf8Mb3Symbols(String name) {
        return name.codePoints()
                .filter(cp -> Utf8.encodedLength(String.valueOf(Character.toChars(cp))) <= 3)
                .collect(StringBuilder::new, StringBuilder::appendCodePoint, StringBuilder::append)
                .toString();
    }

    /**
     * Маппер содержимого таблицы YT в {@link ApiMobileContentYT}.
     *
     * @param store Тип стора
     * @param row   Строка таблицы с необходимыми данными
     * @return Смапленные данные мобильного приложения
     */
    public ApiMobileContentYT ytToApiMobileContent(String store, YTreeMapNode row) {
        return new ApiMobileContentYT()
                .withStore(store)
                .withAppId(row.getString("app_id"))
                .withLang(row.getString("lang"))
                .withBundle(row.getStringO("bundle").orElse(null))
                .withName(retainOnlyUtf8Mb3Symbols(row.getString("name")))
                .withIcon(row.getStringO("icon").orElse(null))
                .withCurrency(row.getStringO("currency").orElse(null))
                .withPublisher(row.getString("publisher"))
                .withWebsite(row.getStringO("website").orElse(null))
                .withRatingCount(row.getLongO("rating_count").orElse(null));
    }

    @NotNull
    static BigDecimal convertRating(Optional<Double> ytRating) {
        // Округляем до двух знаков, т.к. в ppc.mobile_content `rating` decimal(4,2) unsigned
        // и при записи будет всё равно округлён
        // неокруглённый же рейтинг невозможно сравнить с рейтингом из БД
        double rawRating = ytRating.orElse(0d);
        if (rawRating < 0) {
            rawRating = 0d;
        }
        return BigDecimal.valueOf(rawRating).setScale(2, RoundingMode.HALF_UP);
    }

    /**
     * Получение кол-во скачиваний приложения
     * <p>
     * Может быть в виде таких строк: "10+", "1000000+", "1.000.000+", "10,000+"
     */
    @Nullable
    static Long convertDownloads(String ytDownloads) {
        try {
            return Long.valueOf(ytDownloads.replaceAll("[+.,]", ""));
        } catch (NumberFormatException e) {
            logger.warn(String.format("Couldn't convert 'downloads' value '%s' to number", ytDownloads));
            return null;
        }
    }

    static List<Map<String, String>> getScreensEx(YTreeMapNode row) {
        Optional<String> screensYson = row.getStringO("screens_ex");
        if (screensYson.isEmpty() || screensYson.get().isEmpty()) {
            return List.of();
        }
        YTreeNode node = YTreeTextSerializer.deserialize(screensYson.get());
        if (!node.isListNode()) {
            logger.warn("Couldn't convert 'screens' value '{}'", node);
            return emptyList();
        }
        return StreamEx.of(node.listNode().iterator())
                .filter(YTreeNode::isMapNode)
                .map(YTreeNode::mapNode)
                .map(MobileContentYtConverter::convertMapOfString)
                .toList();
    }

    private static Map<String, String> convertMapOfString(YTreeMapNode map) {
        return EntryStream.of(map.iterator())
                .mapValues(value -> {
                    String converted = null;
                    if (value.isStringNode()) {
                        converted = value.stringValue();
                    } else if (value.isIntegerNode()) {
                        converted = Integer.toString(value.intValue());
                    } else if (value.isDoubleNode()) {
                        converted = Double.toString(value.doubleValue());
                    }
                    return converted;
                })
                .nonNullValues()
                .toMap();
    }

    /**
     * Некрасивый костыль для приведения списка доменов из Yson в comma-separated строку.
     */
    String getGenres(YTreeMapNode row) {
        Optional<String> genres = row.getStringO("genres");
        if (genres.isEmpty()) {
            return null;
        }
        String genresJson = genres.get().replaceAll(";", ",");
        try {
            String[] genresArrays = MAPPER.readerFor(String[].class).readValue(genresJson);
            return Arrays.stream(genresArrays).filter(Objects::nonNull).collect(joining(","));
        } catch (IOException e) {
            logger.error("Unable to parse genres string: " + genres.get());
            return null;
        }
    }

    /**
     * Возвращает ID доменов в табличке ppc.domains в привязке в appId.
     * <p>
     * Если приложение занесено в список
     * {@link ru.yandex.direct.core.entity.mobilecontent.service.MobileContentServiceConstants#FIXED_DOMAINS},
     * то домен берется из этого списка.
     * <p>
     * Если домена в табличке нет - создается новая запись и возвращается ее ID.
     */
    @Nonnull
    public Map<String, Long> getDomainIds(int shard, Map<String, Optional<String>> websitesByAppId) {
        List<Map.Entry<String, String>> domainsToAppIds = EntryStream.of(websitesByAppId)
                .mapToValue((appId, website) -> {
                    if (FIXED_DOMAINS.containsKey(appId)) {
                        return FIXED_DOMAINS.get(appId);
                    } else if (website.isPresent()) {
                        try {
                            return new URL(website.get()).getHost();
                        } catch (Exception e) {
                            logger.warn("Unable to parse domain: " + website);
                        }
                    }
                    return null;
                })
                .filterValues(Objects::nonNull)
                .toList();

        List<String> appIds = mapList(domainsToAppIds, Map.Entry::getKey);
        List<String> domains = mapList(domainsToAppIds, Map.Entry::getValue);
        List<Long> domainIds = domainService.getOrCreate(dslContextProvider.ppc(shard), domains);
        if (domainIds.size() != domains.size()) {
            throw new IllegalStateException("Domain repository returned less IDs than it was asked for");
        }
        return EntryStream.zip(appIds, domainIds).toMap();
    }

    /**
     * Получение численного значения минимальной версии ОС. Сделано по аналогии с перлом.
     * <p>
     * Раньше определение минимальной версии не всегда работало корректно, но вроде как это поправили в EXTDATA-1723
     */
    String getMinOsVersion(String fromYt) {
        String[] osVersionParts = fromYt.trim().split(OS_VERSION_PARTS_REGEX, MAX_VERSION_PARTS_NUM + 1);
        return Arrays.stream(osVersionParts)
                .limit(MAX_VERSION_PARTS_NUM)
                .map(String::trim)
                .filter(StringUtils::isNumeric)
                .collect(joining("."));
    }
}
