package ru.yandex.direct.core.entity.mobilecontent.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.InsertValuesStep1;
import org.jooq.types.ULong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentQueueItem;
import ru.yandex.direct.dbschema.ppcdict.tables.records.MobileContentFetchQueueRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.dbschema.ppcdict.tables.MobileContentFetchQueue.MOBILE_CONTENT_FETCH_QUEUE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Репозиторий для работы с таблицей ppcdict.mobile_content_fetch_queue
 */
@Repository
@ParametersAreNonnullByDefault
public class MobileContentFetchQueueRepository {

    private static final JooqMapperWithSupplier<MobileContentQueueItem> QUEUE_MAPPER =
            JooqMapperWithSupplierBuilder.builder(MobileContentQueueItem::new)
                    .map(convertibleProperty(MobileContentQueueItem.ID, MOBILE_CONTENT_FETCH_QUEUE.ID, ULong::longValue, ULong::valueOf))
                    .map(property(MobileContentQueueItem.URL, MOBILE_CONTENT_FETCH_QUEUE.URL))
                    .map(property(MobileContentQueueItem.ADD_TIME, MOBILE_CONTENT_FETCH_QUEUE.ADD_TIME))
                    .build();

    private static final Collection<Field<?>> QUEUE_FIELDS_TO_READ = QUEUE_MAPPER.getFieldsToRead();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public MobileContentFetchQueueRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Добавляет ссылку на мобильное приложение в очередь
     *
     * @param url ссылка на мобильное приложение в store
     * @return количество записей, добавленных в таблицу
     */
    public int addUrl(String url) {
        return addUrls(singletonList(url));
    }

    /**
     * Добавляет ссылки на мобильные приложения в очередь
     *
     * @param urls ссылки на мобильные приложения в store
     * @return количество записей, добавленных в таблицу
     */
    public int addUrls(List<String> urls) {
        InsertValuesStep1<MobileContentFetchQueueRecord, String> insertStep = dslContextProvider.ppcdict()
                .insertInto(MOBILE_CONTENT_FETCH_QUEUE, MOBILE_CONTENT_FETCH_QUEUE.URL);

        urls.forEach(insertStep::values);
        return insertStep.execute();
    }

    /**
     * Возвращает все записи таблицы
     *
     * @return все записи таблицы ppcdict.mobile_content_fetch_queue
     */
    public List<MobileContentQueueItem> getAllItems() {
        return dslContextProvider.ppcdict()
                .select(QUEUE_FIELDS_TO_READ)
                .from(MOBILE_CONTENT_FETCH_QUEUE)
                .fetch(QUEUE_MAPPER::fromDb);
    }

    /**
     * Удаляет указанные ссылки из таблицы
     *
     * @param urls ссылки, которые нужно удалить
     * @return количество удаленных записей
     */
    public int deleteItems(List<MobileContentQueueItem> urls) {
        if (urls.isEmpty()) {
            return 0;
        }
        List<Long> ids = mapList(urls, MobileContentQueueItem::getId);

        return dslContextProvider.ppcdict()
                .deleteFrom(MOBILE_CONTENT_FETCH_QUEUE)
                .where(MOBILE_CONTENT_FETCH_QUEUE.ID.in(ids))
                .execute();
    }
}
