package ru.yandex.direct.core.entity.mobilecontent.repository;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record3;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileContentWithExtraData;
import ru.yandex.direct.core.entity.mobilecontent.model.AgeLabel;
import ru.yandex.direct.core.entity.mobilecontent.model.ContentType;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentForBsTransport;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.mobilecontent.model.StatusIconModerate;
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser;
import ru.yandex.direct.core.entity.notification.container.MobileContentMonitoringNotification;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbschema.ppc.enums.MobileContentContentType;
import ru.yandex.direct.dbschema.ppc.enums.MobileContentOsType;
import ru.yandex.direct.dbschema.ppc.enums.MobileContentStatusbssynced;
import ru.yandex.direct.dbschema.ppc.tables.records.MobileContentRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.time.LocalDateTime.now;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentMapping.statusBsSyncedToDbFormat;
import static ru.yandex.direct.dbschema.ppc.tables.AdgroupsMobileContent.ADGROUPS_MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.MobileContent.MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.dbschema.ppc.tables.Users.USERS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromPropertyToField;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Репозиторий для работы с таблицей ppc.mobile_content
 */
@Repository
@ParametersAreNonnullByDefault
public class MobileContentRepository {
    public static final JooqMapperWithSupplier<MobileContent> MAPPER_FOR_MOBILE_CONTENT_FIELDS =
            createMobileContentMapper();

    private static final Collection<TableField<?, ?>> MOBILE_CONTENT_DB_KEY_AND_ID_FIELDS = ImmutableList.copyOf(
            Iterables.concat(MobileContentDbKey.FIELDS_TO_READ, singleton(MOBILE_CONTENT.MOBILE_CONTENT_ID)));

    final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final JooqMapperWithSupplier<MobileContent> mobileContentJooqMapper;
    private final Set<Field<?>> allFields;

    @Autowired
    public MobileContentRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;

        mobileContentJooqMapper = createMobileContentMapper();
        allFields = mobileContentJooqMapper.getFieldsToRead();
    }

    /**
     * Получить список самых старых объектов пригодных для экспорта мобильного контента в БК.
     *
     * @param shard            шард
     * @param mobileContentIds список идентификаторов объектов, которые нужно получить, или null, если нужно получить
     * @param limitOffset      максимальное количество объектов в результате и смещение относительно начала выборки
     */
    public List<MobileContentForBsTransport> getMobileContentForBsExport(int shard, Collection<Long> mobileContentIds,
                                                                         LimitOffset limitOffset) {
        return mapList(getMobileContentWithStatusBsSynced(shard, StatusBsSynced.SENDING, mobileContentIds,
                mobileContentJooqMapper.getFieldsToRead(MobileContentForBsTransport.allModelProperties()),
                limitOffset), m -> m);
    }

    /**
     * Получить список самых старых объектов с заданным статусом синхронизации с БК с заданными полями.
     * Список будет отсортирован по дате модификации объектов и, затем, по их идентификатору
     *
     * @param shard            шард
     * @param statusBsSynced   ожидаемый статус синхронизации с БК для получаемых объектов
     * @param mobileContentIds список идентификаторов объектов, которые нужно получить, или null, если нужно получить
     *                         все доступные в рамках лимита
     * @param fieldsToRead     список полей для получения
     * @param limitOffset      максимальное количество объектов в результате и смещение относительно начала выборки
     */
    private List<MobileContent> getMobileContentWithStatusBsSynced(int shard, StatusBsSynced statusBsSynced,
                                                                   Collection<Long> mobileContentIds,
                                                                   Collection<Field<?>> fieldsToRead,
                                                                   LimitOffset limitOffset) {
        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.STATUS_BS_SYNCED
                        .eq(MobileContentStatusbssynced.valueOf(statusBsSynced.toDbFormat()))
                        .and(MOBILE_CONTENT.MOBILE_CONTENT_ID.in(mobileContentIds)))
                .orderBy(MOBILE_CONTENT.MODIFY_TIME, MOBILE_CONTENT.MOBILE_CONTENT_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(mobileContentJooqMapper::fromDb);
    }

    /**
     * Выбирает из таблицы ppc.mobile_content все записи со store_refresh_time старше определенного возраста.
     *
     * @param shard         Шард
     * @param maxAge        Возраст данных для выборки
     * @param maxTriesCount Максимальное число предыдущих попыток получения данных, после которого мы больше не будем
     */
    public Stream<MobileContent> getMobileContentOlderThan(int shard, Duration maxAge, long maxTriesCount,
                                                           boolean isAvailable) {
        return dslContextProvider.ppc(shard)
                .select(allFields)
                .from(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.STORE_REFRESH_TIME.le(LocalDateTime.now().minus(maxAge)))
                .and(MOBILE_CONTENT.TRIES_COUNT.le(maxTriesCount))
                .and(MOBILE_CONTENT.IS_AVAILABLE.eq(isAvailable ? 1L : 0L))
                .fetchStream()
                .map(mobileContentJooqMapper::fromDb);
    }


    /**
     * Получить список идентификаторов объектов с заданным статусом синхронизации с БК
     *
     * @param shard          шард
     * @param expectedStatus ожидаемый статус синхронизации с БК
     */
    public List<Long> getMobileContentIdsWithStatusBsSynced(int shard, StatusBsSynced expectedStatus) {
        return getMobileContentIdsWithStatusBsSynced(shard, Collections.singletonList(expectedStatus));
    }

    /**
     * Получить список идентификаторов объектов с заданным статусом синхронизации с БК
     *
     * @param shard              шард
     * @param expectedStatusList ожидаемые статус синхронизации с БК
     */
    public List<Long> getMobileContentIdsWithStatusBsSynced(int shard, Collection<StatusBsSynced> expectedStatusList) {
        return dslContextProvider.ppc(shard)
                .select(MOBILE_CONTENT.MOBILE_CONTENT_ID)
                .from(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.STATUS_BS_SYNCED.in(mapList(expectedStatusList,
                        MobileContentMapping::statusBsSyncedToDbFormat)))
                .fetch(MOBILE_CONTENT.MOBILE_CONTENT_ID);
    }

    /**
     * Поменять статус синхронизации с БК для объектов с заданными идентификаторами
     *
     * @param shard            шард
     * @param from             статус, в котором должны находиться объекты. Если их статус отличается, новый статус
     *                         установлен не будет.
     * @param to               статус синхронизации, который нужно установить
     * @param mobileContentIds список идентификаторов объектов, для которых нужно применить изменения
     */
    public int changeStatusBsSynced(int shard, StatusBsSynced from, StatusBsSynced to,
                                    Collection<Long> mobileContentIds) {
        return dslContextProvider.ppc(shard)
                .update(MOBILE_CONTENT)
                .set(MOBILE_CONTENT.STATUS_BS_SYNCED, statusBsSyncedToDbFormat(to))
                .where(MOBILE_CONTENT.STATUS_BS_SYNCED.eq(statusBsSyncedToDbFormat(from))
                        .and(MOBILE_CONTENT.MOBILE_CONTENT_ID.in(mobileContentIds)))
                .execute();
    }

    /**
     * Добавить объект мобильного контента в базу данных
     *
     * @param shard         шард
     * @param mobileContent описание мобильного контента для добавления.
     *                      В этом описании <b>уже должен быть установлен уникальный идентификатор объекта</b>
     * @return идентификатор добавленного объекта
     */
    public Long addMobileContent(int shard, MobileContent mobileContent) {
        return addMobileContent(shard, Collections.singletonList(mobileContent)).get(0);
    }

    /**
     * Добавить несколько объектов мобильного контента в базу данных
     *
     * @param shard             шард
     * @param mobileContentList список описаний мобильного контента для добавления.
     *                          В каждом описании <b>уже должен быть установлен уникальный идентификатор объекта</b>
     * @return список идентификаторов добавленных объектов
     */
    public List<Long> addMobileContent(int shard, Collection<MobileContent> mobileContentList) {
        insertMobileContent(shard, mobileContentList);
        return mapList(mobileContentList, MobileContent::getId);
    }

    /**
     * Добавить несколько объектов мобильного контента в базу данных
     *
     * @param shard             шард
     * @param mobileContentList список описаний мобильного контента для добавления.
     *                          В каждом описании <b>уже должен быть установлен уникальный идентификатор объекта</b>
     */
    private void insertMobileContent(int shard, Collection<MobileContent> mobileContentList) {
        new InsertHelper<>(dslContextProvider.ppc(shard), MOBILE_CONTENT)
                .addAll(mobileContentJooqMapper, mobileContentList)
                .executeIfRecordsAdded();
    }

    /**
     * Получить описание объекта мобильного контента по его идентификатору
     */
    public MobileContent getMobileContent(int shard, long mobileContentId) {
        return getMobileContent(shard, Collections.singletonList(mobileContentId)).get(0);
    }

    /**
     * Получить список описаний объектов мобильного контента по их идентификатору
     */
    public List<MobileContent> getMobileContent(int shard, Collection<Long> mobileContentIds) {
        return getMobileContent(dslContextProvider.ppc(shard), mobileContentIds);
    }

    /**
     * Получить список описаний объектов мобильного контента по их идентификатору для заданных клиентов.
     */
    public List<MobileContent> getMobileContent(int shard, Collection<Long> clientIds,
                                                Collection<Long> mobileContentIds) {
        return getMobileContent(dslContextProvider.ppc(shard), clientIds, mobileContentIds);
    }

    /**
     * Возвращает все записи из ppc.mobile_content с данными id для заданных клиентов.
     *
     * @param dslContext       контекст
     * @param clientIds        id клиентов
     * @param mobileContentIds id мобильных контентов
     */
    public List<MobileContent> getMobileContent(DSLContext dslContext, Collection<Long> clientIds,
                                                Collection<Long> mobileContentIds) {
        if (mobileContentIds.isEmpty()) {
            return emptyList();
        }

        return dslContext.select(allFields)
                .from(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.MOBILE_CONTENT_ID.in(mobileContentIds))
                .and(MOBILE_CONTENT.CLIENT_ID.in(clientIds))
                .fetch(mobileContentJooqMapper::fromDb);
    }

    /**
     * Возвращает все записи из ppc.mobile_content с данными ID.
     * Метод НЕ проверяет ClientID, использовать его в клиентских запросах не следует.
     *
     * @param dslContext       Контекст
     * @param mobileContentIds Набор ID
     */
    public List<MobileContent> getMobileContent(DSLContext dslContext, Collection<Long> mobileContentIds) {
        if (mobileContentIds.isEmpty()) {
            return Collections.emptyList();
        }
        return dslContext.select(allFields)
                .from(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.MOBILE_CONTENT_ID.in(mobileContentIds))
                .fetch(mobileContentJooqMapper::fromDb);
    }

    /**
     * Получить список описаний объектов мобильного контента клиента
     */
    public List<MobileContent> getMobileContent(int shard, ClientId clientId, MobileContentContentType contentType,
                                                @Nullable Boolean isAvailable, LimitOffset limitOffset) {
        SelectConditionStep<Record> selectConditionStep = dslContextProvider.ppc(shard)
                .select(allFields)
                .from(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.CLIENT_ID.eq(clientId.asLong()))
                .and(MOBILE_CONTENT.CONTENT_TYPE.eq(contentType));

        if (isAvailable != null) {
            selectConditionStep = selectConditionStep.and(MOBILE_CONTENT.IS_AVAILABLE.eq(isAvailable ? 1L : 0L));
        }
        return selectConditionStep
                .orderBy(MOBILE_CONTENT.MOBILE_CONTENT_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(mobileContentJooqMapper::fromDb);
    }

    /**
     * Возвращает все записи из ppc.mobile_content с чанками.
     *
     * @param shard       шард
     * @param firstId     начальный mobile_content_id
     * @param limit       размер chunk
     */
    public List<MobileContent> getMobileContentsChunk(int shard, Long firstId, Long limit) {
        return dslContextProvider.ppc(shard)
                .select(allFields)
                .from(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.MOBILE_CONTENT_ID.gt(firstId))
                .orderBy(MOBILE_CONTENT.MOBILE_CONTENT_ID.asc())
                .limit(limit)
                .fetch(mobileContentJooqMapper::fromDb);
    }

    /**
     * Получить список описаний приложений MobileContent с урлом, сохранённым для соответствющей РМП группе
     */
    public List<MobileContentWithExtraData> getStoreHrefWithMobileAppContent(int shard, ClientId clientId,
                                                                             LimitOffset limitOffset) {
        ru.yandex.direct.dbschema.ppc.tables.MobileContent subqueryMobileContentTable =
                MOBILE_CONTENT.as("subqueryMobileContent");
        Field<Long> subqueryMobileContentIdField =
                subqueryMobileContentTable.MOBILE_CONTENT_ID.as("subqueryMobileContentId");

        Field<Long> subquerySomePidField = DSL.select(ADGROUPS_MOBILE_CONTENT.PID)
                .from(ADGROUPS_MOBILE_CONTENT)
                .where(ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID.eq(subqueryMobileContentIdField))
                .limit(1)
                .asField("subquerySomePid");

        Field<Long> subquerySomeBidField = DSL.select(BANNERS.BID)
                .from(BANNERS)
                .innerJoin(ADGROUPS_MOBILE_CONTENT).on(BANNERS.PID.eq(ADGROUPS_MOBILE_CONTENT.PID))
                .where(ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID.eq(subqueryMobileContentIdField))
                .limit(1)
                .asField("subquerySomeBid");

        SelectConditionStep<Record3<Long, Long, Long>> subquery =
                DSL.select(subqueryMobileContentIdField, subquerySomePidField, subquerySomeBidField)
                        .from(subqueryMobileContentTable)
                        .where(subqueryMobileContentTable.CLIENT_ID.eq(clientId.asLong()))
                        .and(subqueryMobileContentTable.CONTENT_TYPE.eq(MobileContentContentType.app))
                        .and(subqueryMobileContentTable.IS_AVAILABLE.eq(1L))
                        .and(subqueryMobileContentTable.ICON_HASH.isNotNull())
                        .and(subqueryMobileContentTable.NAME.isNotNull());

        return dslContextProvider.ppc(shard)
                .select(Sets.union(Sets.newHashSet(ADGROUPS_MOBILE_CONTENT.STORE_CONTENT_HREF, BANNERS.HREF),
                        new HashSet<>(allFields)))
                .from(subquery)
                .innerJoin(MOBILE_CONTENT).on(subqueryMobileContentIdField.eq(MOBILE_CONTENT.MOBILE_CONTENT_ID))
                .innerJoin(ADGROUPS_MOBILE_CONTENT).on(subquerySomePidField.eq(ADGROUPS_MOBILE_CONTENT.PID))
                .innerJoin(BANNERS).on(subquerySomeBidField.eq(BANNERS.BID))
                .orderBy(MOBILE_CONTENT.MOBILE_CONTENT_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(r -> new MobileContentWithExtraData(
                        mobileContentJooqMapper.fromDb(r),
                        r.get(ADGROUPS_MOBILE_CONTENT.STORE_CONTENT_HREF),
                        r.get(BANNERS.HREF)));

    }

    /**
     * Возвращает мап id баннера -> id мобильного контента, привязанный к группе.
     *
     * @param shard     шард
     * @param bannerIds id баннеров
     * @return мап id баннера -> id мобильного контента, привязанного к группе
     */
    public Map<Long, Long> getMobileContentIdsByBannerIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID)
                .from(BANNERS)
                .join(ADGROUPS_MOBILE_CONTENT)
                .on(BANNERS.PID.eq(ADGROUPS_MOBILE_CONTENT.PID))
                .where(BANNERS.BID.in(bannerIds))
                .fetchMap(BANNERS.BID, ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID);
    }

    /**
     * Возвращает отображения Id группы в привязанный к группе объект мобильного контента
     *
     * @param shard      шард для запроса
     * @param adgroupIds коллекция id групп
     */
    public Map<Long, MobileContent> getMobileContentByAdGroupIds(int shard, Collection<Long> adgroupIds) {
        Collection<Field<?>> fields = new HashSet<>(allFields);
        fields.add(ADGROUPS_MOBILE_CONTENT.PID);

        return dslContextProvider.ppc(shard)
                .select(fields)
                .from(ADGROUPS_MOBILE_CONTENT)
                .join(MOBILE_CONTENT)
                .on(MOBILE_CONTENT.MOBILE_CONTENT_ID.eq(ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID))
                .where(ADGROUPS_MOBILE_CONTENT.PID.in(adgroupIds))
                .fetchMap(ADGROUPS_MOBILE_CONTENT.PID, mobileContentJooqMapper::fromDb);
    }


    public List<Long> getAdgroupIdsForMobileContentIds(int shard, Collection<Long> mobileContentIds) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(ADGROUPS_MOBILE_CONTENT.PID)
                .from(ADGROUPS_MOBILE_CONTENT)
                .where(ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID.in(mobileContentIds))
                .fetch(ADGROUPS_MOBILE_CONTENT.PID);
    }

    /**
     * Получить ссылку на приложение в магазине мобильных приложений, привязанных к группе
     *
     * @return отображение adGroupId -> store_href
     */
    public Map<Long, String> getStoreUrlByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        return dslContextProvider.ppc(shard)
                .select(ADGROUPS_MOBILE_CONTENT.PID, ADGROUPS_MOBILE_CONTENT.STORE_CONTENT_HREF)
                .from(ADGROUPS_MOBILE_CONTENT)
                .where(ADGROUPS_MOBILE_CONTENT.PID.in(adGroupIds))
                .fetchMap(ADGROUPS_MOBILE_CONTENT.PID, ADGROUPS_MOBILE_CONTENT.STORE_CONTENT_HREF);
    }

    /**
     * Получить или вставить элементы MobileContent соответствующие разобранным урлам в {@code parsedUrls}.
     * <p>
     * Для каждого входящего элемента в {@code parsedUrls} возвращается идентификатор присутствующей в БД записи
     * MobileContent или вновь вставленной
     *
     * @return id в порядке соответствующем списку {@code parsedUrls}
     */
    public List<Long> getOrCreate(DSLContext dslContext, ClientId clientId, List<MobileAppStoreUrl> parsedUrls) {
        if (parsedUrls.isEmpty()) {
            return emptyList();
        }
        List<MobileContent> mobileContents = StreamEx.of(parsedUrls)
                .map(MobileAppStoreUrl::toMobileContent)
                .peek(mbc -> mbc
                        .withIsAvailable(false)
                        .withCreateTime(now())
                        .withStatusBsSynced(StatusBsSynced.NO)
                        .withStatusIconModerate(StatusIconModerate.READY)
                        .withTriesCount(0))
                .peek(mbc -> mbc.setClientId(clientId.asLong()))
                .toList();
        return getOrCreateMobileContentList(dslContext, clientId, mobileContents);
    }

    /**
     * Для всех объектов {@link MobileAppStoreUrl} возвращает список их ID в базе.
     * Если такого объекта в базе еще нет, то он будет создан.
     *
     * @param shard      Шард
     * @param clientId   ID клиента
     * @param parsedUrls Список обработанных URL
     */
    public List<Long> getOrCreate(int shard, ClientId clientId, List<MobileAppStoreUrl> parsedUrls) {
        return getOrCreate(dslContextProvider.ppc(shard), clientId, parsedUrls);
    }

    public List<Long> getOrCreateMobileContentList(int shard, ClientId clientId, List<MobileContent> mobileContents) {
        return getOrCreateMobileContentList(dslContextProvider.ppc(shard), clientId, mobileContents);
    }

    public List<Long> getOrCreateMobileContentList(DSLContext dslContext, ClientId clientId,
                                                   List<MobileContent> mobileContents) {
        if (mobileContents.isEmpty()) {
            return emptyList();
        }
        List<MobileContentDbKey> allDbKeys = StreamEx.of(mobileContents)
                .map(MobileContentDbKey::fromMobileContent)
                .toList();

        Map<MobileContentDbKey, Long> existingKeyToId = getExistingByDbKey(dslContext, clientId, allDbKeys);

        var dbKeyToMobileContentWithGeneratedId = EntryStream.zip(allDbKeys, mobileContents)
                .removeKeys(existingKeyToId::containsKey)
                .distinctKeys()
                .toMap();

        List<Long> mcIds = shardHelper.generateMobileContentIds(dbKeyToMobileContentWithGeneratedId.size());
        StreamEx.of(dbKeyToMobileContentWithGeneratedId.values()).zipWith(mcIds.stream())
                .forKeyValue(MobileContent::setId);

        new InsertHelper<>(dslContext, MOBILE_CONTENT)
                .addAll(mobileContentJooqMapper, dbKeyToMobileContentWithGeneratedId.values())
                .executeIfRecordsAdded();

        Map<MobileContentDbKey, Long> allDbKeysToId = EntryStream.of(dbKeyToMobileContentWithGeneratedId)
                .mapValues(MobileContent::getId)
                .append(existingKeyToId)
                .toMap();

        List<Long> ids = StreamEx.of(allDbKeys)
                .map(allDbKeysToId::get)
                .nonNull()
                .toList();

        Preconditions.checkState(ids.size() == mobileContents.size());
        return ids;
    }

    public Map<String, MobileContent> getByMobileAppStoreUrl(int shard, ClientId clientId, Collection<String> urls) {
        return getByMobileAppStoreUrl(dslContextProvider.ppc(shard), clientId, urls);
    }

    public Map<String, MobileContent> getByMobileAppStoreUrl(DSLContext dslContext, ClientId clientId,
                                                             Collection<String> urls) {
        List<String> listOfUrls = new ArrayList<>(urls);
        List<MobileContentDbKey> mobileContentDbKeys = StreamEx.of(listOfUrls)
                .map(MobileAppStoreUrlParser::parseStrict)
                .map(MobileAppStoreUrl::toMobileContent)
                .peek(mbc -> mbc.setClientId(clientId.asLong()))
                .map(MobileContentDbKey::fromMobileContent)
                .toList();

        Map<MobileContentDbKey, Long> existingKeyToId = getExistingByDbKey(dslContext, clientId, mobileContentDbKeys);

        Map<Long, MobileContent> existentMobileContent =
                StreamEx.of(getMobileContent(dslContext, existingKeyToId.values()))
                        .mapToEntry(MobileContent::getId, Function.identity())
                        .toMap();

        return EntryStream.zip(listOfUrls, mobileContentDbKeys)
                .mapValues(existingKeyToId::get)
                .nonNullValues()
                .mapValues(existentMobileContent::get)
                .nonNullValues()
                .toMap();
    }

    /**
     * @return отображение MobileContentDbKey в MobileContent.id
     */
    private Map<MobileContentDbKey, Long> getExistingByDbKey(DSLContext dslContext, ClientId clientId,
                                                             List<MobileContentDbKey> allDbKeys) {
        if (allDbKeys.isEmpty()) {
            return emptyMap();
        }

        Condition condition = allDbKeys.stream()
                .map(MobileContentDbKey::toDbCondition)
                .reduce(DSL::or)
                // исходный список не может быть пустым, мы это проверили выше
                .orElseThrow(IllegalStateException::new);

        Result<MobileContentRecord> existingRecords = dslContext
                .select(MOBILE_CONTENT_DB_KEY_AND_ID_FIELDS)
                .from(MOBILE_CONTENT)
                .where(condition)

                // это условие, может быть, избыточное, потому что clientId есть в MobileContentDbKey,
                // зато так база точно заиспользует индекс и мы точно не извлечём значений
                // про других клиентов
                .and(MOBILE_CONTENT.CLIENT_ID.eq(clientId.asLong()))
                .fetch()
                .into(MOBILE_CONTENT);

        return existingRecords.stream()
                .collect(toMap(MobileContentDbKey::fromDbRecord, MobileContentRecord::getMobileContentId));
    }

    /**
     * Возвращает записи из таблицы ppc.mobile_content для данного клиента, идентификатора (магазинного), типа контента,
     * типа ОС и региона.
     */
    public List<MobileContent> getMobileContent(int shard, ClientId clientId, String storeContentId,
                                                ContentType contentType, OsType osType, String country) {
        return dslContextProvider.ppc(shard)
                .select(allFields)
                .from(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.CLIENT_ID.eq(clientId.asLong()))
                .and(MOBILE_CONTENT.STORE_CONTENT_ID.eq(storeContentId))
                // TODO: Этого поля пока нет в YT, непонятно откуда оно приходит
                // upd: ниоткуда не приходит (https://st.yandex-team.ru/DIRECT-80538#1532590933000)
                .and(MOBILE_CONTENT.CONTENT_TYPE.eq(ContentType.toSource(contentType)))
                .and(MOBILE_CONTENT.OS_TYPE.eq(OsType.toSource(osType)))
                .and(MOBILE_CONTENT.STORE_COUNTRY.eq(country))
                .fetchStream()
                .map(mobileContentJooqMapper::fromDb)
                .collect(toList());
    }

    /**
     * Обновляет записи в таблице ppc.mobile_content
     */
    public int updateMobileContent(int shard, List<AppliedChanges<MobileContent>> changes) {
        JooqUpdateBuilder<MobileContentRecord, MobileContent> builder =
                new JooqUpdateBuilder<>(MOBILE_CONTENT.MOBILE_CONTENT_ID, changes);

        builder.processProperty(MobileContent.ID, MOBILE_CONTENT.MOBILE_CONTENT_ID);
        builder.processProperty(MobileContent.STORE_CONTENT_ID, MOBILE_CONTENT.STORE_CONTENT_ID);
        builder.processProperty(MobileContent.STORE_COUNTRY, MOBILE_CONTENT.STORE_COUNTRY);
        builder.processProperty(MobileContent.OS_TYPE, MOBILE_CONTENT.OS_TYPE, OsType::toSource);
        builder.processProperty(MobileContent.IS_AVAILABLE, MOBILE_CONTENT.IS_AVAILABLE,
                RepositoryUtils::booleanToLong);
        builder.processProperty(MobileContent.BUNDLE_ID, MOBILE_CONTENT.BUNDLE_ID);
        builder.processProperty(MobileContent.MODIFY_TIME, MOBILE_CONTENT.MODIFY_TIME);
        builder.processProperty(MobileContent.STORE_REFRESH_TIME, MOBILE_CONTENT.STORE_REFRESH_TIME);
        builder.processProperty(MobileContent.STATUS_BS_SYNCED, MOBILE_CONTENT.STATUS_BS_SYNCED,
                MobileContentMapping::statusBsSyncedToDbFormat);
        builder.processProperty(MobileContent.NAME, MOBILE_CONTENT.NAME);
        builder.processProperty(MobileContent.PRICES, MOBILE_CONTENT.PRICES_JSON,
                MobileContentMapping::pricesToDbFormat);
        builder.processProperty(MobileContent.RATING, MOBILE_CONTENT.RATING);
        builder.processProperty(MobileContent.RATING_VOTES, MOBILE_CONTENT.RATING_VOTES);
        builder.processProperty(MobileContent.ICON_HASH, MOBILE_CONTENT.ICON_HASH);
        builder.processProperty(MobileContent.STATUS_ICON_MODERATE, MOBILE_CONTENT.STATUS_ICON_MODERATE,
                StatusIconModerate::toSource);
        builder.processProperty(MobileContent.MIN_OS_VERSION, MOBILE_CONTENT.MIN_OS_VERSION);
        builder.processProperty(MobileContent.PUBLISHER_DOMAIN_ID, MOBILE_CONTENT.PUBLISHER_DOMAIN_ID);
        builder.processProperty(MobileContent.GENRE, MOBILE_CONTENT.GENRE);
        builder.processProperty(MobileContent.AGE_LABEL, MOBILE_CONTENT.AGE_LABEL, AgeLabel::toSource);
        builder.processProperty(MobileContent.TRIES_COUNT, MOBILE_CONTENT.TRIES_COUNT, Long::valueOf);
        builder.processProperty(MobileContent.AVAILABLE_ACTIONS, MOBILE_CONTENT.AVAILABLE_ACTIONS,
                MobileContentMapping::availableActionToDbFormat);
        builder.processProperty(MobileContent.APP_SIZE, MOBILE_CONTENT.APP_SIZE_BYTES,
                MobileContentMapping::appSizeBytesToDbFormat);
        builder.processProperty(MobileContent.DOWNLOADS, MOBILE_CONTENT.DOWNLOADS);
        builder.processProperty(MobileContent.SCREENS, MOBILE_CONTENT.SCREENS,
                MobileContentMapping::screensToDbFormat);

        return dslContextProvider.ppc(shard)
                .update(MOBILE_CONTENT)
                .set(builder.getValues())
                .where(MOBILE_CONTENT.MOBILE_CONTENT_ID.in(builder.getChangedIds()))
                .execute();
    }

    /**
     * Для набора ID контента подготавливает объекты для отправки в ленивую очередь в БК.
     *
     * @param shard Шард
     * @param ids   Набор ID контента
     */
    public List<BsResyncItem> getMobileContentForResync(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(PHRASES.CID, PHRASES.PID, BANNERS.BID)
                .from(MOBILE_CONTENT)
                .join(ADGROUPS_MOBILE_CONTENT)
                .on(ADGROUPS_MOBILE_CONTENT.MOBILE_CONTENT_ID.eq(MOBILE_CONTENT.MOBILE_CONTENT_ID))
                .join(PHRASES)
                .on(PHRASES.PID.eq(ADGROUPS_MOBILE_CONTENT.PID))
                .join(BANNERS)
                .on(BANNERS.PID.eq(ADGROUPS_MOBILE_CONTENT.PID))
                .where(MOBILE_CONTENT.MOBILE_CONTENT_ID.in(ids))
                .fetch(r -> new BsResyncItem(
                        BsResyncPriority.ON_MOBILE_CONTENT_CHANGED,
                        r.get(PHRASES.CID),
                        r.get(BANNERS.BID),
                        r.get(PHRASES.PID)));
    }

    /**
     * Подготавливает набор уведомлений {@link MobileContentMonitoringNotification} об изменении доступности
     * контента в магазине.
     *
     * @param shard         Шард
     * @param notifyChanged Набор ID контента с указанием его доступности (true - доступен)
     */
    public Collection<MobileContentMonitoringNotification> getContentAvailabilityNotifications(int shard,
                                                                                               Map<Long, Boolean> notifyChanged) {
        Map<Long, MobileContentMonitoringNotification> notificationsByUid = new HashMap<>();
        dslContextProvider.ppc(shard)
                .select(MOBILE_CONTENT.MOBILE_CONTENT_ID, USERS.UID, USERS.FIO, CAMPAIGNS.CID,
                        ADGROUPS_MOBILE_CONTENT.STORE_CONTENT_HREF)
                .from(MOBILE_CONTENT)
                .join(ADGROUPS_MOBILE_CONTENT)
                .using(MOBILE_CONTENT.MOBILE_CONTENT_ID)
                .join(PHRASES)
                .using(ADGROUPS_MOBILE_CONTENT.PID)
                .join(CAMPAIGNS)
                .using(PHRASES.CID)
                .join(USERS)
                .using(CAMPAIGNS.UID)
                .where(MOBILE_CONTENT.MOBILE_CONTENT_ID.in(notifyChanged.keySet()))
                .and(CAMPAIGNS.TYPE.in(CampaignsType.mobile_content))
                .and(CAMPAIGNS.STATUS_ACTIVE.eq(CampaignsStatusactive.Yes))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .forEach(r -> notificationsByUid
                        .computeIfAbsent(r.get(USERS.UID), uid -> new MobileContentMonitoringNotification(
                                uid,
                                r.get(USERS.FIO),
                                notifyChanged.get(r.get(MOBILE_CONTENT.MOBILE_CONTENT_ID))
                                        ? MobileContentMonitoringNotification.State.ALIVE
                                        : MobileContentMonitoringNotification.State.DEAD,
                                new ArrayList<>()))
                        .addStateChangeRecord(MobileContentMonitoringNotification.Record.builder()
                                .withUid(r.get(USERS.UID))
                                .withFio(r.get(USERS.FIO))
                                .withCampaignId(r.get(CAMPAIGNS.CID))
                                .withMobileContentId(r.get(MOBILE_CONTENT.MOBILE_CONTENT_ID))
                                .withStoreContentHref(r.get(ADGROUPS_MOBILE_CONTENT.STORE_CONTENT_HREF))
                                .build()));
        return notificationsByUid.values();
    }

    /**
     * Удаляет записи из таблицы ppc.mobile_content по их ID.
     */
    public void deleteMobileContentById(int shard, Collection<Long> ids) {
        dslContextProvider.ppc(shard)
                .deleteFrom(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.MOBILE_CONTENT_ID.in(ids))
                .execute();
    }

    private static JooqMapperWithSupplier<MobileContent> createMobileContentMapper() {
        return JooqMapperWithSupplierBuilder.builder(MobileContent::new)
                .map(property(MobileContent.ID, MOBILE_CONTENT.MOBILE_CONTENT_ID))
                .map(property(MobileContent.CLIENT_ID, MOBILE_CONTENT.CLIENT_ID))
                .map(property(MobileContent.STORE_CONTENT_ID, MOBILE_CONTENT.STORE_CONTENT_ID))
                .map(property(MobileContent.STORE_COUNTRY, MOBILE_CONTENT.STORE_COUNTRY))
                .map(convertibleProperty(MobileContent.OS_TYPE, MOBILE_CONTENT.OS_TYPE,
                        OsType::fromSource,
                        OsType::toSource))
                .map(convertibleProperty(MobileContent.CONTENT_TYPE, MOBILE_CONTENT.CONTENT_TYPE,
                        ContentType::fromSource,
                        ContentType::toSource))
                .map(property(MobileContent.BUNDLE_ID, MOBILE_CONTENT.BUNDLE_ID))
                .map(booleanProperty(MobileContent.IS_AVAILABLE, MOBILE_CONTENT.IS_AVAILABLE))
                .map(property(MobileContent.CREATE_TIME, MOBILE_CONTENT.CREATE_TIME))
                .writeField(MOBILE_CONTENT.MODIFY_TIME, fromPropertyToField(MobileContent.MODIFY_TIME)
                        .by(RepositoryUtils::zeroableDateTimeToDb))
                .readProperty(MobileContent.MODIFY_TIME, fromField(MOBILE_CONTENT.MODIFY_TIME))
                .map(property(MobileContent.NAME, MOBILE_CONTENT.NAME))
                .writeField(MOBILE_CONTENT.STORE_REFRESH_TIME, fromPropertyToField(MobileContent.STORE_REFRESH_TIME)
                        .by(RepositoryUtils::zeroableDateTimeToDb))
                .readProperty(MobileContent.STORE_REFRESH_TIME, fromField(MOBILE_CONTENT.STORE_REFRESH_TIME))
                .map(convertibleProperty(MobileContent.STATUS_BS_SYNCED, MOBILE_CONTENT.STATUS_BS_SYNCED,
                        MobileContentMapping::statusBsSyncedFromDbFormat,
                        MobileContentMapping::statusBsSyncedToDbFormat))
                .map(property(MobileContent.ICON_HASH, MOBILE_CONTENT.ICON_HASH))
                .map(convertibleProperty(MobileContent.STATUS_ICON_MODERATE, MOBILE_CONTENT.STATUS_ICON_MODERATE,
                        StatusIconModerate::fromSource,
                        StatusIconModerate::toSource))
                .map(convertibleProperty(MobileContent.APP_SIZE, MOBILE_CONTENT.APP_SIZE_BYTES,
                        MobileContentMapping::appSizeBytesFromDbFormat,
                        MobileContentMapping::appSizeBytesToDbFormat))
                .map(convertibleProperty(MobileContent.AVAILABLE_ACTIONS, MOBILE_CONTENT.AVAILABLE_ACTIONS,
                        MobileContentMapping::availableActionFromDbFormat,
                        MobileContentMapping::availableActionToDbFormat))
                .map(property(MobileContent.PUBLISHER_DOMAIN_ID, MOBILE_CONTENT.PUBLISHER_DOMAIN_ID))
                .map(property(MobileContent.GENRE, MOBILE_CONTENT.GENRE))
                .map(convertibleProperty(MobileContent.AGE_LABEL, MOBILE_CONTENT.AGE_LABEL,
                        AgeLabel::fromSource,
                        AgeLabel::toSource))
                .map(property(MobileContent.MIN_OS_VERSION, MOBILE_CONTENT.MIN_OS_VERSION))
                .map(convertibleProperty(MobileContent.PRICES, MOBILE_CONTENT.PRICES_JSON,
                        MobileContentMapping::pricesFromDbFormat,
                        MobileContentMapping::pricesToDbFormat))
                .map(property(MobileContent.RATING, MOBILE_CONTENT.RATING))
                .map(property(MobileContent.RATING_VOTES, MOBILE_CONTENT.RATING_VOTES))
                .map(integerProperty(MobileContent.TRIES_COUNT, MOBILE_CONTENT.TRIES_COUNT))
                .map(property(MobileContent.DOWNLOADS, MOBILE_CONTENT.DOWNLOADS))
                .map(convertibleProperty(MobileContent.SCREENS, MOBILE_CONTENT.SCREENS,
                        MobileContentMapping::screensFromDbFormat,
                        MobileContentMapping::screensToDbFormat))
                .build();
    }

    public Map<Long, Long> getClientIdsByMobileContentIds(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(MOBILE_CONTENT.MOBILE_CONTENT_ID, MOBILE_CONTENT.CLIENT_ID)
                .from(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.MOBILE_CONTENT_ID.in(ids))
                .fetchMap(MOBILE_CONTENT.MOBILE_CONTENT_ID, MOBILE_CONTENT.CLIENT_ID);
    }

    private static class MobileContentDbKey {
        static final List<TableField<?, ?>> FIELDS_TO_READ = ImmutableList.of(
                MOBILE_CONTENT.CLIENT_ID,
                MOBILE_CONTENT.OS_TYPE,
                MOBILE_CONTENT.CONTENT_TYPE,
                MOBILE_CONTENT.STORE_COUNTRY,
                MOBILE_CONTENT.STORE_CONTENT_ID
        );

        final Long clientId;
        final MobileContentOsType osType;
        final MobileContentContentType contentType;
        final String storeCountry;
        final String storeContentId;

        MobileContentDbKey(Long clientId,
                           MobileContentOsType osType, MobileContentContentType contentType, String storeCountry,
                           String storeContentId) {
            this.clientId = clientId;
            this.osType = osType;
            this.contentType = contentType;
            this.storeCountry = storeCountry;
            this.storeContentId = storeContentId;
        }

        static MobileContentDbKey fromMobileContent(MobileContent mobileContent) {
            return new MobileContentDbKey(
                    checkNotNull(mobileContent.getClientId()),
                    checkNotNull(OsType.toSource(mobileContent.getOsType())),
                    checkNotNull(ContentType.toSource(mobileContent.getContentType())),
                    checkNotNull(mobileContent.getStoreCountry()),
                    checkNotNull(mobileContent.getStoreContentId()));
        }

        static MobileContentDbKey fromDbRecord(MobileContentRecord record) {
            return new MobileContentDbKey(
                    record.getClientid(),
                    record.getOsType(),
                    record.getContentType(),
                    record.getStoreCountry(),
                    record.getStoreContentId());
        }

        Condition toDbCondition() {
            return MOBILE_CONTENT.CLIENT_ID.eq(clientId)
                    .and(MOBILE_CONTENT.OS_TYPE.eq(osType))
                    .and(MOBILE_CONTENT.CONTENT_TYPE.eq(contentType))
                    .and(MOBILE_CONTENT.STORE_COUNTRY.eq(storeCountry))
                    .and(MOBILE_CONTENT.STORE_CONTENT_ID.eq(storeContentId));
        }

        @Override
        public String toString() {
            return "MobileContentDbKey{" +
                    "clientId=" + clientId +
                    ", osType=" + osType +
                    ", contentType=" + contentType +
                    ", storeCountry='" + storeCountry + '\'' +
                    ", storeContentId='" + storeContentId + '\'' +
                    '}';
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            MobileContentDbKey that = (MobileContentDbKey) o;
            return Objects.equals(clientId, that.clientId) &&
                    osType == that.osType &&
                    contentType == that.contentType &&
                    Objects.equals(storeCountry, that.storeCountry) &&
                    Objects.equals(storeContentId, that.storeContentId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(clientId, osType, contentType, storeCountry, storeContentId);
        }
    }
}
