package ru.yandex.direct.core.entity.mobilecontent.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileContentWithExtraData;
import ru.yandex.direct.core.entity.mobilecontent.model.ApiMobileContentYT;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentAvatarSize;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentForAppId;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.mobilecontent.model.StatusIconModerate;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentFetchQueueRepository;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository;
import ru.yandex.direct.core.service.storeurlchecker.StoreUrlInstantChecker;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.mobilecontent.model.OsType.ANDROID;
import static ru.yandex.direct.core.entity.mobilecontent.service.MobileContentServiceConstants.OS_TYPE_TO_AVATARS_INSTANCE;
import static ru.yandex.direct.core.entity.mobilecontent.service.MobileContentServiceConstants.OS_TYPE_TO_STORE_NAME;

@Lazy
@Service
@ParametersAreNonnullByDefault
public class MobileContentService {
    private static final Logger logger = LoggerFactory.getLogger(MobileContentService.class);

    private final ShardHelper shardHelper;
    private final MobileContentRepository mobileContentRepository;
    private final String avatarsMdsHost;
    private final MobileContentYtHelper mobileContentYtHelper;
    private final StoreUrlInstantChecker storeUrlInstantChecker;
    private final MobileContentFetchQueueRepository mobileContentFetchQueueRepository;

    @Autowired
    public MobileContentService(ShardHelper shardHelper,
                                MobileContentRepository mobileContentRepository,
                                @Value("${avatars.mds.host}") String avatarsMdsHost,
                                MobileContentYtHelper mobileContentYtHelper,
                                StoreUrlInstantChecker storeUrlInstantChecker,
                                MobileContentFetchQueueRepository mobileContentFetchQueueRepository) {
        this.shardHelper = shardHelper;
        this.mobileContentRepository = mobileContentRepository;
        this.avatarsMdsHost = avatarsMdsHost;
        this.mobileContentYtHelper = mobileContentYtHelper;
        this.storeUrlInstantChecker = storeUrlInstantChecker;
        this.mobileContentFetchQueueRepository = mobileContentFetchQueueRepository;
    }

    public String getStoreName(OsType osType) {
        checkNotNull(osType);
        checkArgument(OS_TYPE_TO_STORE_NAME.containsKey(osType), "Нет маппинга для типа ОС %s", osType);

        return OS_TYPE_TO_STORE_NAME.get(osType);
    }

    public String getStoreAppId(MobileContentForAppId mobileContent) {
        return getStoreAppIdFromMobileContent(mobileContent);
    }

    public static String getStoreAppIdFromMobileContent(MobileContentForAppId mobileContent) {
        checkNotNull(mobileContent);
        checkNotNull(mobileContent.getOsType());

        OsType osType = mobileContent.getOsType();
        if (osType == OsType.IOS) {
            return mobileContent.getBundleId();
        } else if (osType == ANDROID) {
            return mobileContent.getStoreContentId();
        }

        throw new IllegalStateException(String.format("Unknown mobile os type: %s", osType));
    }

    public List<MobileContentAvatarSize> getAvatarsSizes() {
        return Arrays.asList(MobileContentAvatarSize.values());
    }

    @SuppressWarnings("unused")
    public String generateUrlString(OsType osType, String iconHash) {
        // по умолчанию возвращаем ссылку на иконку 65x65
        return generateUrlString(osType, iconHash, MobileContentAvatarSize.ICON);
    }

    public String generateUrlString(OsType osType, String iconHash, MobileContentAvatarSize size) {
        return generateUrlString(avatarsMdsHost, osType, iconHash, size);
    }

    public static String generateUrlString(String customAvatarsMdsHost, OsType osType, String iconHash,
                                           MobileContentAvatarSize size) {
        checkNotNull(osType);
        checkNotNull(iconHash);
        checkArgument(OS_TYPE_TO_AVATARS_INSTANCE.containsKey(osType), "Нет маппинга для типа ОС %s", osType);

        return String.format(
                "//%s/get-%s/%s/%s", customAvatarsMdsHost, OS_TYPE_TO_AVATARS_INSTANCE.get(osType), iconHash,
                size.getName());
    }

    /**
     * Получить список описаний объектов MobileContent с урлом, сохранённым для соответствющей РМП группы
     */
    public List<MobileContentWithExtraData> getStoreHrefWithMobileContent(ClientId clientId, LimitOffset limitOffset) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return mobileContentRepository.getStoreHrefWithMobileAppContent(shard, clientId, limitOffset);
    }

    /**
     * Получает информацию о мобильном приложении.
     * Первый запрос идет в БД, и если там приложения с такими параметрами нет - идет второй запрос в YT.
     *
     * @param clientId               ID клиента
     * @param parsedStoreUrl         Запрос
     * @param dryRun                 True если после обращения к YT следует сохранять значения в базу
     * @param doInstantCheckStoreUrl Если приложение не находится ни в базе ни в Yt, проверить урл и если там всё ОК,
     *                               вернуть минимальную информацию о приложении, полученную из его урла
     * @return Информация о мобильном приложении
     */
    public Optional<MobileContent> getMobileContent(ClientId clientId, String storeUrl,
                                                    MobileAppStoreUrl parsedStoreUrl, boolean dryRun,
                                                    boolean doInstantCheckStoreUrl) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<MobileContent> mobileContent = mobileContentRepository.getMobileContent(shard, clientId,
                parsedStoreUrl.getStoreContentId(), parsedStoreUrl.getContentType(), parsedStoreUrl.getOsType(),
                parsedStoreUrl.getStoreCountry());
        if (!mobileContent.isEmpty()) {
            return StreamEx.of(mobileContent).findFirst();
        }

        mobileContent = getMobileContentFromYt(shard, singletonList(parsedStoreUrl))
                .get(parsedStoreUrl)
                .map(Collections::singletonList)
                .orElse(emptyList());
        if (mobileContent.isEmpty()) {
            mobileContentFetchQueueRepository.addUrl(storeUrl);
            try {
                if (doInstantCheckStoreUrl && storeUrlInstantChecker.isStoreUrlReachable(storeUrl)) {
                    mobileContent = singletonList(parsedStoreUrl.toMobileContent().withIsAvailable(true));
                } else {
                    return Optional.empty();
                }
            } catch (Exception ex) {
                logger.error("Store url check failed", ex);
                return Optional.empty();
            }
        }

        if (!dryRun) {
            LocalDateTime now = LocalDateTime.now();
            List<Long> ids = shardHelper.generateMobileContentIds(mobileContent.size());
            StreamEx.of(mobileContent).zipWith(ids.stream())
                    .forKeyValue((mc, id) -> mc
                            .withId(id)
                            .withClientId(clientId.asLong())
                            .withTriesCount(0)
                            .withCreateTime(now)
                            .withModifyTime(now)
                            .withStoreRefreshTime(now)
                            .withStatusBsSynced(StatusBsSynced.NO)
                            .withStatusIconModerate(StatusIconModerate.READY));
            mobileContentRepository.addMobileContent(shard, mobileContent);
        }

        return StreamEx.of(mobileContent).findFirst();
    }

    public Map<MobileAppStoreUrl, Optional<MobileContent>> getMobileContentFromYt(
            int shard,
            Collection<MobileAppStoreUrl> parsedUrlCollection) {
        if (parsedUrlCollection.isEmpty()) {
            return emptyMap();
        }
        // код ниже будет использовать порядок урлов для матчинга результатов
        ArrayList<MobileAppStoreUrl> parsedUrls = new ArrayList<>(parsedUrlCollection);

        List<Pair<OsType, YTreeMapNode>> osTypeAndYtKeyList = StreamEx.of(parsedUrls)
                .map(u -> Pair.of(u.getOsType(), mobileContentYtHelper.createLookupKey(u)))
                .toList();
        Map<OsType, List<YTreeMapNode>> osTypeToKeys = StreamEx.of(osTypeAndYtKeyList)
                .mapToEntry(Pair::getLeft, Pair::getRight)
                .grouping();
        Map<YTreeMapNode, MobileContent> resultFromYt = EntryStream.of(osTypeToKeys)
                .mapKeyValue((osType, ytKeys) -> mobileContentYtHelper.getMobileContentFromYt(shard, osType, ytKeys))
                .flatMap(List::stream)
                .mapToEntry(mobileContentYtHelper::createLookupKey, identity())
                .toMap();
        checkState(osTypeAndYtKeyList.size() == parsedUrls.size());
        return EntryStream.of(parsedUrls)
                .invert()
                .mapValues(osTypeAndYtKeyList::get)
                .mapValues(Pair::getRight)
                .mapValues(resultFromYt::get)
                .mapValues(Optional::ofNullable)
                .toMap();
    }

    public List<ApiMobileContentYT> getApiMobileContentFromYt(String store, Collection<String> appIds) {
        return mobileContentYtHelper.getApiMobileContentFromYt(store, appIds);
    }

    @SuppressWarnings("UnusedReturnValue")
    public int updateMobileContent(int shard, List<AppliedChanges<MobileContent>> changes) {
        return mobileContentRepository.updateMobileContent(shard, changes);
    }

    public List<MobileContent> getMobileContent(ClientId clientId, Collection<Long> ids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return mobileContentRepository.getMobileContent(shard, ids);
    }

}
