package ru.yandex.direct.core.entity.mobilecontent.service;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Timer;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import org.jooq.Field;
import org.jooq.impl.DSL;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.common.mobilecontent.MobileContentYtTable;
import ru.yandex.direct.common.mobilecontent.MobileContentYtTablesConfig;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.converter.MobileContentYtConverter;
import ru.yandex.direct.core.entity.mobilecontent.model.ApiMobileContentYT;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.grid.schema.yt.tables.ExtdatamobileDirect;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.dynamic.context.YtDynamicContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.direct.ytwrapper.dynamic.selector.AttributeBasedClusterChooser;
import ru.yandex.direct.ytwrapper.dynamic.selector.ClusterChooser;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeMapNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeStringNodeImpl;
import ru.yandex.inside.yt.kosher.tables.types.YsonTableEntryType;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.core.entity.mobilecontent.service.MobileContentServiceConstants.OS_TYPE_TO_YT_PATH;
import static ru.yandex.direct.grid.schema.yt.Tables.EXTDATAMOBILE_DIRECT;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.dynamic.selector.AttributeBasedClusterChooser.REVERSED_INSTANT_COMPARATOR;

@ParametersAreNonnullByDefault
@Lazy
@Component
public class MobileContentYtHelper {
    private static final String MOBILE_ALIAS = "MOBILE";
    private static final ExtdatamobileDirect MOBILE = EXTDATAMOBILE_DIRECT.as(MOBILE_ALIAS);
    private static final Field<String> APP_ID = aliased(MOBILE.APP_ID);
    private static final Field<String> LANG = aliased(MOBILE.LANG);
    private static final Field<String> BUNDLE = aliased(MOBILE.BUNDLE);
    private static final Field<String> NAME = aliased(MOBILE.NAME);
    private static final Field<String> ICON = aliased(MOBILE.ICON);
    private static final Field<String> CURRENCY = aliased(MOBILE.CURRENCY);
    private static final Field<String> PUBLISHER = aliased(MOBILE.PUBLISHER);
    private static final Field<String> WEBSITE = aliased(MOBILE.WEBSITE);
    private static final Field<Long> RATING_COUNT = aliased(MOBILE.RATING_COUNT);

    private static final YsonTableEntryType TABLE_ENTRY_TYPE = new YsonTableEntryType(true, true);
    private static final Duration YT_CLUSTER_REFRESH_INTERVAL = Duration.ofMinutes(1);
    private static final Duration YT_SELECT_ROWS_TIMEOUT = Duration.ofSeconds(3);

    private final Map<String, ClusterChooser<?>> clusterChooserByShopName;
    private final MobileContentYtTablesConfig ytTablesConfig;
    private final YtProvider ytProvider;
    private final MobileContentYtConverter converter;
    private final Timer timer;

    public MobileContentYtHelper(YtProvider ytProvider, MobileContentYtTablesConfig ytTablesConfig,
                                 MobileContentYtConverter converter) {
        this.timer = new Timer(getClass().getSimpleName() + "-Times", /* isDaemon = */ true);
        this.converter = converter;
        this.ytTablesConfig = ytTablesConfig;
        this.ytProvider = ytProvider;
        this.clusterChooserByShopName = Collections.unmodifiableMap(createClusterChoosers(ytProvider, ytTablesConfig));
    }

    /**
     * Создает ClusterChooser'ы для каждого магазина. Сами ClusterChooser'ы будут запущены при первом запросе.
     */
    private Map<String, ClusterChooser<?>> createClusterChoosers(YtProvider ytProvider,
                                                                 MobileContentYtTablesConfig ytTablesConfig) {
        Map<String, ClusterChooser<?>> clusterChooserByShopName = new HashMap<>();
        ytTablesConfig.getAllShops().forEach((shopName, shopConfig) -> {
            AttributeBasedClusterChooser chooser = new AttributeBasedClusterChooser(
                    timer,
                    YT_CLUSTER_REFRESH_INTERVAL,
                    ytProvider,
                    mapList(shopConfig.getClusters(), YtCluster::parse),
                    YPath.simple(shopConfig.getTable()), "@creation_time",
                    REVERSED_INSTANT_COMPARATOR);
            clusterChooserByShopName.put(shopName, chooser);
        });
        return clusterChooserByShopName;
    }

    private static String getStoreYtPath(OsType osType) {
        return Optional.ofNullable(OS_TYPE_TO_YT_PATH.get(osType))
                .orElseThrow(() -> new IllegalArgumentException("Нет маппинга для типа ОС " + osType));
    }

    private static void checkStoreYtPath(String storePath) {
        if (!OS_TYPE_TO_YT_PATH.containsValue(storePath)) {
            throw new IllegalArgumentException("There is no store called " + storePath);
        }
    }

    @PreDestroy
    public void stopTimer() {
        if (timer != null) {
            timer.cancel();
        }
    }

    /**
     * Создает объект-ключ для поиска информации о приложении в таблице YT.
     */
    @SuppressWarnings("WeakerAccess")
    public YTreeMapNode createLookupKey(MobileAppStoreUrl parsedUrl) {
        return createLookupKey(parsedUrl.getStoreContentId(), parsedUrl.getStoreCountry());
    }

    /**
     * Создает объект-ключ для поиска информации о приложении в таблице YT.
     */
    @SuppressWarnings("WeakerAccess")
    public YTreeMapNode createLookupKey(MobileContent mobileContent) {
        return createLookupKey(mobileContent.getStoreContentId(), mobileContent.getStoreCountry());
    }

    /**
     * Создает объект-ключ для поиска информации о приложении в таблице YT.
     *
     * @param contentId ID приложения в таблице
     * @param country   Язык/страна
     * @return Ключ для поиска в YT
     */
    @SuppressWarnings("WeakerAccess")
    public static YTreeMapNode createLookupKey(String contentId, String country) {
        YTreeMapNode key = new YTreeMapNodeImpl(Cf.map());
        key.put("app_id", new YTreeStringNodeImpl(contentId, Cf.map()));
        key.put("lang", new YTreeStringNodeImpl(country.toLowerCase(), Cf.map()));
        return key;
    }

    /**
     * Получает данные о приложении из YT.
     *
     * @param shard  Шард
     * @param osType Тип ОС
     * @param keys   Список ключей. Ключи можно создавать при помощи
     *               метода {@link MobileContentYtHelper#createLookupKey(String, String)}
     * @return Список приложений с таким ключом
     */
    @SuppressWarnings("WeakerAccess")
    public List<MobileContent> getMobileContentFromYt(int shard, OsType osType,
                                                      Collection<YTreeMapNode> keys) {
        String ytShop = getStoreYtPath(osType);
        YPath yPath = ytTablesConfig.getShopInfo(ytShop)
                .map(MobileContentYtTable::getTable)
                .map(YPath::simple)
                .orElseThrow(() -> new IllegalStateException(String.format("Unable to find table for %s", ytShop)));

        YtCluster cluster = Optional.ofNullable(clusterChooserByShopName.get(ytShop))
                .flatMap(ClusterChooser::getCluster)
                .orElseThrow(() -> new IllegalStateException(String.format("No cluster for %s", ytShop)));

        List<MobileContent> result = new ArrayList<>();
        Map<String, Optional<String>> websites = new HashMap<>();
        ytProvider.get(cluster).tables().lookupRows(yPath, TABLE_ENTRY_TYPE, Cf.x(keys), TABLE_ENTRY_TYPE,
                (Consumer<YTreeMapNode>) row -> result.add(converter.ytToMobileContent(row, osType, websites)));
        Map<String, Long> domainsByAppId = converter.getDomainIds(shard, websites);
        result.forEach(mc -> mc.withPublisherDomainId(domainsByAppId.getOrDefault(mc.getStoreContentId(), null)));
        return result;
    }

    /**
     * Получает данные о мобильном приложении из YT для intapi.
     *
     * @param store  Тип стора
     * @param appIds Список app_id приложений
     * @return Список приложений с такими id
     */
    @SuppressWarnings("WeakerAccess")
    public List<ApiMobileContentYT> getApiMobileContentFromYt(String store, Collection<String> appIds) {
        checkStoreYtPath(store);
        checkArgument(clusterChooserByShopName.containsKey(store), "No cluster for " + store);
        var tablePath = ytTablesConfig.getShopInfo(store)
                .map(MobileContentYtTable::getTable)
                .orElseThrow(() -> new IllegalStateException(String.format("Unable to find table for %s", store)));
        var query = YtDSL.ytContext()
                .select(APP_ID, LANG, BUNDLE, NAME, ICON, CURRENCY, PUBLISHER, WEBSITE, RATING_COUNT)
                .from(DSL.name("[" + tablePath + "] AS " + MOBILE_ALIAS))
                .where(APP_ID.in(appIds));
        var ytDynamicProvider = new YtDynamicContextProvider<>(clusterChooserByShopName.get(store), ytProvider,
                YT_SELECT_ROWS_TIMEOUT);

        return ytDynamicProvider.getContext()
                .executeTimeoutSafeSelect(query)
                .getYTreeRows().stream()
                .map(row -> converter.ytToApiMobileContent(store, row))
                .collect(Collectors.toList());
    }
}
